%% RAPTOR Exercise 1: Ramp-up current diffusion with varying heating
% In this exercise you will test the effect of differently timed heating
% during a plasma current ramp.

%% ------------------ ITER simulation setup ------------------
% First we set up the RAPTOR simulation.
[~, current_folder,~] = fileparts(pwd);
assert(strcmp(current_folder,'exercises'), 'You are not in the correct folder. Please open the exercices folder');
clear
run ../../RAPTOR_path.m

% set to true if you want to save figures
save_figure = false;

%% Configuration
% load the default parameters for ITER
config = RAPTOR_config('ITER'); % load default params
config.grid.tgrid = [0:0.1:1,3:2:140];

% tranport model: choose empirical model with
% transport reduction at low shear.
config.chi_e = RAPTORmodule('chi_FF');

% Heating actuators: EC and NBI
config.nbhcd = RAPTORmodule('hcd_gaussian');
config.echcd = RAPTORmodule('echcd_gaussian');
config.echcd.Configuration.n_units = 3;
config.echcd.Configuration.n_rdep_external = 0; % put 0 if you want to fill rdep directly in params.echcd
config.nbhcd.Configuration.n_units = 1;
config.nbhcd.Configuration.n_rdep_external = 0; % put 0 if you want to fill rdep directly in params.nbhcd

[model,params,init,g,v] = build_RAPTOR_model(config); % generate model structure for these params.

%% Parametrization

% radiation modules etc
params.prad.active = false;
params.pei.active = true;
params.palpha.active = false;

% for chi_e
params.chi_e.cano = 4;
params.chi_e.cneo = 0.1;
params.chi_e.te_constant = 1.2; % includes a factor Te^1.2 which mimics the impact of stiffness

% for nbhcd
% be careful that array sizes are consistent with nb_units
params.nbhcd.active = true;
params.nbhcd.rdep = [0.26]; %
params.nbhcd.wdep = [0.15]; % broad heating
params.nbhcd.wdep_out = [0.75]; % broad heating
params.nbhcd.cd_eff = [5]; % current drive
% the index for the power trace in the input vector is stored in the model
% structure
disp(model.nbhcd.uind(model.nbhcd.ind_power))
% for echcd
% be careful that array sizes are consistent with nb_units
params.echcd.active = true;
params.echcd.rdep = [0. 0.3 0.4];
params.echcd.cd_eff = [20 20 20]; % current drive efficiency factor
params.echcd.wdep = [.05 .05 .05];  % wdep =0.05 for each
% the indices for the power trace in the input vector is stored in the model
% structure
disp(model.echcd.uind(model.echcd.ind_power))

params.debug.iterdisp = 10; % display every 10

% Plasma current ramp (Ip) & Heating powers
rampfun = @(t,tmin,ymin,tmax,ymax) max(ymin,min((ymax-ymin)/(tmax-tmin)*(t-tmin),ymax-ymin)+ymin); % anonymous function for ramps

U(1,:) = rampfun(params.tgrid,0,4e6,100,12e6); % input Ip trace
U(2,:) = zeros(size(U(1,:))); % placeholder, will replace this with heating later.
U(3,:) = zeros(size(U(1,:))); % placeholder, will replace this with heating later.
U(4,:) = zeros(size(U(1,:))); % placeholder, will replace this with heating later.
U(5,:) = zeros(size(U(1,:))); % placeholder, will replace this with heating later.

figure(1);
plot(params.tgrid, U);
title('Ohmic current & Heatings inputs');
xlabel('t (s)');
legend('U1/Ip [A]', 'U2 [W]', 'U3 [W]', 'U4 [W]', 'U5 [W]');

% initial conditions
init.Ip0 = U(1,1);
init.te0 = 2e3;
init.ne0 = 1e19;

x0 = RAPTOR_initial_conditions(model,init,g,v);  % Define the initial condition

% initialize V
V = repmat(v, 1, numel(params.tgrid));

% specify density ramp: multiply initial profile by ne0(t)
negauss = model.ne.Lamgauss*V(model.ne.vind,1);
ne0 = rampfun(params.tgrid,0,1,100,5);
negauss_varying = negauss*ne0;
V(model.ne.vind,:) = model.ne.Lamgauss \ negauss_varying;
V(model.ni.vind,:) = 0.9*V(model.ne.vind,:);

figure(2); hold on;
plot(params.tgrid, V(model.ne.vind(1),:), 'b');
plot(params.tgrid, V(model.ni.vind(1),:), 'r');
title('Electron & Ion density inputs'); xlabel('t (s)');
legend('n_e(\rho=0) [m\^-3]', 'n_i(\rho=0) [m\^-3]');
hold off;

%% Run the simulation
simres = RAPTOR_predictive(x0,g,V,U,model,params);
out = RAPTOR_out(simres,model,params);
RAPTOR_plot_GUI(out);

%% ------------------ EXERCISES ------------------


%% SOLUTION 1

tflattopgrid = 20:40:100;
cgrid = {'b','r','k','m'};
clear out simres
for ii=1:numel(tflattopgrid)
    tflattop = tflattopgrid(ii);
    U(1,:) = rampfun(params.tgrid,0,4e6,tflattop,12e6); % input Ip trace

    negauss = model.ne.Lamgauss*V(model.ne.vind,1);
    ne0 = rampfun(params.tgrid,0,1,tflattop,5);
    negauss_varying = negauss*ne0;
    V(model.ne.vind,:) = model.ne.Lamgauss \ negauss_varying;
    V(model.ni.vind,:) = 0.9*V(model.ne.vind,:);

    simres{ii} = RAPTOR_predictive(x0,g,V,U,model,params); % run sim
    out{ii} = RAPTOR_out(simres{ii},model,params); % get outputs
end

RAPTOR_plot_GUI(out)
%%
% The fastest ramp rate leads to a non-monotonic current
% density profile and flatter, or more reversed, q profiles. This can lead
% to transport barriers. In reality, the ramp rate is limited by MHD modes
% which can appear, especially for very steep current density profiles.


% Let us plot the time evolution of some global quantities and a set of
% profiles at the end of ramp-up
% note:
% * a faster ramp-up rate leads to a lower value of the internal inductance
%   throughout the entire ramp-up phase (consider the comparison in the
%   (Ip,li3)-plane), corresponding to a broader jpar profile
% * the quantity out.f_ss is a metric for the relaxation of the current
%   diffusion, defined as  f_ss = sum(out.dupldrho.^2) (remember: jpar is
%   fully diffused when dupldrho zero
%   note: by applying a slower ramp-up the current profile is more
%   stationary by the beginning of flat-top
% * the faster ramp-up requires larger values of the applied loop voltage

colseq = {'b','r',[0 0.4 0]};
figure;
for ii=1:numel(out)
  fig_1a = figure(1);
  subplot(341)
  plot(out{ii}.time, 1e-6*out{ii}.Ip(end,:),'color',colseq{ii})
  xlabel('t [s]')
  ylabel('Ip [MA]')
  ylim([0 15])
  legend(num2str(tflattopgrid'),'Location','SouthEast')

  hold on
  subplot(342)
  plot(out{ii}.time, out{ii}.upl(end,:),'color', colseq{ii})
  xlabel('t [s]')
  ylabel('Upl [V]')
  ylim([0 3])

  hold on
  plot(tflattopgrid(ii)*ones(2,1),[0 100],'color', colseq{ii},'LineStyle','--')
  hold on
  subplot(343)
  plot(out{ii}.time, out{ii}.li3,'color', colseq{ii})
  xlabel('t [s]')
  ylabel('li3')
  ylim([0 1])

  hold on
  plot(tflattopgrid(ii)*ones(2,1),[0 100],'color', colseq{ii},'LineStyle','--')
  hold on
  subplot(344)
  plot(out{ii}.time, out{ii}.f_ss,'color', colseq{ii})
  xlabel('t [s]')
  ylabel('fss')
  ylim([0 50])

  hold on
  plot(tflattopgrid(ii)*ones(2,1),[0 100],'color', colseq{ii},'LineStyle','--')
  hold on
  subplot(345)
  plot(out{ii}.rhogauss, 1e-6*interp1(out{ii}.time, out{ii}.jpargauss',tflattopgrid(ii)))
  xlabel('rho')
  ylabel('jpar [MA/m\^{-2}]')

  title('End of ramp-up')
  hold on
  subplot(346)
  plot(out{ii}.rhogauss, interp1(out{ii}.time, out{ii}.qgauss',tflattopgrid(ii)))
  hold on
  xlabel('rho')
  ylabel('q')

  hold on
  subplot(347)
  plot(out{ii}.rhogauss, interp1(out{ii}.time, out{ii}.uplgauss',tflattopgrid(ii)))
  xlabel('rho')
  ylabel('upl')

  hold on
  subplot(348)
  plot(out{ii}.rhogauss, 1e-3*interp1(out{ii}.time, out{ii}.tegauss',tflattopgrid(ii)))
  xlabel('rho')
  ylabel('te [keV]')

  hold on
  subplot(349)
  plot(out{ii}.rhogauss, 1e-6*interp1(out{ii}.time, out{ii}.johgauss',tflattopgrid(ii)))
  xlabel('rho')
  ylabel('joh [MA/m\^{-2}]')

  hold on
  subplot(3,4,10)
  plot(out{ii}.rhogauss, 1e-6*interp1(out{ii}.time, out{ii}.jbsgauss',tflattopgrid(ii)))
  xlabel('rho')
  ylabel('jbs [MA/m\^{-2}]')

  hold on
  subplot(3,4,11)
  plot(out{ii}.rho, 1e-6*interp1(out{ii}.time, out{ii}.Ip',tflattopgrid(ii)))
  xlabel('rho')
  ylabel('encl. Ip [MA]')

  hold on
  subplot(3,4,12)
  plot(out{ii}.rhogauss, 1e-19*interp1(out{ii}.time, out{ii}.negauss',tflattopgrid(ii)))
  xlabel('rho')
  ylabel('ne [1e19m\^{-3}]')

  hold on
  set(gcf,'pos', [282         171        1267         771])

  fig_1b = figure(2);
  plot(1e-6*out{ii}.Ip(end,:),out{ii}.li3,'color',colseq{ii})
  hold on
  xlabel('Ip [MA]')
  ylabel('li3')
  legend(num2str(tflattopgrid'))
end

if save_figure % Save figures in .png format
  if exist('Images','dir')~=7; system('mkdir Images'); end % if not /Images create folder
  print(fig_1a,'Images/ex1a.png','-dpng','-painters');
  print(fig_1b,'Images/ex1b.png','-dpng','-painters');
end

%% SOLUTION 2
clear simres out
% Back to default Ip trace
tFT = 100;
U(1,:) = rampfun(params.tgrid,0,2e6,tFT,12e6); % input Ip trace
tstartgrid = [30 50 70];
for ii=1:numel(tstartgrid)
    tstart = tstartgrid(ii);
    U(model.nbhcd.uind(model.nbhcd.ind_power),:) = 0;
    U(model.nbhcd.uind(model.nbhcd.ind_power),params.tgrid>tstart) = 16.5e6;
    U(model.echcd.uind(model.echcd.ind_power),:) = 0;
    simres{ii} = RAPTOR_predictive(x0,g,V,U,model,params);
    out{ii} = RAPTOR_out(simres{ii},model,params);
    % we can run a second set of simulations, turning of the reduction of
    % diffusion due to negative magnetic shear
    params_noITB = params;
    params_noITB.chi_e.aitb = 0;
    simres_noITB{ii} = RAPTOR_predictive(x0,g,V,U,model,params_noITB);
    out_noITB{ii} = RAPTOR_out(simres_noITB{ii},model,params_noITB);
 end
RAPTOR_plot_GUI(out);

% Turning on the heating very early slows the current penetration

%%
% Let us plot some global time traces
% starting the NBI heating leads to a rise in electron temperature (Te0)
% slowing down current diffusion
% note that starting the NBI heating leads to:
% * slower rise of the central current density (jpar0)
% * reduction of the edge loop voltage (Inb and higher signeo)
% * negative shear

colseq = {'b','r',[0 0.4 0]};
figure;
for ii=1:numel(out)
  fig_2a = figure(1);
  subplot(321)
  plot(out{ii}.time, 1e-6*(out{ii}.Pnbe(end,:)+out{ii}.Pnbi(end,:)),'color',colseq{ii})
  xlabel('t [s]')
  ylabel('Pnb [MW]')
  ylim([0 20])

  hold on
  subplot(322)
  % plot(out{ii}.time, 1e-6*(out{ii}.Ip(end,:)),'color',colseq{ii})
  plot(out{ii}.time, min(out{ii}.sheargauss,[],1),'color',colseq{ii})
  xlabel('t [s]')
  ylabel('min shear')
  ylim([-2 0])

  hold on
  plot(tstartgrid(ii)*ones(2,1),[-100 0],'color', colseq{ii},'LineStyle','--')
  hold on
  subplot(323)
  plot(out{ii}.time, out{ii}.upl(end,:),'color', colseq{ii})
  xlabel('t [s]')
  ylabel('Upl [V]')
  ylim([0 2.5])

  hold on
  plot(tstartgrid(ii)*ones(2,1),[0 100],'color', colseq{ii},'LineStyle','--')
  hold on
  subplot(324)
  plot(out{ii}.time, out{ii}.li3,'color', colseq{ii})
  xlabel('t [s]')
  ylabel('li3')
  ylim([0 1.3])

  hold on
  plot(tstartgrid(ii)*ones(2,1),[0 100],'color', colseq{ii},'LineStyle','--')
  hold on
  subplot(325)
  plot(out{ii}.time, 1e-3*out{ii}.te0,'color', colseq{ii})
  xlabel('t [s]')
  ylabel('te0 [keV]')
  ylim([0 2])

  hold on
  plot(tstartgrid(ii)*ones(2,1),[0 100],'color', colseq{ii},'LineStyle','--')
  hold on
  subplot(326)
  plot(out{ii}.time, 1e-6*interp1(out{ii}.rho,out{ii}.jpar,0.),'color', colseq{ii})
  xlabel('t [s]')
  ylabel('jpar0 [MA/m\^2]')
  ylim([0 1.5])

  hold on
  plot(tstartgrid(ii)*ones(2,1),[0 100],'color', colseq{ii},'LineStyle','--')
  hold on
end
subplot(321)
plot(out{ii}.time, 1e-6*(out{ii}.Ip(end,:)),'k--')
text(135,12,'Ip [MA]')
legend(num2str(tstartgrid'),'Location','SouthEast')

set(gcf,'pos', [625   257   657   683])

if save_figure % Save figures in .png format
  if exist('Images','dir')~=7; system('mkdir Images'); end % if not /Images create folder
  print(fig_2a,'Images/ex2a.png','-dpng','-painters');
end
%%
% the case with early heating develops the strongest magnetic shear
% we can compare the radial profiles, with respectively the improved
% confinement for negative shear turned on/off:
tplot = 35;

fig_2b = figure;
subplot(221)
plot(out{1}.rho,interp1(out{1}.time,out{1}.q',tplot), ...
     out_noITB{1}.rho,interp1(out_noITB{1}.time,out_noITB{1}.q',tplot))
ylabel('q')
legend('ITB','no ITB')
subplot(222)
plot(out{1}.rho,interp1(out{1}.time,out{1}.shear',tplot), ...
     out_noITB{1}.rho,interp1(out_noITB{1}.time,out_noITB{1}.shear',tplot))
ylabel('shear')
subplot(223)
plot(out{1}.rho,1e-3*interp1(out{1}.time,out{1}.te',tplot), ...
     out_noITB{1}.rho,1e-3*interp1(out_noITB{1}.time,out_noITB{1}.te',tplot))
xlabel('rho')
ylabel('Te [keV]')
subplot(224)
plot(out{1}.rho,1e-3*interp1(out{1}.time,out{1}.tep',tplot), ...
     out_noITB{1}.rho,1e-3*interp1(out_noITB{1}.time,out_noITB{1}.tep',tplot))
ylim([-Inf 0])
xlabel('rho')
ylabel('Teprime [keV]')

% in case chi_e.aitb = 1, we indeed see negative shear leading to a steeper
% radial gradient of Te

if save_figure % Save figures in .png format
  if exist('Images','dir')~=7; system('mkdir Images'); end % if not /Images create folder
  print(fig_2b,'Images/ex2b.png','-dpng','-painters');
end
%% SOLUTION 3
clear simres out
tFT = 100;
U(1,:) = rampfun(params.tgrid,0,2e6,tFT,12e6); % input Ip trace
U(model.nbhcd.uind(model.nbhcd.ind_power),:) = 0;
U(model.nbhcd.uind(model.nbhcd.ind_power),params.tgrid>50) = 33e6; % early NBI

tstart = 75;
for ii=1:3
  U(model.echcd.uind(model.echcd.ind_power(1:3)),:) = 0; % ECCD
  U(model.echcd.uind(model.echcd.ind_power(ii)),params.tgrid>tstart) = 20e6; % ECCD
  simres{ii} = RAPTOR_predictive(x0,g,V,U,model,params);
  out{ii} = RAPTOR_out(simres{ii},model,params);
end
hf = RAPTOR_plot_GUI(out);

%%
% Adding off-axis ECCD at rho=0.4 gives (at least transiently) a
% reverse-shear q profile and thus triggers a transport barrier, as is
% visible from the locally very steep temperature gradient.

colseq = {'b','r',[0 0.4 0]};
figure;
for ii=1:numel(out)
  fig_3a = figure(1);
  subplot(311)
  plot(out{ii}.rhogauss, 1e-6*interp1(out{1}.time ,out{ii}.jecgauss',100))
  ylabel('jec [MA/m\^2]')

  hold on
  subplot(312)
  plot(out{ii}.rhogauss, interp1(out{1}.time ,out{ii}.qgauss',100))
  ylabel('q')

  hold on
  subplot(313)
  plot(out{ii}.rhogauss, 1e-3*interp1(out{1}.time ,out{ii}.tegauss',100))
  xlabel('rho')
  ylabel('Te [keV]')
  legend('ECH on-axis','ECCD at rho=0.3','ECCD at rho=0.4')
  hold on

  fig_3b = figure(2);
  subplot(321)
  plot(out{ii}.time, 1e-6*(out{ii}.Pnbe(end,:)+out{ii}.Pnbi(end,:)),'color',colseq{ii})
  ylabel('Pnb [MW]')
  ylim([0 40])

  hold on
  subplot(322)
  plot(out{ii}.time, 1e-6*(out{ii}.Pec(end,:)),'color',colseq{ii})
  ylabel('Pec [MW]')
  ylim([0 30])

  hold on
  plot(100*ones(2,1),[0 100],'k--')
  hold on
  subplot(323)
  plot(out{ii}.time, out{ii}.upl(end,:),'color', colseq{ii})
  ylabel('Upl [V]')
  ylim([0 2.5])

  hold on
  plot(100*ones(2,1),[0 100],'k--')
  hold on
  subplot(324)
  plot(out{ii}.time, out{ii}.li3,'color', colseq{ii})
  xlabel('t [s]')
  ylabel('li3')
  ylim([0 1.3])

  hold on
  plot(100*ones(2,1),[0 100],'k--')
  hold on
  subplot(325)
  plot(out{ii}.time, 1e-3*out{ii}.te0,'color', colseq{ii})
  xlabel('t [s]')
  ylabel('te0 [keV]')
  ylim([0 4])

  hold on
  plot(100*ones(2,1),[0 100],'k--')
  hold on
  subplot(326)
  plot(out{ii}.time, min(out{ii}.qgauss,[],1),'color',colseq{ii})
  xlabel('t [s]')
  ylabel('min q')
  ylim([0 3])

  hold on
  plot(100*ones(2,1),[0 100],'k--')
  hold on
end
figure(1)
set(gcf,'pos', [625   257   657   683])
subplot(311);
ylim([0,0.5])
figure(2)
set(gcf,'pos', [625   257   657   683])
ax = subplot(321);
plot(100*ones(2,1),[0 100],'k--')
plot(out{ii}.time, 1e-6*(out{ii}.Ip(end,:)),'k--')
lgd = legend('ECH on-axis','ECCD at rho=0.3','ECCD at rho=0.4');
pos = lgd.Position; lgd.Position = [pos(1)+0.05,pos(2)-0.05,pos(3),pos(4)];
text(135,12,'Ip [MA]')

if save_figure % Save figures in .png format
  if exist('Images','dir')~=7; system('mkdir Images'); end % if not /Images create folder
  print(fig_3a,'Images/ex3a.png','-dpng','-painters');
  print(fig_3b,'Images/ex3b.png','-dpng','-painters');
end
