%% ITER H-mode scenario
close all hidden;
clear;
% paths
run(fullfile(fileparts(mfilename('fullpath')),'..','..','RAPTOR_path.m')); % add RAPTOR path

% ramp function
rampfun = @(t,tmin,ymin,tmax,ymax) max(ymin,min((ymax-ymin)/(tmax-tmin)*(t-tmin),ymax-ymin)+ymin); % anonymous function for ramps
[config] = RAPTOR_config('ITER'); % load default ITER params

% RAPTOR run options
config.numerics.restol = 1e-8;

config.debug.iterplot = 0;
config.debug.iterdisp = 100;
config.debug.verbosity = -1;
% physics modules
config.chi_e = RAPTORmodule('chi_BgB');

% ECHCD - gaussian
config.echcd = RAPTORmodule('echcd_gaussian');
config.echcd.Configuration.n_units = 3;

% add physics-based NBI module
config.nbhcd = RAPTORmodule('hcd_gaussian');
config.nbhcd.Configuration.n_units = 1;

%% Timing
t_0    = 40;
t_end  = 140;

t_flattop = 100; % flattop current time
t_nbistart = 85; t_nbimax = 110; % NBI timing
% H mode
t_LH = 100; % LH transition time
d_LH = 5; % LH transition duration

% define time grid
Ts1 = 0.2; % fastest sample time
Ts2 = 1;    % slow sample time

% time grid
config.grid.tgrid = unique(...
  [t_0  :Ts1 :t_0+1,...
  t_0+1:Ts2 :t_LH,...
  t_LH:Ts1:t_LH+2,...
  t_LH+2:Ts2:t_end]);

%% spatial grid
nrho=21;
config.grid.rhogrid = linspace(0,1,nrho);
config.grid.rhogrid = [0:0.05:0.9,0.925:0.025:1];
config.te.BC.defaultValue = 300;
config.ni.method = 'nescal';
config.ti.method = 'tescal';

%% config modification
config.saw.params.active=false;
%% generate model, params, init, geometry g, kinetic profiles v
[model,params,init,g,v,U] = build_RAPTOR_model(config);

params.chi_e.cneo = 0.3;
params.chi_e.aegb = 1.8 * params.chi_e.aegb;
params.chi_e.aeb  = 0.3 * params.chi_e.aeb;

params.palpha.active = true;
params.pbrem.active = true;
params.pei.active = true;
params.prad.active = true;

% ECHCD - gaussian
params.echcd.active = true;
params.echcd.rdep = [0.05 0.2 0.3]; %
params.echcd.wdep = [.1 .15 .15];  %
params.echcd.cd_eff = [5 5 15];  % current drive efficiency factor

% add physics-based NBI module
params.params.rdep = [0.23]; %
params.params.wdep = [0.15]; % broad heating
params.wdep_out = [1.1]; % broad heating
params.cd_eff = 4; % current drive
params.active = true;
params.frac_to_electrons = 0.72;

%% Inputs
% plasma current
Ip0 = 4e6;
Ipflattop = 12e6;

U(1,:) = rampfun(params.tgrid,t_0,Ip0,t_flattop,Ipflattop);

% NBI
Pnb_max = 33e6;
U(model.nbhcd.uind,:) = rampfun(params.tgrid,t_nbistart,0,t_nbimax,Pnb_max); % NBI

% ECCD
% rdep = [0.05 0.2 0.3]; %
U(model.echcd.uind(1),:) = interp1([40 60 95 105],[0e6 2e6 0 0],params.tgrid,'linear',0);
U(model.echcd.uind(2),:) = interp1([40 60 95 105],[5e6 5e6  14e6 0],params.tgrid,'linear',0);
U(model.echcd.uind(3),:) = interp1([95 105 140],[0 18e6 37e6],params.tgrid,'linear',0);
%plot(params.tgrid,U0(3:5,:),'r',params.tgrid,sum(U0(3:5,:)))

%% initial conditions
init.newidth = 2;
init.ni0scal = 0.9;
init.niescal = 0.9;
init.ti0scal = 1;
init.tiescal = 1;

init.te0 = 2e3;
init.tewidth = 0.6;
init.ze0 = 1.6;
init.zee = 1.6;
init.jpow = 1;
init.Ip0 = Ip0;

% Construct kinetic profiles according to the new settings.
v = build_kinetic_profiles(model,init);
% Define the initial condition
x0 = RAPTOR_initial_conditions(model,init,g(:,1),v(:,1));

%% Assign V

V = v*ones(1,numel(params.tgrid));

% density
neparam = neHmodeEvo; % LH transition effect on ne evolution
neparam.LHtime = t_LH;
neparam.LHduration = d_LH; % duration
neparam.HLtime = t_end+10; % not in simulation

neparam.peaking = 1;
neparam.neL_t = rampfun(params.tgrid,t_LH,4e19,t_LH+d_LH,8e19);
neparam.neH_t = 6e19;

ne = neHmodeEvo(model,params,neparam);
V(model.ne.vind,:)    =  model.ne.Lamgauss\ne;

% hmode
params.hmode.active = 1;

V(model.hmode.vind.activation,:) = rampfun(params.tgrid,t_LH,0,t_LH+d_LH,1);
V(model.hmode.vind.rhoped,:) = 0.9;
V(model.hmode.vind.gteped,:) = -(4.3e3-config.te.BC.defaultValue)/0.1;

%% H-mode simulation
simres = RAPTOR_predictive(x0,g,V,U,model,params);

%% output
out = RAPTOR_out(simres,model,params);

RAPTOR_plot_GUI(out,'overview');