%% Script to define and run AUG H-mode example
close all hidden; clear;
dosave = false;

%%
% paths
run(fullfile(fileparts(mfilename('fullpath')),'..','..','RAPTOR_path.m')); % add RAPTOR path

% ramp function
rampfun = @(t,tmin,ymin,tmax,ymax) max(ymin,min((ymax-ymin)/(tmax-tmin)*(t-tmin),ymax-ymin)+ymin); % anonymous function for ramps
[config] = RAPTOR_config('AUG'); % load default ITER params

% Spatial grid
nrho=21;
config.grid.rhogrid = unique(sort([linspace(0,1,nrho),0.9:0.025:1]));

% RAPTOR run options
config.numerics.restol = 1e-4;

config.debug.plotEachNewtonStep = false;
config.debug.plotJacobians      = false;
config.debug.checkstateequation = false;
config.debug.checkfwsens = false;
config.debug.keyboard_at = [];

config.debug.iterplot = 0;
config.debug.iterdisp = 100;
config.debug.verbosity = -1;

% % physics modules
config.chi_e = RAPTORmodule('chi_BgB');

% config.chi_e = RAPTORmodule('FF');

config.te.BC.defaultValue = 100; % boundary c. for Te [eV]
config.te.BC.rho = 1.;

config.saw = saw('porcelli'); % sawtooth model
config.saw.params.s1crit = 0.3;
config.saw.params.check = false;

%config.saw = saw('none'); % deselect this to deactivate sawteeth

% add physics-based NBI module
config.nbhcd = RAPTORmodule('hcd_gaussian');
config.nbhcd.Configuration.n_units = 1;

% ECHCD - gaussian
config.echcd = RAPTORmodule('echcd_gaussian');
config.echcd.Configuration.n_units = 2;

config.equi.B0 = 1.5;
%% Timing   
t_0    = 0;
t_LH = 0.8; % LH transition time
t_flattop = 1; % flattop current time
t_rampdown = 5; % flattop end time
t_HL = 5.3;
t_end  = 6;

t_nbistart = 0.7;
t_nbimax = 1; % NBI timing

% define time grid
Ts1=0.001; % fastest sample time
Ts2=0.005;    % slow sample time

% time grids
tgrid_standard = [t_0:Ts2:t_end];
tgrid_startup = [t_0:Ts1:t_0+10*Ts1]; % startup covergence
tgrid_LHtransition = t_LH + [-10*Ts1:Ts1:20*Ts1]; % grid around LH transition
tgrid_HLtransition = t_HL + [-10*Ts1:Ts1:20*Ts1]; % grid around LH transition

config.grid.tgrid = sort(unique([tgrid_standard,...
                                 tgrid_startup,...
                                 0*tgrid_LHtransition,...
                                 0*tgrid_HLtransition]));
                  

%% define model
[model,params,init,g,v,U] = build_RAPTOR_model(config);

% params.chi_e.aitb = 0;

params.chi_e.check = false;
params.chi_e.aegb = 2*params.chi_e.aegb;
params.chi_e.aeb  = 2*params.chi_e.aeb;

params.palpha.active = false;
params.pbrem.active = true;
params.pei.active = true;
params.prad.params.active = true;

% add physics-based NBI module
params.nbhcd.active = true;
params.nbhcd.rdep = [0.0]; %
params.nbhcd.wdep = 0.3;
params.nbhcd.wdep_out = 0.5;

% ECHCD - gaussian
params.echcd.active = true;
params.rdep = [0.35 0.45]; %
params.wdep = [.1 .1];  %
params.cd_eff = [10 10];  % current drive efficiency factor

% Hmode settings
params.hmode.active = true; %{JL} Precedente: config.hmode.params.active = true;

%% inputs
% plasma current
% inputs
Ip0 = 300e3;
Ipflattop = 1e6;

% Ip
U0(1,:) = rampfun(params.tgrid,t_0,Ip0,t_flattop,Ipflattop)  - ... % rampup
     rampfun(params.tgrid,t_end-1,0,t_end,Ipflattop-Ip0) ; % rampdown

% NBI power
PNBI_max = 5e6;
U0(model.nbhcd.uind,:) = rampfun(params.tgrid,t_nbistart,0,t_nbimax,PNBI_max) - ... % NBI
          rampfun(params.tgrid,t_rampdown,0,t_end,PNBI_max);

% ECCD powers (rho fixed)
ioff = (params.tgrid<3)|(params.tgrid>3.5);
U0(model.echcd.uind(1),:) = 1000e3;
U0(model.echcd.uind(1),ioff) = 0;

ioff = (params.tgrid<4)|(params.tgrid>4.5);
U0(model.echcd.uind(2),:) = 1000e3;
U0(model.echcd.uind(2),ioff) = 0;

%% initial conditions 
init.newidth = 2;
init.niwidth = 2;
init.te0 = 500;
init.tewidth = 0.6;
init.ze0 = 2.5;
init.zee = 2.5;
init.jpow = 3;
init.Ip0 = Ip0;

% Construct kinetic profiles according to the new settings.
v = build_kinetic_profiles(model,init);
% Define the initial condition
x0 = RAPTOR_initial_conditions(model,init,g(:,1),v(:,1)); 

%% Density profile evolution
v = v*ones(1,numel(params.tgrid)); % init
 
neparam = neHmodeEvo; % LH transition effect on ne evolution
neparam.LHtime = t_LH;
neparam.LHduration  = 0.1;
neparam.HLtime  = t_HL;
neparam.HLduration  = 0.1;
neparam.peaking = 1;
neparam.neL_t = rampfun(params.tgrid,t_0,1e19,t_flattop,4e19)- ...
                rampfun(params.tgrid,t_rampdown,0e19,t_end,3e19);

neparam.neH_t = 5e19;

ne = neHmodeEvo(model,params,neparam);

v(model.ne.vind,:) = model.ne.Lamgauss\ne; % electron density
v(model.ni.vind,:) = .9*v(model.ne.vind,:); % ion density

%% Hmode timing
LH_duration = 0.05; % duration for LH/HL transition
v(model.hmode.vind.activation,:) = rampfun(params.tgrid,t_LH,0,t_LH+LH_duration,1) - ...
                        rampfun(params.tgrid,t_HL,0,t_HL+LH_duration,1);
v(model.hmode.vind.rhoped,:) = 0.9;
v(model.hmode.vind.gteped,:) = -(2.3e3-config.te.BC.defaultValue)/0.1;
    
%% BASELINE H-mode simulation
simres = RAPTOR_predictive(x0,g,v,U0,model,params);
%%
out = RAPTOR_out(simres,model,params);
%%
pp = RAPTOR_plot_GUI('overview');
pp.plots{29}.ylim = [-1 5];
pp.plots{40}.ylim = [0 10];
%
RAPTOR_plot_GUI(out,pp);