%% Simple example using the MS model
close all hidden;
run(fullfile(pwd,'..','RAPTOR_path.m'));

%% Configuration
% Load default parameters
config = RAPTOR_config('TCV'); 
config.equi.filenames = [{'TCV_41083_cheasedata'}];
% Specify time grid for input data
config.equi.times = [0.];
% Define time grid: time interval from 0.0 to 450.0 ms
config.grid.tgrid = [0:0.001:1.]; 

% MS model
config.ne.method = 'state';
config.vpdn_e = RAPTORmodule('vpdn_MS');
config.chi_e = RAPTORmodule('chi_MS');
config.hmode.params.active = true;
config.hmode = hmode('none');

% ECHCD
config.echcd.Configuration.n_units = 1;

% Build model and get params, init structures, geometry g, kinetic profiles
% v, actuator inputs U
[model,params,init,g,v,U] = build_RAPTOR_model(config);

params.pei.active = false;
params.pbrem.active = false;
params.prad.active = false;

% set ECH/ECCD parameters
params.echcd.active = true;
params.echcd.rdep = [0.0]; %
params.echcd.wdep = [0.35]; % deposition width
params.echcd.cd_eff = [0]; % CD efficiency

% plasma current with a ramp-up & a ramp-down
rampfun = @(t,tmin,ymin,tmax,ymax) max(ymin,min((ymax-ymin)/(tmax-tmin)*(t-tmin),ymax-ymin)+ymin); % anonymous function for ramps

ramp_up = rampfun(params.tgrid, 0., 0.7, 0.15, 1);
ramp_down = rampfun(params.tgrid, 1., 0.7, 0.85, 1);
Ip_ramp = ramp_up.*ramp_down; 
U(1,:)= 1.e6.*Ip_ramp; % input Ip trace with a nominal flat-top value to 500kA

% We want to see an H-mode from 0.3 to 0.7s with 0.05s ramp, L-mode elsewhere
tLH = [0.25 0.3]; 
tHL = [0.7 0.75]; 
hmode_ramp = ones(size(config.grid.tgrid)); 
ramp_LH = rampfun(params.tgrid, tLH(1), 0, tLH(2), 1);
ramp_HL = rampfun(params.tgrid, tHL(2), 0, tHL(1), 1);
hmode_ramp = hmode_ramp.*ramp_LH.*ramp_HL; 

% Now let's say ECHCD triggers the H-mode
U(2,:) = hmode_ramp.*0.5e6;

% He98 
params.chi_e.hh = [0.8 1.];

% nel
params.vpdn_e.nel = [6. 10.].*1e19;

% MS params
params.chi_e.gp = 1000; 
params.chi_e.fb_on = true; 

% V matrix
Zeff = 2.; 
V = repmat(v, 1, numel(params.tgrid));
V(model.ze.vind, :) = Zeff;
V(model.hmode.vind.activation,:) = hmode_ramp; 
V(model.te.BC.vind_value,:) = (60-25).*hmode_ramp+25; 
V(model.ne.BC.vind_value,:) = (0.5.*hmode_ramp+0.5).*1e19; % (Ip_ramp+hmode_ramp)

% Define initial conditions for plasma state
init.te0 = 500;
init.ti0 = init.te0;
init.ne0 = 5e19; 
init.ni0 = 5e19;
init.Ip0 = U(1,1);
x0 = RAPTOR_initial_conditions(model,init,g(:,1),v(:,1)); 

%% Run RAPTOR simulation
disp('jac_rhoinv = false');
params.chi_e.jac_rhoinv = false; 
params.vpdn_e.jac_rhoinv = false; 
tic
simres = RAPTOR_predictive(x0,g,v,U,model,params);
toc
if numel(simres.exitflags) ~= 1
  out_jac_false = RAPTOR_out(simres,model,params);
else
  error('1 time step only');
end

disp('jac_rhoinv = true');
params.chi_e.jac_rhoinv = true; 
params.vpdn_e.jac_rhoinv = true; 
tic
simres = RAPTOR_predictive(x0,g,v,U,model,params);
toc
if numel(simres.exitflags) ~= 1
  out_jac_true = RAPTOR_out(simres,model,params);
else
  error('1 time step only');
end

RAPTOR_plot_GUI({out_jac_false, out_jac_true});