%% Tutorial running RAPTOR using IMAS IDS inputs
disp('This tutorial requires an IMAS installation')
return
%%

close all hidden;
%% INPUT/OUTPUT CONFIGURATION
% IMAS shot and run number
% corresponding to a DINA-JINTRAC simulation
shot                = 134173; % shot number
run_in              = 106; % run number
input_user_or_path  = 'public'; % owner of the database
input_database      = 'ITER'; % name of the machine
run_out             = 20; % run number of new entry
output_user_or_path = getenv('USER');
output_database     = input_database;

%% OPEN INPUT DATAFILE TO GET DATA FROM IMAS SCENARIO DATABASE

if ~exist('imas_open_env','file')
  warning('imas_ope_env.m not found. Can not complete tutorial. Please check IMAS installation.\n');
  return
end
disp('=> Open input datafile') 
% opening an existing Data entry
idx_in  = imas_open_env('ids',shot,run_in,input_user_or_path,input_database,'3');

%% READ INPUT IDSS FROM LOCAL DATABASE
% to load ftom input IDS over whole time window 

disp('=> Read input IDSs')
% reading the content of an IDS variable from a Data Entry
fs_equilibrium = ids_get(idx_in,'equilibrium');
fs_core_profiles = ids_get(idx_in,'core_profiles');
fs_core_sources = ids_get(idx_in,'core_sources');
fullshot_summary  = ids_get(idx_in,'summary');

%% CREATE OUTPUT DATAFILE
disp('=> Create output datafile')
% creating a new Data entry
idx_out = imas_create_env('ids',shot,run_out,0,0,output_user_or_path,output_database,'3');

%% EXECUTE PHYSICS CODE: RAPTOR
disp('=> Configure RAPTOR')

run ../RAPTOR_path.m
timegrid= [fullshot_summary.time(1):790];

% Load default configuration structure for ITER (default settings)
[config] = RAPTOR_config('ITER');
config.grid.rhogrid = linspace(0,1,26);
rhoped = 0.9333; % we set the pedestal location, where the T_e BC acts, at rhoped
config.grid.rhogrid = unique(sort([config.grid.rhogrid rhoped]));
config.grid.tgrid = timegrid; 

% configuration physics models
config.ti.method = 'direct'; 
% find ion density and Zeff from quasi-neutrality and Zeff formula
config.ni.method = 'qnze';
config.ze.method = 'qnze';
config.n1.method = 'nescal';
config.n2.method = 'nescal';
config.n3.method = 'nescal';
% transport
config.chi_e = RAPTORmodule('chi_FF');
% set the impurity species
config.atom.Zi = 1; config.atom.Ai = 2.5; % D,T main ions
config.atom.Z1=2; config.atom.A1=4.003; % helium
config.atom.Z2=74; config.atom.A2=183.84; % tungsten
config.atom.Z3=4; config.atom.A3=9.0122; % beryllium
config.palpha = RAPTORmodule('palpha');
config.pei = RAPTORmodule('pei');
config.pbrem = RAPTORmodule('pbrem');
% ADAS cooling factors for impurity radiation:
config.prad = RAPTORmodule('prad_adas');
config.saw = saw('none');
config.hmode = hmode('imposed');
config.te.BC.rhoSource = 'v';
config.te.BC.rho = rhoped;
config.hmode.params.rhoped = rhoped;

config.echcd = RAPTORmodule('hcd_manual');
config.ichcd = RAPTORmodule('hcd_manual');
config.nbhcd = RAPTORmodule('hcd_manual');
config.lhhcd = RAPTORmodule('hcd_manual');

equil_path = fullfile('~/equi_imas');
config.equi.source = 'IMAS';
config.equi.times = timegrid(1:10:end);
config.equi.filenames = cell(1, numel(config.equi.times));
for tii = 1:numel(config.equi.times)
  t_now = config.equi.times(tii);
  if tii==1
    folderID = fullfile(equil_path, sprintf('shot_%i_run_%i_database_%s', shot,run_in,input_database));
    if ~exist(folderID,'dir')
      fprintf('mkdir %s for IMAS equil\n',folderID);
      mkdir(folderID)
    end
    config.envopts.equil_path = folderID;
    fprintf('RAPTOR equil path %s\n',folderID);
  end
  filename = sprintf('time_%2.4fs_imasdata.mat',t_now);
  fileID = fullfile(folderID, filename);
  if ~exist(fileID,'file')
    % get the equilibrium IDS slice at the requested time
    equi = ids_get_slice(idx_in,'equilibrium',t_now,1);
    eval(['save ' fileID ' equi;']);
    fprintf('saved IMAS equi into file: %s \n',fileID);
  end
  config.equi.filenames{tii} = filename;
end

disp('=> Build RAPTOR model')
[model,params,init,g,v,U] = build_RAPTOR_model(config);

disp('=> Map IDSs to RAPTOR_inputs')
RAPTOR_inputs = map_IDS_to_RAPTOR_inputs(fullshot_summary, fs_core_profiles, fs_core_sources, fs_equilibrium, model, config.grid.tgrid);
%%
ntime = numel(timegrid);
V = repmat(v,1,ntime);

% set HCD
hcdsrc = {'nb','lh','ic'};
for hcdidx = 1:numel(hcdsrc)
  idstr = strcat(hcdsrc{hcdidx}, 'hcd');
  if strcmp(model.(idstr).name, 'hcd_manual')
  	params.(idstr).Pauxe = RAPTOR_inputs.(strcat('p',hcdsrc{hcdidx},'egauss'));
  	params.(idstr).Pauxi = RAPTOR_inputs.(strcat('p',hcdsrc{hcdidx},'igauss'));
  	params.(idstr).jaux  = RAPTOR_inputs.(strcat('j',hcdsrc{hcdidx},'gauss'));
  end
end
switch model.echcd.name
  case 'hcd_manual'
    params.echcd.Pauxe = RAPTOR_inputs.pecgauss;
    params.echcd.Pauxi = zeros(model.rgrid.nrhogauss, ntime);
    params.echcd.jaux = RAPTOR_inputs.jecgauss;
end

% physics model settings
params.palpha.active = 1;
params.palpha.DTfraction = 0.485;
params.pei.active = 1;
params.pei.active = params.palpha.DTfraction;
% radiation
params.pbrem.active = 1;
params.prad.active = 1;
% let's use a signeo patch to mimic sawteeth
params.neos.sigma_csawtooth = 1;

% transport model
V(model.hmode.vind.activation, :) = 1;
V(model.hmode.vind.te_rhoedge, :) = RAPTOR_inputs.te(end,:);
V(model.te.BC.vind_value, :) = interp1(model.rgrid.rhogauss, RAPTOR_inputs.tegauss, rhoped);

params.chi_e.cano = 0.02; 
params.chi_e.chiecentral = 0;
params.chi_e.aitb = 0;
params.chi_e.te_constant = 1.2; 

% set kinetic profiles that are not solved for:
V(model.ti.vind, :) = abs(model.ti.Lamgauss\abs(RAPTOR_inputs.tigauss));
V(model.ne.vind, :) = abs(model.ne.Lamgauss\abs(RAPTOR_inputs.negauss));
V(model.n1.vind, :) = abs(model.n1.Lamgauss\abs(bsxfun(@rdivide, RAPTOR_inputs.nHegauss, RAPTOR_inputs.negauss)));
V(model.n2.vind, :) = abs(model.ze.Lamgauss\abs(bsxfun(@rdivide, RAPTOR_inputs.nWgauss, RAPTOR_inputs.negauss)));
V(model.n3.vind, :) = abs(model.ze.Lamgauss\abs(bsxfun(@rdivide, RAPTOR_inputs.nBegauss, RAPTOR_inputs.negauss)));

U(1,:) = RAPTOR_inputs.Ip;

% to start the simulation from an initial state corresponding to the IDS
% slice at t0:
x0 = zeros(model.dims.nx,1);
[~,diota_dx] = eval_iota(x0,g(:,1),V(:,1),model,true);
[~,dpsi_dx]  = eval_psi(x0,g(:,1),V(:,1),model,true);
x0(model.psi.xind) = [diota_dx(:,model.psi.xind);dpsi_dx(end,model.psi.xind)]\...
  [1./RAPTOR_inputs.qgauss(:,1); 0];
if strcmp(model.hmode.modeltype, 'imposed')
  core_indices = model.rgrid.rhogauss < model.hmode.rhoped;
  x0(model.te.xind) = model.te.Lamgauss_hmode(core_indices,1:end-1)\RAPTOR_inputs.tegauss(core_indices,1);
  if strcmp(model.ti.method,'state')
    x0(model.ti.xind) = model.ti.Lamgauss_hmode(core_indices,1:end-1)\RAPTOR_inputs.tigauss(core_indices,1);
  end
else
  x0(model.te.xind) = (model.te.Lamgauss\RAPTOR_inputs.tegauss(:,1));
end

% set verbosity
params.debug.verbosity = 1;
params.debug.iterdisp = 10;
params.debug.iterplot = 10;

disp('=> Execute RAPTOR')
simres = RAPTOR_predictive(x0,g,V,U,model,params);
out = RAPTOR_out(simres,model,params);


%% SAVE IDSs INTO OUTPUT FILE

disp('=> Export output IDSs to local database')
RAPTOR_imas_write(idx_out,out,model)
% ids_put(idx_out,'equilibrium',output_eq);
imas_close(idx_in);
imas_close(idx_out);
disp('Done exporting.')

%% some figures: compare to DINA-JINTRAC simulation 

xwdw = [RAPTOR_inputs.time(1) RAPTOR_inputs.time(end)];

figure(1);
subplot(221)
plot(RAPTOR_inputs.time, 1e-6*RAPTOR_inputs.Ip, 'b', ...
     out.time, 1e-6*out.Ip(end,:), 'r--')
legend('JINTRAC', 'RAPTOR')
title('I_p [MA]')   
xlim(xwdw)
   
subplot(222)
plot(RAPTOR_inputs.time, 1e-6*RAPTOR_inputs.Wth, 'b', ...
     out.time, 1e-6*out.Wth, 'r--')
title('W_{th} [MJ]')   
xlim(xwdw)
   
subplot(223)
plot(RAPTOR_inputs.time, RAPTOR_inputs.li3, 'b', ...
     out.time, out.li3, 'r--')  
title('l_{i3}')   
xlim(xwdw)
xlabel('t [s]')
   
subplot(224)
plot(RAPTOR_inputs.time, RAPTOR_inputs.psi_equi(end,:), 'b', ...
     out.time, out.psi(end,:)-out.psi(end,1), 'r--')
title('\psi_e [Wb]')   
xlim(xwdw)
xlabel('t [s]')

%%
plot_time = 200;

figure;
subplot(211)
plot(RAPTOR_inputs.rhogauss,1e-3*interp1(RAPTOR_inputs.time,RAPTOR_inputs.tegauss',plot_time), 'b', ...
     out.rhogauss,1e-3*interp1(out.time,out.tegauss',plot_time),'r--')
legend({'JINTRAC', 'RAPTOR'})
title(sprintf('T_e [keV] @ t=%.1fs',plot_time))
   
subplot(212)
plot(RAPTOR_inputs.rhogauss,interp1(RAPTOR_inputs.time,RAPTOR_inputs.qgauss',plot_time), 'b', ...
     out.rhogauss,interp1(out.time,out.qgauss',plot_time),'r--')
title(sprintf('q @ t=%.1fs',plot_time))   
xlabel('\rho')