%% demo: EKF observer with synthetic measurement Te0
% RAPTOR simulation without EKF observer

run(fullfile(pwd,'..','RAPTOR_path.m')); % add RAPTOR path
addpath(fullfile(pwd,'..','code/diagnostics/generic_diag')); % add path with some basic synthtic diagnostics

[config] = RAPTOR_config; 
config.grid.tgrid = 0:0.001:.1;
[model,params,init,g,v,U] = build_RAPTOR_model(config);
x0 = RAPTOR_initial_conditions(model,init,g(:,1),v(:,1)); 
U(1,:) = 80e3;
simres = RAPTOR_predictive(x0,g,v,U,model,params,'verbosity',2);
out = RAPTOR_out(simres,model,params);

sim_signal = out.te0/1e3; % the measurement we would expect from simulation

%% offline EKF observer

% noise defined additive to the state
% (alternative: noise defined additive to jpar, power sources)
exts.cov_direct_rep = true;
exts.model_cov_factor = 1e-6; 
exts.stddev_diag = 0.1; % try: 1, 0.1, 0.01 define sqrt(covariance) on Te0 measurement

KFparams = RAPTOR_EKF_config;
KFparams.ny = 1;
KFparams.ndx = 0;

KFparams.Ipstop = 50e3; % threshold for stopping filter

% params for design process noise covariance matrix Q
KFparams.jx_sm = 0.6; % radial smoothing psi (first row and column exp(-rho^2/jx_sm^2))
KFparams.px_sm = 0.6; % radial smoothing Te
KFparams.noise_model = ~exts.cov_direct_rep;
KFparams.jx_cov = exts.model_cov_factor*[1^2 0.1^2]; % covariance on axis, at the edge 
KFparams.px_cov = 1e6*exts.model_cov_factor*[1^2 0.1^2];  
KFparams.s0 = 1e-3; % initialize state error covariance matrices Sx0 = KFp.s0*eye(KFm.nx)

KFparams.cov_usegauss = false; % option to choose gauss shape
KFparams.cov_rhopow = 2; % convariance rho scaling: (1-rho^cov_rhopow)
KFparams.t_Rkdecay = 0;
KFparams.g_Rkdecay = 0;
KFparams.Lgain = 1;
KFparams.alpha = 1; % extra factor for weighing past meaurements less (>1)

KFparams.verbose = 1; % no text printed if = 0

[KFmodel,KFparams] = RAPTOR_EKF_config(x0,model,params,KFparams);

model.diagnostics.diagfun = @(x,g,v,u,it,model)diag_te0(x,g,v,u,it,model, exts.stddev_diag);
model.equi.updateequi = 0;

G = repmat(g,1,numel(params.tgrid));
V = repmat(v,1,numel(params.tgrid));

% measurement
Hn = sim_signal;
Hn(out.time>0.04) = 0.35; % after 0.4s, the measurement follows no longer the simulation

[obsres,simres_EKF] = RAPTOR_EKF_sim(G,V,U,Hn,model,params,KFmodel,KFparams);
outobs = RAPTOR_out(simres_EKF,model,params);

%% now let's use a smoother! (Anderson (1979 ) Ch. 7)

[smoothres, simres_RTS] = RAPTOR_RTSsmoother(obsres, simres_EKF, model, KFmodel, KFparams);
outsmo = RAPTOR_out(simres_RTS,model,params);

%% we can calculate errorbars on the estimates

obs_signal = outobs.te0/1e3;
smo_signal = outsmo.te0/1e3;
obs_stddev = zeros(1,numel(params.tgrid));
obs_stddev_check = zeros(1,numel(params.tgrid));
smo_stddev = zeros(1,numel(params.tgrid));
li3_stddev = zeros(1,numel(params.tgrid));
for tii = 1:numel(params.tgrid)
  obs_stddev(tii) = sqrt(obsres.Sk_kmat(model.te.xind(1),model.te.xind(1),tii));
  smo_stddev(tii) = sqrt(smoothres.Sk_Mmat(model.te.xind(1),model.te.xind(1),tii));
  % if you want to calculate errorbar on other quantities:
  % example for li3
  [~,dli3_dx] = eval_li3(simres_EKF.X(:,tii),G(:,tii),V(:,tii),model);
  li3cov = dli3_dx(end,:)*obsres.Sk_kmat(:,:,tii)*dli3_dx(end,:)';
  li3_stddev(tii) = sqrt(li3cov);
end

figure;
plot(out.time, sim_signal, out.time, Hn,'k*:', out.time, obs_signal,'--');
hold on
plot(out.time,smo_signal, 'color', [0 0.4 0])
hold on
plot_shaded_deviation(out.time, Hn, exts.stddev_diag*ones(size(out.time)), [0 0 0], 0.1)
hold on
plot_shaded_deviation(out.time, obs_signal, obs_stddev, [1 0 0], 0.1)
hold on
plot_shaded_deviation(out.time, smo_signal, smo_stddev, [0 0.4 0], 0.1)

title('Te0 [keV]')
ylim([0.2 0.45])
xlabel('t [s]')
legend({'RAPTOR','synthetic measurement','RAPTOR EKF','RAPTOR RTS smoother'})
text(0.02, 0.2 ,sprintf('sqrt(Rk)=%1.1e sqrt(Qk(1,1))=%1.1e\n',sqrt(obsres.Rkmat(:,:,1)),sqrt(obsres.Qk(1,1))))
set(legend,'Position',[0.3565    0.3778    0.4268    0.1333])


%% Bayes illustration
% EKF is a Bayesian estimator!
tind = iround(outobs.time,0.06);

x = 0.2 : 0.001 : 0.5;
normal_dist = @(mu, sigma, x) ( 1/(sigma*sqrt(2*pi)) ) .*exp(-(x-mu).^2 ./ (2*sigma^2));
meas = normal_dist(Hn(tind), exts.stddev_diag, x);
simu = normal_dist(obsres.Xk_km1(model.te.xind(1),tind),sqrt(obsres.Sk_km1mat(model.te.xind(1),model.te.xind(1),tind)), x);
post = normal_dist(obsres.Xk_k(model.te.xind(1),tind),sqrt(obsres.Sk_kmat(model.te.xind(1),model.te.xind(1),tind)), x);
int_product = cumtrapz(x, meas.*simu);
figure;
plot(x, meas,'k', ...
     x, simu, 'b',...
     x, post, 'r')
%hold on
%plot(x, meas.*simu./int_product(end), ':','color',[0 0.4 0])
xlim([0.3 0.4])
xlabel('Te0')
title('probability distribution')
legend('likelihood: measurement','prior: previous estimate propagated', 'posterior: estimate')

%% plot matrices

figure; imagesc(obsres.Qk)
title('Process noise covariance matrix')
xlabel('state vector index')
ylabel('state vector index')

%%

tind = numel(params.tgrid);
figure;
subplot(221)
imagesc(obsres.Sk_kmat(model.psi.xind,model.psi.xind,tind))
subplot(222)
imagesc(obsres.Sk_kmat(model.te.xind,model.psi.xind,tind))
subplot(223)
imagesc(obsres.Sk_kmat(model.psi.xind,model.te.xind,tind))
subplot(224)
imagesc(obsres.Sk_kmat(model.te.xind,model.te.xind,tind))
 

subplot(221);ylabel('psi: state vector index'); title('state cov S'); 
subplot(223);xlabel('psi: state vector index');ylabel('te: state vector index'); 
subplot(224);xlabel('te: state vector index'); 

%% 

tind = numel(params.tgrid);
Gk = eye(KFmodel.nz); 
KF_plot_matrices(obsres.Fkmat(:,:,tind), obsres.Hkmat(:,:,tind), obsres.Sk_km1mat(:,:,tind), obsres.Sk_kmat(:,:,tind), ...
                 obsres.Qk, obsres.Rkmat(:,:,tind), obsres.Omks{tind}, obsres.Lks{tind}, Gk, KFmodel)
