function TS_model = build_TS_model_meq(R_TS, Z_TS, L)
% Builds Thomson scattering model, providing the (R,Z) position of TS 
% measurements, the indexing to map the (R,Z) position of TS onto the 
% LIUQE grid vectorized and dense toroidal grid and correspondent splines.  
%
% NB: no interpolation has been made to obtain psi values on the TS position,
%     just closest neighbour by now with iround

TS_model.R_TS = R_TS;
TS_model.Z_TS = Z_TS;
TS_model.n_TS = numel(TS_model.Z_TS);

%% Indexing between TS and LIUQE poloidal position

rx = L.G.rx;
zx = L.G.zx;
nr = numel(rx);
nz = numel(zx);

% Given a TS point in (R,Z), find the 4 points on LIUQE grid enclosing it
% and save the indexes. The output is a matrix [n_TS,4], containing on each
% row the indexes on the vectorized grid.
% Additionally, store in a [n_TS,1] array the position of the closest LIUQE 
% grid point (useful to discard grid points outside the TS domain).
% Then, compute the corresponding weights for each of the points of the
% grid for the bilinear interpolation.

% Find the closest index between TS and LIUQE grid,
% separately for R and Z coordinates
ir_closest = interp1(rx,1:length(rx),R_TS,'nearest','extrap');
iz_closest = interp1(zx,1:length(zx),Z_TS,'nearest','extrap');

% Vectorize the indexing in i_grid_TS
[Z,R] = meshgrid(1:nz,1:nr);
R = R(:);
Z = Z(:);
w_TS      = zeros(TS_model.n_TS,4);
i_grid_TS = zeros(TS_model.n_TS,4);
i_grid_TS_closest = zeros(TS_model.n_TS,1);

% Store the measurement position in the corresponding
% vectorized R,Z position 
for ii= 1:TS_model.n_TS
  
  % Find index of the lower-west LIUQE grid point, out of the 4 points enclosing
  % each ii-th TS point. 
  rx_aux = rx; zx_aux = zx;
  index_neg_R = (R_TS(ii)-rx_aux)<=0;
  index_neg_Z = (Z_TS(ii)-zx_aux)<=0;
  
  rx_aux(index_neg_R) = NaN;
  zx_aux(index_neg_Z) = NaN;
  
  % lower-west
  iR_i_j = find((R_TS(ii)-rx_aux)==min(R_TS(ii)-rx_aux));
  iZ_i_j = find((Z_TS(ii)-zx_aux)==min(Z_TS(ii)-zx_aux));
  R_i_j = rx(iR_i_j);
  Z_i_j = zx(iZ_i_j);
  
  % lower-east
  iR_i1_j  = iR_i_j+1; 
  iZ_i1_j  = iZ_i_j;
  R_i1_j = rx(iR_i1_j);
  
  % higher-east
  iR_i1_j1 = iR_i_j+1; 
  iZ_i1_j1 = iZ_i_j+1;
  
  % higher-west
  iR_i_j1  = iR_i_j;   
  iZ_i_j1  = iZ_i_j+1;
  Z_i_j1 = zx(iZ_i_j1);
 
  % Find the position on the vectorized grid 
  index_R = R==iR_i_j;
  index_Z = Z==iZ_i_j;
  i_grid_TS(ii,1) = find((index_R)&(index_Z));
  
  index_R = R==iR_i1_j;
  index_Z = Z==iZ_i1_j;
  i_grid_TS(ii,2) = find((index_R)&(index_Z));
  
  index_R = R==iR_i_j1;
  index_Z = Z==iZ_i_j1;
  i_grid_TS(ii,3) = find((index_R)&(index_Z));
  
  index_R = R==iR_i1_j1;
  index_Z = Z==iZ_i1_j1;
  i_grid_TS(ii,4) = find((index_R)&(index_Z));
  
  % Find the closest position on the vectorized grid 
  index_R = R==ir_closest(ii);
  index_Z = Z==iz_closest(ii);
  i_grid_TS_closest(ii) = find((index_R)&(index_Z));
  
  w_TS(ii,1) = (R_i1_j-R_TS(ii))*(Z_i_j1-Z_TS(ii));
  w_TS(ii,2) = (R_TS(ii)-R_i_j) *(Z_i_j1-Z_TS(ii)); 
  w_TS(ii,3) = (R_i1_j-R_TS(ii))*(Z_TS(ii)-Z_i_j);
  w_TS(ii,4) = (R_TS(ii)-R_i_j) *(Z_TS(ii)-Z_i_j);
  
  w_TS(ii,:) = w_TS(ii,:)/((R_i1_j-R_i_j)*(Z_i_j1-Z_i_j));
end

TS_model.w_TS = w_TS;
TS_model.i_grid_TS = i_grid_TS;
TS_model.i_grid_TS_closest = i_grid_TS_closest;

% %% Precompute density basis functions on dense rho_psi-grid
% spline_dense_select = 1;
% [grid_dense,LAMBDA_dense] = build_dense_splines(spline_dense_select,densitymodel);
% TS_model.grid_dense   = grid_dense;
% TS_model.LAMBDA_dense = LAMBDA_dense;
% 
% %% Zero-order hold
% TS_model.zoh = false;
% 
% %% Channel selector
% TS_model.channel_selector = [1,1,1];
% 
% %% Correction using FTR
% TS_model.corr_ftr = true;

end