%% Exercise Session 2:  Plasma current and position determination

%% Load the data and get plasma grid

clear;
fromfile = true; % set true to load from file
fname = 'ex2_data.mat'; 
if fromfile && ~isempty(which(fname))
  load('ex2_data.mat'); % if file exists, load it
  disp('loaded TCV data from file');
else
  % Loading an equilibrium from TCV
  [L,~,LY] = fbt('tcv',61400,1,'selu','v','izgrid',1);
  % uncomment this to save file
  % save(fname,'L','LY');
  disp('file not found, loaded from fbt');
end

%% Define plasma distribution on the x grid

Ix  = zeros(L.nzx,L.nrx); 
Ix(2:end-1,2:end-1) = LY.Iy;

% determine current distribution centroid position
r0  = sum(L.rrx(:).*Ix(:))/sum(Ix(:));
z0  = sum(L.zzx(:).*Ix(:))/sum(Ix(:));
Ip0 = sum(Ix(:)); % calculating the total current 

% plot plasma current distribution inside TCV vac. vessel
figure(1); clf;
imagesc(L.rx,L.zx,Ix); %plots current distribution
axis tight; axis xy equal; 
hold on;
plot(L.G.rl,L.G.zl,'LineWidth',3);  % plot limiter: plasma facing tiles
plot(L.G.rv,L.G.zv,'LineWidth',3);  % plot vacuum vessel behind the limiter
plot(r0,z0,'g*');     % plot current centroid determined above
cb = colorbar; colormap('hot');
cb.Label.String = 'Plasma current [A]';
legend('limiter','vac. vessel','centroid', 'Location', 'northeastoutside');
title('Typical TCV current distribution');

%% Compute expected measurements for the Ix plasma distribution

Ia = LY.Ia;   % pull active coil current values. No noise
% measurements = plasma current + active coil current contributions
Bm  = L.G.Bmx*Ix(:) + L.G.Bma*Ia; % magnetic probes measurements
Ff  = L.G.Mfx*Ix(:) + L.G.Mfa*Ia; % flux loops measurements

%% Add noise to measurements and turn into time-dependant signal

eBm = 10e-3; % 10mT noise
eFf = 10e-3; % 10mWb noise
eIa = 0;     % assume no measurement error on currents

% time grid
time = 0:1e-3:0.1;
nt = numel(time); 
% time evolution
Bm_meas = Bm + eBm*randn(numel(Bm),nt);
Ff_meas = Ff + eFf*randn(numel(Ff),nt);
Ia_meas = Ia + eIa*randn(numel(Ia),nt);

%% Plot measurment data and measurment locations 

figure(2); clf;
% magnetic probes time evolution
subplot(221)
plot(time,Bm_meas);
title('B probe data over time'); xlabel('time [s]'); ylabel('Tesla [T]');
% flux loops time evolution
subplot(223)
plot(time,Ff_meas);
title('\psi_{fluxloop} data over time'); xlabel('time [s]'); ylabel('Weber [Wb]');

% measurements per magnetic probe
subplot(222)
plot(1:numel(Bm),Bm_meas,'.',1:numel(Bm),Bm,'or')
title('B probe spatial dependace')
xlabel('B probe number'); ylabel('Teslas [T]');

% measurements per flux loop
subplot(224)
plot(1:numel(Ff),Ff_meas,'.',1:numel(Ff),Ff,'or');
title('flux loop spatial dependace')
xlabel('flux loop number'); ylabel('Weber [Wb]');

% Plot the positons of the magentic probes and flux loops
figure(3); clf;
subplot(121)
imagesc(L.rx,L.zx,Ix);  axis tight; axis xy equal; 
cb = colorbar('westoutside'); colormap('hot');
cb.Label.String = 'Plasma current [A]';
hold on;
plot(L.G.rl,L.G.zl,'LineWidth',3);
plot(L.G.rv,L.G.zv,'LineWidth',3); 
plot(L.G.rf,L.G.zf,'gx');
text(L.G.rf+0.02,L.G.zf,L.G.dimf, 'FontSize', 8);
legend('limiter','vac. vessel','Flux loops', ...
        'Location', 'southoutside');
title('flux loop array TCV');
xlim([0.55, 1.3]); ylim([-0.85, +0.85]);

subplot(122)
imagesc(L.rx,L.zx,Ix); axis tight; axis xy equal; 
cb = colorbar('westoutside'); colormap('hot');
cb.Label.String = 'Plasma current [A]'; hold on;
plot(L.G.rl,L.G.zl,'LineWidth',3);
plot(L.G.rv,L.G.zv,'LineWidth',3); 
plot(L.G.rm,L.G.zm,'go')
text(L.G.rm+0.02,L.G.zm,L.G.dimm, 'FontSize', 8);
legend('limiter','vac. vessel','B probes', ...
      'Location', 'southoutside');
title('poloidal magnetic probe array TCV');
xlim([0.55, 1.3]); ylim([-0.85, +0.85]);

%% Define new grid boundaries of representation with fewer elements (2x3)
% Split the |x| grid into fewer rectangular regions and let 
% the current distribution be constant in that region.

nr = 2; % number of radial regions
nz = 3; % number of vertical regions
nh = nr*nz; % total number of degrees of freedom
rhgrid = linspace(min(L.G.rx),max(L.G.rx),nr+1);
zhgrid = linspace(min(L.G.zx),max(L.G.zx),nz+1);

%% Build transformation matrix T_{xh} 
% let 'x' be the full (original) grid and 'h' be the reduced representation
% Let I_x = T_{xh}*I_h. Build this matrix T_{xh} here:

Txh = zeros(L.nx, nh);
[rrx,zzx] = meshgrid(L.G.rx,L.G.zx);
for ir = 1:nr
    for iz = 1:nz
        ii = (ir-1)*(nz) + iz; % element index
        % choose region bounded by points defining borders of the rectangles
        txy = (rhgrid(ir) <= rrx) & (rrx < rhgrid(ir+1))...
            & (zhgrid(iz) <= zzx) & (zzx < zhgrid(iz+1)) ;
        Txh(:,ii) = txy(:);
    end
end
% calculate mean r,z of each new grid point for illustration purposes later
rhmean = sum(Txh.*rrx(:),1)./sum(Txh,1);
zhmean = sum(Txh.*zzx(:),1)./sum(Txh,1);

% plot grids
figure(4); clf;
subplot(121);
p   = plot(rrx,zzx,'.b','markersize',2); hold on; 
p1  = plot(rhmean,zhmean,'or'); 
axis equal tight;
[rhh,zhh] = meshgrid(rhgrid,zhgrid); hold on; 
p2 = plot(rhh,zhh,'+k');
title('x and h grid definitions');
hold on; % plot limiter and vac vessel
plot(L.G.rl,L.G.zl,'LineWidth',3);
plot(L.G.rv,L.G.zv,'LineWidth',3); 
legend([p(1) p1(1) p2(1)],...
'x grid (filaments)','h grid midpoints','h grid boundary',...
'location','south');

% example of a (unlikely) current distribution in the smaller h-grid mapped to x-grid
Ih = zeros(nh,0); Ih([1,3,6],1) = 1e3;
Ix = Txh*Ih;
subplot(122)
pcolor(L.G.rx,L.G.zx,reshape(Ix,L.nzx,L.nrx));  axis equal tight;
cb = colorbar; colormap('hot');
cb.Label.String = 'Plasma current [A]'; hold on;
title({'Example of a (unlikely) current', 'distribution on the h grid'});
hold on; % plot limiter and vac vessel
plot(L.G.rl,L.G.zl,'LineWidth',3);
plot(L.G.rv,L.G.zv,'LineWidth',3); 


%% a) Least-squares problem for current distribution on the full grid 
% Write a least-squares problem to determine the current 
% distribution $I_x$ on the full $x$ grid from the measurements of 
% magnetic probes and flux loops, in the form $\min||AI_x - b||_2$. 
% Does this WLS problem have a unique solution?

% Enter your solution here


%% b) Least-squares problem for current distribution at reduced h-grid
% Write a least-squares problem to determine the current 
% distribution $I_h$ on the reduced $h$ grid from the measurements. 
% What are $A$ and $b$? Does this LS problem have a unique solution?

% Enter your solution here


%% c) Solution of the least-squares problem
% Write the solution of the least-squares problem in (b) in the form 
% $I_h = H \cdot b$. How can $H$ be determined from $A$? 
% Calculate and plot the current distribution $I_h$ obtained for the first 
% time slice of the noisy measurement on the $(r,z)$ grid. 
% Write also the solution to the weighted least-squares problem 
% given the measurement uncertainties.

% Enter your solution here


%% d) Write the weighted least squares solution in (c) in the form Ih = Ahy*y
% where y = [Bm; Ff; Ia] see slide 28 for graphical matrix manipulation

% Enter your solution here

%% e) Write a linear estimator for the plasma current 
% Recall that the plasma current can be determined from the current distribution 
% as $I_p=\sum_{j=1}^{n_x}I_{x,j}$. Write a linear estimator for the plasma 
% current from the measurements such that $I_p = H_{I_p} \cdot y$.

% Enter your solution here


%% f) Estimate of r and z based on the measurements
% We can define an estimator for the plasma radial and vertical position, by 
% assuming the position corresponds to the centroid of the current distribution:
% 
% $\hat{r}=\frac{\sum_{j}^{n_x}r_{x,j}I_{x,j}}{\sum_{i}^{n_x}I_{x,i}}$  and
% $\hat{z}=\frac{\sum_{j}^{n_x}z_{x,j}I_{x,j}}{\sum_{i}^{n_x}I_{x,i}}$
% 
% Use these experessions to compute an estimate of $r$ and $z$ from the
% measurement vector y. Is this a linear estimator?

% Enter your solution here.

%% g) Outputs of the linear estimators
% Plot the outputs of the linear estimators and compare to the known values 
% of $r_0I_{p0},z_0I_{p0},I_{p0}$. Improve the estimate of $z_0I_{p0}$ and 
% $I_{p0}$ by increasing the grid size of the $h$ space.
% Explain your solution. 

% Enter your solution here


%% h) (optional) Effect of nr/nz
% Plot the mean and standard deviation of $I_p$, $zI_p$, $rI_p$ estimates 
% for various values of $nr,nz$ (and $nh=nz*nr$).
% Does the estimator quality increase, decrease or stay the same when 
% increasing $nh$? Explain why.

% Enter your solution here



%% i) (optional) Effect on circuit current measurements
% Now assume that the circuit current measurements \texttt{Ia\_meas} is also 
% affected by measurement noise of standard deviation \texttt{eIa=100[A]}. 
% Check the quality of your estimator and compare it to the non-noisy case. 
% Reformulate the least squares problem to jointly estimate the plasma current distribution 
% and circuit currents from the measurements and compare the estimation results.


