classdef (SharedTestFixtures={mexm_fixture}) vizr_test < meq_test
  % Tests for surface and volume integrals
  %
  % [+MEQ MatlabEQuilibrium Toolbox+]

  %    Copyright 2022-2025 Swiss Plasma Center EPFL
  %
  %   Licensed under the Apache License, Version 2.0 (the "License");
  %   you may not use this file except in compliance with the License.
  %   You may obtain a copy of the License at
  %
  %       http://www.apache.org/licenses/LICENSE-2.0
  %
  %   Unless required by applicable law or agreed to in writing, software
  %   distributed under the License is distributed on an "AS IS" BASIS,
  %   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  %   See the License for the specific language governing permissions and
  %   limitations under the License.

  properties
    tol = 1e-4;
    data
  end

  properties (ClassSetupParameter)
    type = {'SND',...
            'DND',...
            'Lim',...
            'Lim-X',...           % limited with a non-limiting X-point
            'Boundary-X',...      % X-point close to the computational boundary
            'Double-Snowflake-Minus',...
            'Double-Snowflake-Plus',...
            'Doublet',...
            'Droplet',...
            'Doublet-div-nomantle',...
            'Doublet-div',...
            'Triplet',...
            'Triplet-madness',...
           };
    sIp = struct('positive',1,'negative',-1);
  end
  
  properties (TestParameter)
    method = {'vizrmex','vizrmexm'};
  end
  
  methods(TestClassSetup)
    function meqintTestSetup(testCase,type,sIp)
      S = testCase.generate_flux_map(type,sIp);
      
      % Need an increased grid size for Triplet-madness resolution
      L = fgs('ana',0,0,'cappav',2,'nz',32,'nr',32);
      
      Fx = S.Fxh(L.rrx,L.zzx);

      [~,~,~,~,~,~,~,~,~,~,~,~,...
        ~,~,~,~,~,Opy] = meqpdom(Fx,sIp,L.P.isaddl,L);

      L.nD = double(max(Opy(:)));

      rBt = L.P.r0*L.P.b0;
      
      testCase.data.type = type;
      testCase.data.S    = S;
      testCase.data.Fx   = Fx;
      testCase.data.rBt  = rBt;
      testCase.data.Opy  = Opy;
      testCase.data.L    = L;
    end
  end
  
  methods (Test,TestTags={'Unit'})
    function test_vizr(testCase,method)
      % Verify that mex and mexm versions give correct answers

      L = testCase.data.L;
      
      Fx  = testCase.data.Fx;
      rBt = testCase.data.rBt;
      Opy = testCase.data.Opy;

      % Expected values
      % Compute integrand of Wp
      dFdr = (Fx(2:end-1,3:end) - Fx(2:end-1,1:end-2))/2/L.drx;
      dFdz = (Fx(3:end,2:end-1) - Fx(1:end-2,2:end-1))/2/L.dzx;
      gradPsi2y = dFdr.^2+dFdz.^2;
      gradPsioR = (gradPsi2y.*L.iry'); % |GradPsi|^2/R
      Vp_  = 2*pi*(sum(Opy>0)*L.ry )*L.dsx;
      Ft0_ =  rBt*(sum(Opy>0)*L.iry)*L.dsx;
      Wp_  = 1/(4*pi*4e-7*pi)*sum(gradPsioR(Opy>0))*L.dsx;
      
      % test vs mex/mexm
      [Wp,Ft0,Vp] = feval(method,Fx,Opy,L.ry,L.iry,rBt,L.drx,L.dzx);
      
      % verify
      testCase.verifyEqual(Vp ,Vp_ ,'RelTol',testCase.tol)
      testCase.verifyEqual(Ft0,Ft0_,'RelTol',testCase.tol)
      testCase.verifyEqual(Wp ,Wp_ ,'RelTol',testCase.tol)
      
    end
  end
end
