classdef visualization_tests < meq_test
  % Tests for MEQ visualizations
  %
  % [+MEQ MatlabEQuilibrium Toolbox+]

  %    Copyright 2022-2025 Swiss Plasma Center EPFL
  %
  %   Licensed under the Apache License, Version 2.0 (the "License");
  %   you may not use this file except in compliance with the License.
  %   You may obtain a copy of the License at
  %
  %       http://www.apache.org/licenses/LICENSE-2.0
  %
  %   Unless required by applicable law or agreed to in writing, software
  %   distributed under the License is distributed on an "AS IS" BASIS,
  %   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  %   See the License for the specific language governing permissions and
  %   limitations under the License.
  
  properties
    tok = 'ana';
    verbosity = 0; % if 0, figures are hidden entirely
    L,LX,LY,LY2;
    hf; % figure handle
  end
  
  properties(ClassSetupParameter)
    shot = {101,82,91}
    circuit = {true,false,false}
  end
  
  properties(TestParameter)
    moviestyle = {'plain','fancy','full','custom'}; % movie plotting styles
    multipleLY = {false,true}
    plotFz = {true,false}; % plot extended contour in fancy plot or not
    vacuum = {true,false};
    plotArrows = {true,false};
  end
  
  methods (TestClassSetup,ParameterCombination='sequential')
    
    function run_code(testCase,shot,circuit)
      s = rng(1);
      testCase.addTeardown(@() rng(s));
      
      [L,LX,LY] = fbt(testCase.tok,shot,[],'circuit',circuit,...
        'selu','e','nu',30,'izgrid',true,'ivacuum',true,'ifield',true,'iterq',50);
      testCase.L = L; testCase.LX = LX; testCase.LY = LY; %#ok<*PROP>
      
      % rerun a different case
      LX2 = testCase.LX; 
      if L.nD>1, LX2.bpD = 2*LX2.bpD;
      else,      LX2.bp  = 2*LX2.bp;
      end
      testCase.LY2 = fbtt(testCase.L,LX2);
    end
    
    function setup_fig(testCase)
      if testCase.verbosity==0
        Visible = 'off';
      else
        Visible = 'on';
      end
      testCase.hf = figure('name',sprintf('%s figure',mfilename),...
        'Visible',Visible);
      testCase.addTeardown(@() delete(testCase.hf) );
    end
  end
  
  methods (Test,TestTags={'Integration'})
    
    function test_meqplott(testCase)
      % tests of meqplott
      
      % basic test
      hax = meqplott(testCase.L,meqxk(testCase.LY,1));
      testCase.assertInstanceOf(hax,'handle');
      
      % test with title function
      titlefun = @(L,LY) sprintf('t=%+2.0fms',LY.t*1e3);
      hax = meqplott(testCase.L,meqxk(testCase.LY,1),'titlefun',titlefun,'parent',gca);
      testCase.assertInstanceOf(hax,'handle');
    end
    
    function test_meqplots(testCase,plotFz,vacuum,plotArrows)
      % test various ways of running meqplotfancy();
      L = testCase.L; LY = testCase.LY; %#ok<*PROPLC>
         
      % normal fancy plot
      testCase.clf;
      meqplotfancy(L,meqxk(LY,1),...
        'plotFz',plotFz,'vacuum',vacuum,'plotArrows',plotArrows);
      testCase.drawnow;

      % fancy plot without plasma
      testCase.clf;
      meqplotfancy(L);
      testCase.drawnow;
     
      testCase.clf
      meqplotQ(L,meqxk(LY,1));
      testCase.drawnow;

      if L.nD>1
        % multidomain special
        LYt = meqlpack(repmat(LY,1,3)); LYt.t = LY.t+[0 1 2];
        meqplotD(L,LYt);
      end
      
      if numel(LY.t)>1
        testCase.clf
        meqplotevo(L,LY);
        testCase.drawnow;
      end
    end

    function test_meqcompare(testCase)
      L = testCase.L; LX = testCase.LX; LY = testCase.LY; %#ok<*PROPLC>
      nt = numel(LY.t);

      % plot two equilibria with the same L
      testCase.clf;
      meqcompare(L,meqxk(LY,1),meqxk(LY,nt));
      testCase.drawnow;

      % check that passing the same L twice also works
      testCase.clf;
      meqcompare(L,meqxk(LY,1),L,meqxk(LY,nt));
      testCase.drawnow

      % check plotting with different Ls
      P2 = L.P; P2.npq = 2*P2.npq; P2.pq = []; L2 = fbtc(P2,L.G);
      LY2 = fbtt(L2,LX);
      testCase.clf;
      meqcompare(L,meqxk(LY,1),L2,meqxk(LY2,nt));
      testCase.drawnow;

      % check incompatible L throws expected error
      testCase.assertError(@() meqcompare(L,meqxk(LY,1),L,meqxk(LY2,nt)),...
        'MEQCOMPARE:IncompatibleLandLY')
      
    end

    function test_fgeplot(testCase)
      [L,~,LY] = fge('ana',1,0,'lin',1);
      testCase.clf; fgeploteig(L);
      testCase.drawnow;
      testCase.clf; fgeploteig(L,'LX',LY);
      testCase.drawnow;
    end
    
    function test_meqplotliu(testCase)
      testCase.clf;
      [L,LX,LY] = liu('ana',1,0);
      ax = meqplotliu(L,LX,LY,'parent',testCase.hf);
      testCase.assertInstanceOf(ax,'matlab.graphics.axis.Axes');
    end

    function test_fbtplot(testCase)
      testCase.clf;
      hax=fbtplot(testCase.L,meqxk(testCase.LX,1),meqxk(testCase.LY,1));
      testCase.assertTrue(isa(hax,'handle'));
    end
    
    function test_meqplotcontour(testCase)
      hax = meqplotcontours(testCase.L,meqxk(testCase.LY,1));
      testCase.assertTrue(isa(hax,'handle'));
    end

    function test_meqplots_vacuum(testCase,vacuum)
      L = testCase.L; LY = meqxk(testCase.LY,1); %#ok<*PROPLC>

      % plot field, alternating with/without vacuum case
      testCase.clf;
      meqplotfield(L,LY,'FxLevels',-1:0.01:1,...
        'plotArrows',true,'plotBp',true,'BpLevels',0:0.02:0.2,'vacuum',vacuum);
      testCase.drawnow;
    end
    
    function test_meqmovie(testCase,moviestyle,multipleLY)
      % test default movie generation
      fname = 'test-movie';

      testCase.clf;
      if strcmp(moviestyle,'custom')
        PP = {'plotfun',@custom_plotfun,...; % pass plot function handle
          'videoprofile','gif','dosave',true}; % test gif generation in this case
      else
        PP = {'videoprofile','Motion JPEG AVI',...
          'dosave',isequal(moviestyle,'fancy')}; % default: save only fancy one
      end

      if ~multipleLY
        % single LY
        LY_in = testCase.LY;
      else
        % multiple LYs
        testCase.assumeNotEqual(moviestyle,'full','skip multi-LY case for moviestyle=full')
        LY_in = {testCase.LY,testCase.LY};
      end

      testCase.clf;
      fname_out = meqmovie(testCase.L,LY_in,...
        'decimate',5,'style',moviestyle,'filename',fname,PP{:});
      if ~isempty(fname_out), delete(fname_out); end   
      
      function hax = custom_plotfun(hax,L,LY)
        % minimalist custom plotting function
        if LY.nB == 0; FBB = 0; else, FBB = LY.F1(LY.nB); end
        contour(L.rx,L.zx,LY.Fx,FBB*[1 1],'k-'); axis(hax,'equal');
      end
    end

    function test_meqplotseq_default(testCase)
      % Test default case
      testCase.clf;
      h = meqplotseq(testCase.L,testCase.LY,'parent',gcf);
      testCase.assertInstanceOf(h,'handle');
      testCase.drawnow;
    end
    
    function test_meqplotseq_fbtplot(testCase)
      %% Test FBT plotting case
      testCase.clf;
      plotfun = @(ax,L,LX,LY) fbtplot(L,LX,LY,'parent',ax,'legend',false);
      meqplotseq(testCase.L,testCase.LY,...
        'LX',testCase.LX,... % pass additional LX since fbtplot needs it
        'parent',gcf,...
        'plotfun',plotfun)
      testCase.drawnow;
    end

    function test_meqplotseq_custom(testCase)
      % Test with custom plotting and title string and column/row sep
      testCase.clf;
      h = meqplotseq(testCase.L,testCase.LY,'parent',gcf,...
        'plotfun',@custom_plot,...
        'row_spacing',0.1,'row_offset',0.05);
      testCase.assertInstanceOf(h,'handle');
      testCase.drawnow;

      % subfunction
      function custom_plot(ax,L,LY)
        if LY.nB == 0, FBB = 0; else, FBB = LY.F1(LY.nB); end
        % custom plotting function used for meqplotseq
        contour(ax,L.rx,L.zx,LY.Fx-FBB,[0 0],'k'  ,'linewidth',2); hold(ax,'on');
        contour(ax,L.rx,L.zx,LY.Fx-FBB, 11  ,'k--','linewidth',1);
        plot(ax,LY.rA,LY.zA,'ok');
        axis(ax,'equal'); set(ax,'XTick',[],'YTick',[]);
        title(ax,sprintf('ANA#%d t=%3.2f',LY.shot,LY.t));
      end
    end
    
    function test_meqplotseq_multi(testCase)
      %% Multi-step plot
      % Test mode where we only generate axes in one step, 
      % then plot two diffferent runs under each other in the second step
      testCase.clf; set(testCase.hf,'position',[0 0 1500 300]); % long one
      nt = numel(testCase.LY.t); 
      hax = meqplotseq(testCase.L,[],'parent',gcf,... % empty LY means we only generate axes
        'row_spacing',0.01,'row_offset',0.05,...
        'col_spacing',0.01,...
        'nt',2*nt); % need to pass total number of axes to be generated (2x number of time steps)
    
      % plot two time sequences under each other
      h1 = meqplotseq(testCase.L,testCase.LY ,'parent',hax(1:nt    )  ,'plotfun',@plotfun); % previous one
      h2 = meqplotseq(testCase.L,testCase.LY2,'parent',hax((nt+1):end),'plotfun',@plotfun); % new one
      testCase.assertInstanceOf(h1,'handle');
      testCase.assertInstanceOf(h2,'handle');
      testCase.drawnow;
      
      function ax = plotfun(ax,L,LY)
        if LY.nB == 0, FBB = 0; else, FBB = LY.F1(LY.nB); end
        contour(ax,L.rx,L.zx,LY.Fx-FBB,21); title(ax,sprintf('bp=%2.2f',LY.bp)); axis(ax,'equal');
      end
    end
    
    function test_meqplotseq_overlay(testCase)
      %% Overlay plots
      % test running with two LY structures
      LYs = {testCase.LY,testCase.LY2}; % multiple LYs to plot on one timeslice
      testCase.clf;
      meqplotseq(testCase.L,LYs ,'plotfun',@plotfun_multi);
      
      function ax = plotfun_multi(ax,L,LY)
        if LY{1}.nB == 0, FB1 = 0; else, FB1 = LY{1}.F1(LY{1}.nB); end
        if LY{2}.nB == 0, FB2 = 0; else, FB2 = LY{1}.F1(LY{2}.nB); end
        contour(ax,L.rx,L.zx,LY{1}.Fx-FB1,21,'k');
        title(ax,sprintf('t=%2.2f',LY{1}.t)); axis(ax,'equal'); hold(ax,'on');
        contour(ax,L.rx,L.zx,LY{2}.Fx-FB2,21,'r--');
      end
    end
  end
  
  methods % non-Test methods
    function clf(testCase)
      % auxiliary function to make figure current and clear it
      clf(testCase.hf);
      set(0,'CurrentFigure',testCase.hf)
    end
      
    function drawnow(testCase)
      % auxiliary function to draw only if verbosity is on
      if testCase.verbosity
        drawnow(); 
      end
    end
  end
end
