classdef vessel_tests < meq_test
  % Test for vessel representations and calculations
  %
  % [+MEQ MatlabEQuilibrium Toolbox+]

  %    Copyright 2022-2025 Swiss Plasma Center EPFL
  %
  %   Licensed under the Apache License, Version 2.0 (the "License");
  %   you may not use this file except in compliance with the License.
  %   You may obtain a copy of the License at
  %
  %       http://www.apache.org/licenses/LICENSE-2.0
  %
  %   Unless required by applicable law or agreed to in writing, software
  %   distributed under the License is distributed on an "AS IS" BASIS,
  %   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  %   See the License for the specific language governing permissions and
  %   limitations under the License.

  properties(TestParameter)
    ne = {256,30}
    selu = {'v','s','e','n'};
    tok  = {'tcv','ana'};
    code = {'liu','fbt','fgs','fge'}
  end
  
  properties
    L,LX
    solvetol = 1e-10;
    shot = 61400;
    insrc = 'liu';
    t = 0.5;
    nz = 16;
    nr = 16;
    verbosity = 0;
    excludedFields = {'Iu','res','niter','mkryl','nfeval','rese','resy','resC','resFx'}; % fields to exclude in checks
  end
  
  methods(TestClassSetup)
    function define_base_case(testCase)
      if meq_test.check_tok('tcv')
        % Use same vessel resistivity as reference run
        selu_ = 'v';
        if strcmp(testCase.insrc,'liu'), selu_ = [selu_,'x']; end
        [testCase.L,testCase.LX] = ...
          fgs('tcv',testCase.shot,testCase.t,...
          'insrc',testCase.insrc,...
          'tolF',testCase.solvetol,...
          'selu',selu_,...
          'debug',testCase.verbosity);
      end
    end
  end
    
  methods (Test,TestTags={'fgs'})    
    function test_eigenmodes(testCase,ne)
      %% Project on smaller eigenmode space and solve
      [ok,msg] = meq_test.check_tok('tcv');
      testCase.assumeTrue(ok,msg);

      % Use same vessel resistivity as reference run
      selu_ = 'e';
      if strcmp(testCase.insrc,'liu'), selu_ = [selu_,'x']; end

      [Le,LXe] = ...
        fgs('tcv',testCase.shot,testCase.t,...
        'insrc',testCase.insrc,...
        'tolF',testCase.solvetol,...
        'selu',selu_,'nu',ne,'debug',testCase.verbosity);
      
      Iu2 = Le.G.Tvu*(Le.G.Tvu\testCase.LX.Iu); % project to eigenmode currents
      
      if testCase.verbosity>1
        plot([testCase.LX.Iu,Iu2]); drawnow
      end
      
      testCase.LX.Iu = Iu2;
      LY  = fgst(testCase.L,testCase.LX); % rerun full case with projected current
      LYe = fgst(Le,LXe); % solve directly with less eigenmodes

      testCase.verifyEqual(size(LYe.Iu,1),ne,'size of Iu does not match expectation')
      % compare excluding Iu (which changed) and res (which is very small)
      for ifield = fieldnames(LY)'
        field=ifield{:};
        switch field
          case testCase.excludedFields,  continue;
          otherwise
            if isempty(LY.(field))
              testCase.verifyEmpty(LYe.(field));
            else
              testCase.verifyEqual(LY.(field),LYe.(field),'AbsTol',sqrt(testCase.solvetol)*max(abs(LY.(field)(:))),...
                sprintf('field %s is not equal',field));
            end
        end
      end
    end
    
    function test_segments(testCase)
      % solve FGS using segment currents directly or projecting segment
      % currents onto vessel currents. Should give the same result.

      [ok,msg] = meq_test.check_tok('tcv');
      testCase.assumeTrue(ok,msg);

      % Use same vessel resistivity as reference run
      selu_ = 's';
      if strcmp(testCase.insrc,'liu'), selu_ = [selu_,'x']; end
      
      [Ls,LXs] = ...
        fgs('tcv',testCase.shot,testCase.t,...
        'insrc',testCase.insrc,...
        'tolF',testCase.solvetol,...
        'selu',selu_,'debug',testCase.verbosity);
      LYs = fgst(Ls,LXs);      
      
      % project to segment currents
      Iv = Ls.G.Tvu*(Ls.G.Tvu\testCase.LX.Iu); % project to segment currents
      
      if testCase.verbosity>1
        plot([testCase.LX.Iu,Iv]); drawnow
      end
      
      LXm = testCase.LX; LXm.Iu = Iv; % modify Iv
      LY  = fgst(testCase.L,LXm); % run full case
      
      for ifield = fieldnames(LY)'
        field=ifield{:};
        switch field
          case testCase.excludedFields,  continue;
          otherwise
            if isempty(LY.(field))
              testCase.verifyEmpty(LYs.(field));
            else
              testCase.verifyEqual(LY.(field),LYs.(field),'AbsTol',sqrt(testCase.solvetol)*max(abs(LY.(field)(:))),...
                sprintf('failed check for %s',field));
            end
        end
      end
    end
  end
    
  methods(Test,TestTags={'Integration'})
    function test_inits(testCase,code,tok,selu)
      % test initialization for all codes for all vessel types for TCV and
      % anamak      
      switch tok
        case 'tcv', shot = 61400; %#ok<*PROPLC>
        case 'ana', shot = 0;
      end
      [ok,msg] = meq_test.check_tok(tok);
      testCase.assumeTrue(ok,msg);

      fcode = str2func(code);
      try
        L = fcode(tok,shot,[],'selu',selu);
        testCase.verifyEqual(L.P.selu,selu);
      catch ME
        % expected error when running FGE with selu='s'
        if isequal(ME.identifier,'FGE:INVALIDVESSELMODE')
          return
        else
          rethrow(ME)
        end
      end
    end
  end
end
