classdef tools_test < meq_test
  % tests for generic meq tools
  %
  % [+MEQ MatlabEQuilibrium Toolbox+]

  %    Copyright 2022-2025 Swiss Plasma Center EPFL
  %
  %   Licensed under the Apache License, Version 2.0 (the "License");
  %   you may not use this file except in compliance with the License.
  %   You may obtain a copy of the License at
  %
  %       http://www.apache.org/licenses/LICENSE-2.0
  %
  %   Unless required by applicable law or agreed to in writing, software
  %   distributed under the License is distributed on an "AS IS" BASIS,
  %   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  %   See the License for the specific language governing permissions and
  %   limitations under the License.

  properties (TestParameter)
    tgrid = {linspace(0,1,11),linspace(0,1,11).^2}
    shot = {1,82}; % test a singlet and a doublet
  end

  methods(Test, TestTags = {'Unit'})
    
    function test_slicers(testCase)
      % test data slicer/insertion functions
      % get data to user
      [~,~,LY] = fge('ana',1,0:1e-4:1e-3);
      
      % test meqxk (slice extractor) - compare using indices or logical mask
      kk = false(size(LY.t));
      k=kk; k(1)=true;   LY1=meqxk(LY,k);
      k=kk; k(end)=true; LYn=meqxk(LY,k);
      k=kk; k(2:3)=true; LYi=meqxk(LY,k);
      
      testCase.verifyEqual(LY1,meqxk(LY,1),'should match')
      testCase.verifyEqual(LYn,meqxk(LY,'last'),'should match')
      testCase.verifyEqual(LYi,meqxk(LY,2:3),'should match')
      
      % test meqsk (time slice substitutor)
      testCase.verifyEqual(LY,meqsk(LY,true(size(kk)),LY),...
        'should be identity operation')
      testCase.verifyEqual(meqxk(LY,2),  meqxk(meqsk(LY,1  ,meqxk(LY,2)),  1  ),...
        'should have replaced first index');
      testCase.verifyEqual(meqxk(LY,2:3),meqxk(meqsk(LY,1:2,meqxk(LY,2:3)),1:2),...
        'should have replaced first two indices');
      
      %% test meqik (time slice inserter)
      nt = numel(LY.t);
      % insert third time slice at beginning
      LY_expected = meqxk(LY,[3,1:nt]);
      testCase.verifyEqual(LY_expected,meqik(LY,1,meqxk(LY,3)),'Should insert first slice at beginning')
      
      % insert slice 1 after index 3
      s = 1; k = 3;
      LY_expected = meqxk(LY,[1:k-1 s k:nt]);
      testCase.verifyEqual(LY_expected,meqik(LY,k,meqxk(LY,s)),'Unexpected resulting structure')
      
      % insert slice 3 at the end
      s = 3; k = nt+1;
      LY_expected = meqxk(LY,[1:nt,s]);
      testCase.verifyEqual(LY_expected,meqik(LY,k,meqxk(LY,s)),'Unexpected resulting structure')
      
      % check error
      LYrm = rmfield(LY,'IpD');
      testCase.verifyError(@() meqik(LY,1,meqxk(LYrm,1)),'meqik:NotSameFields')
    end

    function test_meqagconfun(testCase)
      % Test meqagconfun call successfully returns a structure
      S = meqagconfun;

      testCase.verifyTrue(isstruct(S),sprintf('meqagconfun should return a structure but returned a %s object',class(S)));
    end

    function test_meqagconfun_help(testCase)
      % Test meqagconfun('help') call returns
      S = evalc('meqagconfun(''help'');');

      testCase.verifyNotEmpty(S,'meqagconfun(''help'') did not print any text');
    end
    
    function test_meqcdefun_help(testCase)
      % Test meqagconfun('help') call returns
      S = evalc('meqcdefun(''help'');');

      testCase.verifyNotEmpty(S,'meqagconfun(''help'') did not print any text');
    end
    
    function test_meqagconc(testCase)
      % sample inputs
      in1={'Ip','bp','qA'}; % simple
      in2={{'Ip','bp','qA'},{'Ip','bp','qA'},{'ag','ag','ag'}}; % complex
      in3={'Ip','bp','FA'}; % does not exist
      in4={'Ip',1;'bp',1;'qA',1}; % type 3 input
      
      % nondoublet
      ng = 3;
      idoublet = 0;
      TDg = ones(1,ng);
      % Working
      testCase.verifyWarningFree(@() meqagconc(in1,TDg,idoublet))
      testCase.verifyWarningFree(@() meqagconc(in4,TDg,idoublet))
      % error
      testCase.verifyError(@() meqagconc(in3,TDg,idoublet),'meqagconc:notfound')
      
      % doublet
      ng = 9; nD = 3;
      idoublet = 1;
      TDg = zeros(nD,ng);
      TDg(1,1:3) = 1; TDg(2,4:6) = 1; TDg(3,7:9) = 1;
      testCase.verifyWarningFree(@() meqagconc(in1,TDg,idoublet))
      testCase.verifyWarningFree(@() meqagconc(in2,TDg,idoublet))
    end

    function test_meqxconvert(testCase,shot)
      %% Test meqxconvert accuracy and iterq
      [L,~,LX]  = fbt('ana',shot,0,'bfct',@bf3pmex,'bfp',[]   ,'nr',32,'nz',16,'iterq',0);
      [L2]      = fbt('ana',shot,0,'bfct',@bfabmex,'bfp',[1 2],'nr',16,'nz',8, 'iterq',1);
      
      LX2 = meqxconvert(L,LX,L2);
      
      equalfields = {'PQ','PpQ','TQ','TTpQ','PpQg','TTpQg','Iv','Ia','Iu','Ip'};
      for ii=equalfields
        testCase.verifyEqual(LX.(ii{:}),LX2.(ii{:}),'AbsTol',sqrt(eps),sprintf('Error in field %s',ii{:}))
      end
      
      % Sum of current Iy should be equal to LX.Ip
      testCase.verifyEqual(sum(LX2.Iy(:)),LX.Ip,'RelTol',sqrt(eps),'Sum of Iy is not equal after conversion')

      % Fields should have correct size
      for field = fieldnames(LX2).'
        switch field{:}
          case {'Sp','Ep','chi','chie','Fb','ID','uerr','werr'}
            % Skipped FBT-specific fields
            continue
          case {'aq','aW'}
            % Skipped fields
            continue
          case {'FA','rA','zA','dr2FA','dz2FA','drzFA','qA'}
            expected_size = [double(LX.nA),1];
          case {'FX','rX','zX','dr2FX','dz2FX','drzFX'}
            expected_size = [double(LX.nX),1];
          otherwise
            expected_size = meqsize(L2,field{:});
        end
        testCase.verifyEqual(size(LX2.(field{:})),expected_size,sprintf('Size of %s field in output of meqxconvert does not match target L definition',field{:}));
      end
    end
    
    function test_meqdiff(testCase)
      % simple test of meqdiff for a case with multple x points
      [~,~,LYd]  = fbt('ana',82,[]); % doublet
      [~,~,LYs]  = fbt('ana',3,[]);  % singlet
      meqdiff(LYd,LYd); % test same doublet
      meqdiff(LYs,LYs); % test same singlet
      meqdiff(LYd,LYs); % test different
    end

    function test_n2k(testCase)
      % Test n2k tool
      n1 = 4; n2 = 6; n3 = 2;
      [kN1,kN2,kN3,nN] = n2k(n1,n2,n3);

      % Verify answers
      testCase.verifyEqual(kN1,       1:n1 ,'First  output of n2k does not match expected value');
      testCase.verifyEqual(kN2,n1+   (1:n2),'Second output of n2k does not match expected value');
      testCase.verifyEqual(kN3,n1+n2+(1:n3),'Third  output of n2k does not match expected value');
      testCase.verifyEqual( nN,    n1+n2+n3,'Last   output of n2k does not match expected value');
    end

    function test_boundingbox(testCase)
      % test regular case
      L = fbt('ana',0);
      rh_ = L.P.r0; zh_ = 0;
      wh_ = 0.2; hh_ = 0.3;
      rH = rh_ + wh_*[-1,0,1,0]; zH = zh_ + hh_*[0,1,0,-1];
      idoublet = false;
      [bb] = meqbbox(rH,zH,L.G.rl,L.G.zl,idoublet);
      rh = (bb(2)+bb(1))/2; zh = (bb(3)+bb(4))/2;
      wh = (bb(2)-bb(1))/2; hh = (bb(4)-bb(3))/2;
      testCase.verifyEqual([rh,zh,wh,hh],[rh_,zh_,wh_,hh_],'abstol',10*eps());

      % test vertically collapsed case
      wh_ = 0; hh_ = 0.3; % zero width 
      rH = rh_ + wh_*[-1,0,1,0]; zH = zh_ + hh_*[0,1,0,-1];
      [bb] = meqbbox(rH,zH,L.G.rl,L.G.zl,idoublet);
      rh = (bb(2)+bb(1))/2; zh = (bb(3)+bb(4))/2;
      wh = (bb(2)-bb(1))/2; hh = (bb(4)-bb(3))/2;
      % expect wh = hh
      testCase.verifyEqual([rh,zh,wh,hh],[rh_,zh_,hh_,hh_],'abstol',10*eps());
      
      % test horizontally collapsed case
      wh_ = 0.2; hh_ = 0; % zero height 
      rH = rh_ + wh_*[-1,0,1,0]; zH = zh_ + hh_*[0,1,0,-1];
      [bb] = meqbbox(rH,zH,L.G.rl,L.G.zl,idoublet);
      rh = (bb(2)+bb(1))/2; zh = (bb(3)+bb(4))/2;
      wh = (bb(2)-bb(1))/2; hh = (bb(4)-bb(3))/2;
      % expect hh = wh
      testCase.verifyEqual([rh,zh,wh,hh],[rh_,zh_,wh_,wh_],'abstol',10*eps())

      % test doublet case
      wh_ = 0.2; hh_ = 0.2;
      rH = rh_ + wh_*[-1,0,1,0,-1,0,1,0]; 
      zH = [zh_ + 1.1*wh_ + hh_*[0,1,0,-1], zh_ - 1.1*wh_ + hh_*[0,1,0,-1]];
      
      idoublet = true;
      [bb] = meqbbox(rH,zH,L.G.rl,L.G.zl,idoublet);
      testCase.assertEqual(size(bb,1),2);
      rh = (bb(:,2)+bb(:,1))/2; zh = (bb(:,3)+bb(:,4))/2;
      wh = (bb(:,2)-bb(:,1))/2; hh = (bb(:,4)-bb(:,3))/2;
      expected = [[rh_, zh_ + 1.1*wh_, wh_, hh_ ];[rh_, zh_ - 1.1*wh_, wh_, hh_]];
      testCase.verifyEqual([rh,zh,wh,hh],expected,'abstol',10*eps())

    end

    function test_meqlpack(testCase)
      % Some stress tests for meqlpack
      
      % singlets
      [L0,~,LY0] = fbt('ana',91,0,'iterq',10); % vacuum
      [L1,~,LY1] = fbt('ana', 1,0,'iterq',10);
      [L2,~,LY2] = fbt('ana', 2,0,'iterq',10);
      LY = meqlpack([LY0,LY1,LY2]); % non-empty aq for all
      testCase.verifySize(LY.aq,[L0.noq,L0.npq,max([L0.nD,L1.nD,L2.nD]),numel(LY.t)])

      % droplets
      [L2,~,LY2] = fbt('ana',81,0,'iterq',10);
      LY = meqlpack([LY0,LY1,LY2]);
      testCase.verifySize(LY.aq,[L0.noq,L0.npq,max([L0.nD,L1.nD,L2.nD]),numel(LY.t)])

      % empty slices
      [~,~,LY1] = fbt('ana', 1,0,'iterq',0); % Empty aq
      LY3(1) = LY1;
      % Skip LY3(2) - all fields will have empty values
      LY3(3) = LY1;
      LY = meqlpack(LY3);
      testCase.verifySize(LY.aq,[0,0,1,numel(LY.t)])
    end
  end
end
