classdef test_timing_performance < meq_test
  % Tests of timing performance
  %
  % [+MEQ MatlabEQuilibrium Toolbox+]

  %    Copyright 2022-2025 Swiss Plasma Center EPFL
  %
  %   Licensed under the Apache License, Version 2.0 (the "License");
  %   you may not use this file except in compliance with the License.
  %   You may obtain a copy of the License at
  %
  %       http://www.apache.org/licenses/LICENSE-2.0
  %
  %   Unless required by applicable law or agreed to in writing, software
  %   distributed under the License is distributed on an "AS IS" BASIS,
  %   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  %   See the License for the specific language governing permissions and
  %   limitations under the License.

  methods (Test,TestTags={'Integration'})
    
    function test_anamak_timing(testCase)
      %% Test timing performance for anamak fge case
      time = 0:1e-3:0.1;
      tok = 'ana';
      
      tL = tic;
      [L] = fge(tok,1,time,'cde','cde_ss_0D','agcon',{'bp','qA'},...
        'usepreconditioner',true);
      tLe = toc(tL);
      
      tX = tic;
      LX = fgex(tok,time,L);
      tLXe = toc(tX);
      
      tl = tic;
      L = fgel(L,meqxk(LX,1));
      tLle = toc(tl);
      
      tY = tic;
      LY = fget(L,LX);
      tLYe = toc(tY);
      
      tTot = toc(tL);
      
      fprintf('time for L  :%2.2fs\n',tLe);
      fprintf('time for LX :%2.2fs\n',tLXe);
      fprintf('time for lin:%2.2fs\n',tLle);
      fprintf('time for LY :%2.2fs\n',tLYe);
      fprintf('Total:      :%2.2fs\n',tTot);

      testCase.assertTrue(meq_test.check_convergence(L,LX.t,LY),'FGE did not converge');
    end

    function test_fgeF_perf(testCase)
      % Estimate time to do one fgeF function/Jacobian evaluation

      % Run perftest
      suite = matlab.unittest.TestSuite.fromFile('../perf/fgeF_perf.m');
      experiment = matlab.perftest.TimeExperiment.limitingSamplingError('NumWarmups',0);
      res = run(experiment,suite);

      % Assemble table
      resTable = res.sampleSummary;

      % Parse name
      A = regexp({res.Name},'fgeF_perf\[grid=(\w)+,shot=(\w+),usecs=(\w+)\]/perf_fgeF_eval\(jacx=(\w+)\)','tokens');
      A = [A{:}];
      A = [A{:}];

      % Grid/shot/icsint/jacx for each run
      grid = A(1:4:end);
      shot = A(2:4:end);
      conf = A(3:4:end);
      jacx = A(4:4:end);

      % Number of unique configurations
      ngrid  = numel(unique(grid));
      nshots = numel(unique(shot));
      nconf  = numel(unique(conf));

      % Create tables
      vnames = unique(grid,'stable');
      rnames = strcat('shot=',shot(1:2:nconf*nshots*2),',usecs=',conf(1:2:nconf*nshots*2));
      yF = array2table(reshape(resTable.Mean(1:2:end),[nconf*nshots,ngrid]),'VariableNames',vnames,'RowNames',rnames);
      yJ = array2table(reshape(resTable.Mean(2:2:end),[nconf*nshots,ngrid]),'VariableNames',vnames,'RowNames',rnames);

      % Display Results
      format('shorte');
      obj = onCleanup(@() format('default'));
      fprintf('## Summary of fgeF function evaluation\n');
      disp(yF);
      fprintf('## Summary of fgeF jacobian evaluation\n');
      disp(yJ);

    end
  end
end
