classdef (SharedTestFixtures={meq_fixture,meqscripts_fixture}) ...
    test_generate_doubletTCV < meq_test
  % tests for generate_doubletTCV script
  %
  % [+MEQ MatlabEQuilibrium Toolbox+]

  %    Copyright 2022-2025 Swiss Plasma Center EPFL
  %
  %   Licensed under the Apache License, Version 2.0 (the "License");
  %   you may not use this file except in compliance with the License.
  %   You may obtain a copy of the License at
  %
  %       http://www.apache.org/licenses/LICENSE-2.0
  %
  %   Unless required by applicable law or agreed to in writing, software
  %   distributed under the License is distributed on an "AS IS" BASIS,
  %   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  %   See the License for the specific language governing permissions and
  %   limitations under the License.

  properties
    verbosity = 0;
    doplot = 0;
    common_params = {'debug',0};
  end
  
  methods (TestClassSetup)
    function checkTCV(testCase)
      % Skip tests if connection to TCV MDSplus server is unavailable
      [ok,msg] = meq_test.check_tok('tcv');
      testCase.assumeTrue(ok,msg);
    end
  end
  
  methods (Test, TestTags = {'fge'})
    function default_doublet_test(testCase)
      % just normal doublet with defaults
      params = {};
      [~ ,~, L, LX0] = generate_doubletTCV(params{:},testCase.common_params{:});

      t = 0:1e-3:5e-3;
      LX = fgex('tcv', t, L, LX0);
      LY = fget(L, LX, 'debug', testCase.verbosity, 'debugplot', testCase.doplot);
      testCase.assertTrue(testCase.check_convergence(L,[],LY),...
        'default doublet evolution case did not converge');
    end
    
    function doublet_mantle_current_test(testCase)
      % also no port protection tiles and fine grid
      params = {'include_pp_tiles', 0, 'constrain_zero_mantle', 0, ...
                'Ip', 2e+5, 'Ip_mantle', 1e+3, 'use_SQP', 0, 'use_fine_grid', 0};
      [~, ~, L, LX0] = generate_doubletTCV(params{:},testCase.common_params{:});

      t = 0:1e-3:5e-3;
      LX = fgex('tcv', t, L, LX0);
      LY = fget(L, LX, 'debug', testCase.verbosity, 'debugplot', testCase.doplot);
      testCase.assertTrue(testCase.check_convergence(L,[],LY),...
        'doublet evolution with mantle current did not converge');
    end
    
    function doublet_CDE_test(testCase)
      params = {'use_cde', 1};
      [~, ~, L, LX0] = generate_doubletTCV(params{:},testCase.common_params{:});

      t = 0:5e-4:15e-3;
      LX = fgex('tcv', t, L, LX0);

      % decrease Ip to force lobes apart
      ramp_rate = -300;
      Iaind = fgeIaind(L,LX0);
      Ia_dir = -ramp_rate * Iaind;
      Va_dir = L.G.Maa * Ia_dir;

      LX.Vadot0 = LX.Vadot0 + repmat(Va_dir, 1, numel(t));
      LX.Va = LX.Va(:, 1) + LX.Vadot0(:, 1)*(LX.t-LX.t(1));
      LX.Iadot0 = LX.Iadot0 + repmat(Ia_dir, 1, numel(t));
      LX.Ia = LX.Ia(:, 1) + LX.Iadot0(:, 1)*(LX.t-LX.t(1));
    
      % have to allow alarms as this evolution directly triggers one
      LY = fget(L, LX, 'debug', testCase.verbosity, 'debugplot', testCase.doplot);
      testCase.assertTrue(testCase.check_convergence(L,[],LY),...
        'doublet evolution with CDE case did not converge');
    end
    
    function droplet_test(testCase)
      % if doublets are generated to close to the wall, droplets will
      % result
      params = {'center_r', 0.85, 'size', 0.35, 'use_SQP', 0};
      [~, ~, L, LX0] = generate_doubletTCV(params{:},testCase.common_params{:});

      t = 0:1e-3:5e-3;
      LX = fgex('tcv', t, L, LX0);
      LY = fget(L, LX, 'debug', testCase.verbosity, 'debugplot', testCase.doplot);
      testCase.assertTrue(testCase.check_convergence(L,[],LY),...
        'droplet_test case did not converge');
    end
    
    function droplet_merging(testCase)
      % using CDE, increasing plama current to shift r-position to
      % reconnect droplets, fine grid required for good convergence
      params = {'center_r', 0.86, 'size', 0.35, 'use_SQP', 0, 'use_cde', 1, 'use_fine_grid', 0};
      [~, ~, L, LX0] = generate_doubletTCV(params{:},testCase.common_params{:});

      t = 0:0.2e-3:5e-3;
      LX = fgex('tcv', t, L, LX0);

      % increase ohmic drive
      ramp_rate = 400;
      Iaind = fgeIaind(L,LX0);
      Ia_dir = -ramp_rate * Iaind;
      Va_dir = L.G.Maa * Ia_dir;

      LX.Vadot0 = LX.Vadot0 + repmat(Va_dir, 1, numel(t));
      LX.Va = LX.Va(:, 1) + LX.Vadot0(:, 1)*(LX.t-LX.t(1));
      LX.Iadot0 = LX.Iadot0 + repmat(Ia_dir, 1, numel(t));
      LX.Ia = LX.Ia(:, 1) + LX.Iadot0(:, 1)*(LX.t-LX.t(1));

      LY = fget(L, LX, 'debug', testCase.verbosity, 'debugplot', testCase.doplot);
      testCase.assertTrue(testCase.check_convergence(L,[],LY),...
        'droplet merging case did not converge');
    end
  end
end
