classdef shapobs_test < meq_test
  % Shape observer test
  %
  % [+MEQ MatlabEQuilibrium Toolbox+]

  %    Copyright 2022-2025 Swiss Plasma Center EPFL
  %
  %   Licensed under the Apache License, Version 2.0 (the "License");
  %   you may not use this file except in compliance with the License.
  %   You may obtain a copy of the License at
  %
  %       http://www.apache.org/licenses/LICENSE-2.0
  %
  %   Unless required by applicable law or agreed to in writing, software
  %   distributed under the License is distributed on an "AS IS" BASIS,
  %   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  %   See the License for the specific language governing permissions and
  %   limitations under the License.

  %%
  properties
    verbosity = false;        % Verbosity level
    SC;                       % Shape Control information
    L0,LX0,LY0;               % Equilibrium information
    perturbation_size = 0.01; % Size of shape perturbation [m]
  end
  properties(ClassSetupParameter)
    tok  = struct('L','tcv','SN','tcv','DN','tcv','SN2','tcv','NT','tcv');
    shot = struct('L',61400,'SN',78531,'DN',68328,'SN2',61400,'NT',79819);
    time = struct('L',0.2  ,'SN',1.2  ,'DN',1.6  ,'SN2',1.6  ,'NT',0.8  );
  end
  properties(TestParameter)
    perturbation = {'radial','vertical'};
  end
  
  %%
  methods(TestClassSetup,ParameterCombination='sequential')
    
    function prepare_equilibrium(testCase,tok,shot,time)
      % Get control points from FBT
      params = {'ilim',3,'icsint',true,'insrc','fbt'};
      [L0_,LX0_] = fbt(tok,shot,time,params{:});
      if shot==61400 && time==1.6
        % Add manually a secondary X-point to test Dsep
        LX0_ = fbtgp(LX0_,...
          0.7,0.6,0,... %r,z,b
           [],[],[],... %fa,fb,fe
           0,0,[],0,... %br,bz,ba,be
        [],[],[],[],... %cr,cz,ca,ce
        [],[],[],[]);   %vrr,vrz,vzz,ve
      end
      % Classify control points
      SC_ = fbt2SC(L0_,LX0_);
      % Add the control points as interpolation points and
      % compute the nominal equilibrium
      [L0_,LX0_,LY0_] = fbt(tok,shot,time,params{:},...
                         'infct',@qintmex,...
                         'rn'   ,SC_.rc,...
                         'zn'   ,SC_.zc);
      res = meq_test.check_convergence(L0_,LX0_.t,LY0_);
      testCase.assertTrue(res,'FBT did not converge');
      % Compute shape observer
      SC_ = shapobs(SC_,L0_,LY0_);
      if testCase.verbosity
        SCdisp(SC_);
      end
      % Store data
      testCase.L0  = L0_;
      testCase.LX0 = LX0_;
      testCase.LY0 = LY0_;
      testCase.SC  = SC_;
    end
    
  end
  
  %%
  methods(Test, TestTags={'TCV'})
    
    function test_shape_observers(testCase,perturbation)
      % Store data for readability purposes
      L0  = testCase.L0; %#ok<*PROPLC>
      LX0 = testCase.LX0;
      LY0 = testCase.LY0;
      SC  = testCase.SC;
      % Perturb equilibrium
      drSC = zeros(size(SC.rc(~SC.lcI)));
      dzSC = zeros(size(SC.zc(~SC.lcI)));
      % rigid radial/vertical perturbations
      LX = LX0;
      switch perturbation
        case 'radial'
          if LY0.lX
            drL  = testCase.perturbation_size*ones(size(LX0.gpr));
            drSC = testCase.perturbation_size*ones(size(SC.rc(~SC.lcI)));
          else
            deltaL  = abs(LX0.gpr - SC.rc(SC.lcL));
            deltaSC = abs(SC.rc(~SC.lcI) - SC.rc(SC.lcL));
            drL  = testCase.perturbation_size*deltaL./max(deltaL);
            drSC = testCase.perturbation_size*deltaSC./max(deltaSC);
          end
          LX.gpr = LX0.gpr+drL;
        case 'vertical'
          dzL  = testCase.perturbation_size*ones(size(LX0.gpz));
          dzSC = testCase.perturbation_size*ones(size(SC.zc(~SC.lcI)));
          LX.gpz = LX0.gpz+dzL;
        otherwise
          error('unknown perturbation');
      end
      % Compute the perturbed equilibrium
      LY = fbtt(L0,LX);
      res = meq_test.check_convergence(L0,LX0.t,LY0);
      testCase.assertTrue(res,'FBT did not converge');
      yc = [LY.Fn-LY0.Fn;
            LY.Brn-LY0.Brn;
            LY.Bzn-LY0.Bzn];
      yc(isnan(yc)) = 0;
      % Compute expected errors
      es0 = SC.drcdes(~SC.lcI,:)'*drSC + ...
            SC.dzcdes(~SC.lcI,:)'*dzSC;
      dr0 = SC.drcdes(~SC.lcI,:)*es0;
      dz0 = SC.dzcdes(~SC.lcI,:)*es0;
      if any(SC.lsD)
        esD0 = es0(SC.lsD)+SC.esD;
        drD0 = SC.drsepdes*esD0;
        dzD0 = SC.dzsepdes*esD0;
      end
      % Compute errors
      es = SC.Mesys*SC.Mysyc*yc;
      dr = SC.drcdes(~SC.lcI,:)*es;
      dz = SC.dzcdes(~SC.lcI,:)*es;
      if any(SC.lsD)
        esD = es(SC.lsD)+SC.esD;
        drD = SC.drsepdes*esD;
        dzD = SC.dzsepdes*esD;
      end
      % Plots
      if testCase.verbosity
        fprintf('Errors: \n');
        for ii=1:numel(es)
          fprintf('%s, %+7.5e\n',SC.dims{ii},es(ii))
        end
        fig = figure;
        ax = axes('Parent',fig);
        SCplot(SC,L0,LY0,...
          'color','b',...
          'parent',ax,...
          'verbose',testCase.verbosity)
        plot(SC.rc(~SC.lcI)+dr0,SC.zc(~SC.lcI)+dz0,...
          '.g','markersize',10,'DisplayName','true errors');
        plot(SC.rc(~SC.lcI)+dr, SC.zc(~SC.lcI)+dz,...
          'or','linewidth',2,'DisplayName','estimated errors');
        if any(SC.lsD)
          plot(SC.rc(SC.lcD)+SC.drsep+drD0,...
               SC.zc(SC.lcD)+SC.dzsep+dzD0,...
            '.g','markersize',10,'HandleVisibility','off');
          plot(SC.rc(SC.lcD)+SC.drsep+drD,...
               SC.zc(SC.lcD)+SC.dzsep+dzD,...
            'or','linewidth',2,'HandleVisibility','off');
        end
        title(sprintf("perturbation: %s",perturbation));
      end
      % Tests
      testCase.verifyEqual(max(abs(es-es0)),0,'AbsTol',testCase.perturbation_size);
    end

  end

end
