classdef (SharedTestFixtures={meq_simulink_fixture}) rzp_slx_test < meq_test
  %% Test class for real time calculations
  % Test the working flow of rzpfastA and rzpfastbgr with different 
  % input parameters their mexification and compare the eigenvalues 
  % calculated with these fast versions versus those calculated with the
  % slow one.
  %
  % [+MEQ MatlabEQuilibrium Toolbox+]

  %    Copyright 2022-2025 Swiss Plasma Center EPFL
  %
  %   Licensed under the Apache License, Version 2.0 (the "License");
  %   you may not use this file except in compliance with the License.
  %   You may obtain a copy of the License at
  %
  %       http://www.apache.org/licenses/LICENSE-2.0
  %
  %   Unless required by applicable law or agreed to in writing, software
  %   distributed under the License is distributed on an "AS IS" BASIS,
  %   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  %   See the License for the specific language governing permissions and
  %   limitations under the License.

  properties
    verbosity = 0;             % 1 to make verbose output, 2 to make plots
   
    rtol      = 1.e-3;          %tolerance on the elements relative difference 
    atol      = 1.e-4;         %tolerance on the elements absolute difference
    gritermax = int32(100);     %maximum number of iteration for the fast growth rate calculation
    grtol     = single(1.e-5);  %tolerance for the fast growth rate calculation
    dt        = 1.e-3;          %time step for all test
    tok       = 'tcv';          %use only tcv for simulink test
    
    t,shot;
    P_error,LYliu,Lrzp;
  end
  
  %% Properties
  properties (ClassSetupParameter)
    shot_time_index = {1,2,3};  %index for the shot number and time
  end

  %% Setup
  methods (TestClassSetup)
    function setup_parameters(testCase,shot_time_index)
      %Setup the shot and time for the test according to the parameter `combination
      shot_list     = [61400,68915,69393];
      time_list     = {0.5:testCase.dt:0.6,0.05:testCase.dt:0.08,0.9:testCase.dt:1};
      testCase.shot = shot_list(shot_time_index);
      testCase.t    = time_list{shot_time_index};
      
      testCase.P_error = sprintf('%s: %s',...
        'tok',testCase.tok,'shot',testCase.shot); 

      [~,~,testCase.LYliu] = liu(testCase.tok,testCase.shot,testCase.t);
      [testCase.Lrzp] = rzpfastgrslx_init();
    end
  end
  
  %% Tests
  methods (Test, TestTags = {'rzp'})   
    function code_test_parameter(testCase)    
      %Test that all rzpfast* functions are operating      
      %Test the simulink is working well
      Ie = single([testCase.LYliu.Ia;testCase.LYliu.Iu]);
      Iy = single(testCase.LYliu.Iy);
      Ip = single(testCase.LYliu.Ip);
      Ieslx = timeseries(Ie,testCase.t,'Name','Ie' );
      Iyslx = timeseries(Iy,testCase.t,'Name','Iy' );
      Ipslx = timeseries(Ip,testCase.t,'Name','Ip' );
      
      TP = struct;
      TP.iter_max = int32(testCase.gritermax);
      TP.tol = single(testCase.grtol);
      
      assignin('base','TP',TP);
      
      configSet = Simulink.ConfigSet;
        set_param(configSet,...
        'name','configurationSettings',...
        'StartTime',num2str(testCase.t(1)),...
        'StopTime' ,num2str(testCase.t(end)),...
        'SolverType', 'Fixed-step',...
        'FixedStep',num2str(testCase.dt)...
        );
      assignin('base','configurationSettings',configSet);

      load_system('rzpfastgrslx');
      simIn = Simulink.SimulationInput('rzpfastgrslx');
      
      inports = createInputDataset('rzpfastgrslx');
      inports{1} = Ieslx;
      inports{2} = Iyslx;
      inports{3} = Ipslx;
      simIn = simIn.setExternalInput(inports);

      simIn = simIn.setModelParameter('SaveOutput','on'); % set to save outport signals
      simIn = simIn.setModelParameter('SignalLogging','off'); % set to save log signals
      simIn = simIn.setModelParameter('OutputSaveName','SimOut');
      simIn = simIn.setModelParameter('SaveFormat','Dataset');
      simIn = simIn.setModelParameter('Solver','FixedStepAuto');
      simIn = simIn.setModelParameter('StartTime',num2str(testCase.t(1)));
      simIn = simIn.setModelParameter('StopTime',num2str(testCase.t(end)));
      simIn = simIn.setModelParameter('FixedStep',num2str(testCase.dt));

      try
        simOut = sim(simIn);
      catch ME
        fprintf('\n %s \n Error caused in simulink model %s by parameters %s \n',...
          ME.message,testCase.model,case_string,testCase.P_error);
          rethrow(ME);
      end
      
      close_system('rzpfastgrslx');
      clear Ieslx Iyslx Ipslx;
      
      %Test simulink vs matlab eigenvalue calculation
      for ii = 1:3
        eval(sprintf('%sslx = simOut.SimOut{ii}.Values.Data'';',simOut.SimOut{ii}.Name));
      end
      
      gamma = zeros(size(testCase.t),'single');
      res = gamma;
      it = gamma;

      for ii = 1:numel(testCase.t)
        Ia = Ie(1:testCase.Lrzp.G.na,ii);
        Iu = testCase.Lrzp.G.Tuuliu*Ie(testCase.Lrzp.G.na+1:end,ii);
        rIp = testCase.Lrzp.rry(:)'*reshape(Iy(:,:,ii),testCase.Lrzp.ny,1);
        [~,~,alpha,~,c] = rzpfastbase(Ia,Iu,Iy(:,:,ii),Ip(:,ii),rIp,...
          testCase.Lrzp.N,testCase.Lrzp.F,testCase.Lrzp.Trd,testCase.Lrzp.Tzd,...
          testCase.Lrzp.ny,testCase.Lrzp.np,testCase.Lrzp.ne,...
          testCase.Lrzp.RBrye,testCase.Lrzp.RBzye,testCase.Lrzp.RBrye_fast,testCase.Lrzp.RBzye_fast);

        [gamma(ii),res(ii),it(ii)] = rzpfastgr(alpha,c,testCase.Lrzp.dd,testCase.Lrzp.dmax,testCase.gritermax,testCase.grtol);
      end
      
      if testCase.verbosity > 1
        figure
        subplot(311)
        plot(testCase.t,gamma);
        hold on;
        plot(testCase.t,gammaslx,'--');
        legend({'meq','slx'})
        title('Gamma')
        ylabel('Gamma [Hz]')
        subplot(312)
        plot(testCase.t,res);
        hold on;
        plot(testCase.t,resslx,'--');
        legend({'meq','slx'})
        title('residual')
        ylabel('res')
        subplot(313)
        plot(testCase.t,it);
        hold on;
        plot(testCase.t,itslx,'--');
        legend({'meq','slx'})
        title('iteration')
        ylabel('it')
        xlabel('time [s]')
      end

     testCase.verifyEqual(double(gamma),double(gammaslx),...
          'AbsTol',testCase.atol,'RelTol',testCase.rtol,sprintf('Growth rate calculation offline vs slx tolerance exceeded with paramters %s',testCase.P_error));
    end
  end
end
