classdef (SharedTestFixtures={mexm_fixture}) rtci_test < meq_test
  % Tests for RTCI (real-time contour integrals)
  %
  % [+MEQ MatlabEQuilibrium Toolbox+]

  %    Copyright 2022-2025 Swiss Plasma Center EPFL
  %
  %   Licensed under the Apache License, Version 2.0 (the "License");
  %   you may not use this file except in compliance with the License.
  %   You may obtain a copy of the License at
  %
  %       http://www.apache.org/licenses/LICENSE-2.0
  %
  %   Unless required by applicable law or agreed to in writing, software
  %   distributed under the License is distributed on an "AS IS" BASIS,
  %   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  %   See the License for the specific language governing permissions and
  %   limitations under the License.

  properties
    L_; LY_;
    rel_tol = 1;
    verbosity = 0;
    method_
  end
  
  properties (ClassSetupParameter)
    nz = {32  64};
    method = {'rtcimex','rtcimexm','rtcics'};
  end
    
  methods(TestClassSetup)
    function rtciTestSetup(testCase,nz,method)
      % Several surfaces to track
      fq = [0.5 0.7 0.95 1];
      
      iterq = 200; % iterations to converge contour calculation
      if strcmp(method,'rtcics'), PP = {'icsint',true ,'ilim',3};
      else,                       PP = {'icsint',false,'ilim',1}; end
      L = liu('ana',1,1,'iterq',iterq,'nz',nz,'pq',sqrt(fq),PP{:});

      % calculate axis location and radius of circle a0
      r0 = 1.05; 
      z0 = 0.05;
      FA = 0;
      FB = 1;
      rBt = 1.4;
      
      [L,LY] = testCase.getCircularEquilibrium(L,r0,z0,FA,FB,rBt);
      testCase.L_ = L;
      testCase.LY_ = LY;
      testCase.method_ = method;
    end
  end

  properties (TestParameter)
    init = {'standard','random'};
  end

  methods(Test, TestTags = {'Unit'})
    function test_rtci_contour(testCase,init)
      
      %%
      L = testCase.L_; % for brevity
      LY = testCase.LY_;
      
      % Initial guess
      switch init
        case 'standard'
          aq = repmat(min([LY.rA-L.rx(1) L.rx(end)-LY.rA LY.zA-L.zx(1) L.zx(end)-LY.zA])*...
            L.pinit*L.pq,L.noq,1); % initial a
        case 'random'
          aq = rand(L.noq,L.npq)*0.5;
      end
      
      F = LY.FB-L.fq*(LY.FB-LY.FA);
      Opo = int8(1);
  
      switch testCase.method_
        case 'rtcics'
          aq = rtcics(aq,LY.Fx,F,LY.Opy,LY.rA,LY.zA,Opo,L.crq,L.czq,L);
          daq = 0;
        case {'rtcimex','rtcimexm'}
          ero=(LY.rA-L.rx(1))*L.idrx;
          ezo=(LY.zA-L.zx(1))*L.idzx;
          for ki = 1:L.P.iterq
            [aq,daq] = feval(testCase.method_,aq,ero,ezo,LY.Fx,F,L.cdrq,L.cdzq,LY.Opy,LY.FA,Opo,L.drx);
          end
      end
      % final call to also get r,z of final contour
      rq = LY.rA + L.crq.*aq;
      zq = LY.zA + L.czq.*aq;

      tol = max(L.drx,L.dzx)*testCase.rel_tol;
      testCase.verifyLessThan(max(abs(LY.aq-aq)),tol,'.mex result failure')
      testCase.verifyLessThan(daq,tol,'rtci did not converge')
            
      if testCase.verbosity
        %%

        txt = sprintf('%dx%d',L.nzx,L.nrx);
        subplot(121)
        plot(LY.rA,LY.zA,'ok')
        hold on
        contour(L.rx,L.zx,LY.Fx,linspace(0,3,31),'-r');
        contour(L.rx,L.zx,LY.Fx,LY.FB*[1 1],'-k','linewidth',2)
        plot(LY.rq,LY.zq,'xm',rq,zq,'ob')
        hold off, axis equal
        title(sprintf('RTCI test: %s, nz=%d',L.P.tokamak,L.nzx))
        
        subplot(122)
        semilogy(L.oq,abs(LY.aq-aq))
        xlabel('\theta'), xlim([0 2*pi]), 
        title('LCFS position error')
        legend(txt,'location','best')
        orient tall
        drawnow; shg;
      end
      
    end
    
    function test_rtci_gaps(testCase)
      
      %%
      L = testCase.L_;
      LY = testCase.LY_;
      
      % Main iteration to compute level curve for Fx = 1 (this call was adapted from liut.m)
      aW = zeros(L.G.nW,L.P.nFW); % initial a
      
      % Values at origin points
      FoW = bintmex(LY.Fx,int32(L.kxW-1),L.cWx);
      
      FW = LY.FB;
      Opo = int8(0);
  
      switch testCase.method_
        case 'rtcics'
          aW = rtcics(aW,LY.Fx,FW,LY.Opy,L.G.rW,L.G.zW,Opo,L.crW,L.czW,L);
          daW = 0;
        case {'rtcimex','rtcimexm'}
          for ki = 1:L.P.iterq
            [aW,daW] = feval(testCase.method_,aW,L.erW,L.ezW,LY.Fx,FW,L.cdrW,L.cdzW,LY.Opy,FoW,Opo,L.drx);
          end
      end

      tol = max(L.drx,L.dzx)*testCase.rel_tol;
      testCase.verifyLessThan(max(abs(LY.aW-aW)),tol,'.mex result failure')
      testCase.verifyLessThan(daW,tol,'rtci did not converge')
            
      if testCase.verbosity
        %%
        % Compute coordinates of the level curves considered points
        r0 = L.G.rW+LY.aW.*L.crW; z0 = L.G.zW+LY.aW.*L.czW; % analytical
        r1 = L.G.rW+   aW.*L.crW; z1 = L.G.zW+   aW.*L.czW; % answer

        clf;
        plot(LY.rA,LY.zA,'ok')
        hold on
        contour(L.rx,L.zx,LY.Fx,linspace(0,3,31),'-r');
        contour(L.rx,L.zx,LY.Fx,LY.FB*[1 1],'-k','linewidth',2)
        plot([L.G.rW,r0].',[L.G.zW,z0].','-xm')
        plot([L.G.rW,r1].',[L.G.zW,z1].','-ob')
        plot(L.G.rl,L.G.zl,'-k')
        hold off, axis equal
        title(sprintf('RTCI test: %s, nz=%d',L.P.tokamak,L.nzx))
      end
      
    end
  end
  
  
end
