classdef meqplotslider_tests < meq_test
  % tests for meqplotslider
  %
  % [+MEQ MatlabEQuilibrium Toolbox+]

  %    Copyright 2022-2025 Swiss Plasma Center EPFL
  %
  %   Licensed under the Apache License, Version 2.0 (the "License");
  %   you may not use this file except in compliance with the License.
  %   You may obtain a copy of the License at
  %
  %       http://www.apache.org/licenses/LICENSE-2.0
  %
  %   Unless required by applicable law or agreed to in writing, software
  %   distributed under the License is distributed on an "AS IS" BASIS,
  %   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  %   See the License for the specific language governing permissions and
  %   limitations under the License.

  properties
    myobj % meqplotslider object
    verbosity = 0;
  end

  properties (ClassSetupParameter)
    calltype = {'simple','LX and LY','multiple LY'}
  end

  methods(TestClassSetup)
    function setup_figure(testCase,calltype)
      % generate sequence of test equilibria
      [L,LX,LY] = fbt('ana',101,[]);

      if testCase.verbosity, vis = 'on'; else, vis = 'off'; end
      hf_ = figure('name','MEQ equilibrium inspector',...
        'Visible',vis);
      testCase.addTeardown(@close,hf_)

      switch calltype
        case 'simple'
          testCase.myobj = meqplotslider(L,LY,...
            'plotfun',@meqplott,... % takes L,LY
            'parent',hf_);
        case 'LX and LY'
          % non-trivial title string
          titstr = @(L,LX,LY) ...
            sprintf('LX.Wk=%3.0f,LY.Wk=%3.0f[kJ]',...
            LX.Wk/1e3,LY.Wk/1e3);

          testCase.myobj = meqplotslider(L,LX,LY,...
            'plotfun',@fbtplot,... % takes L,LX,LY
            'titlefun',titstr,...
            'parent',hf_);
        case 'multiple LY'

           % non-trivial title string
          titstr = @(L,LY1,LY2) ...
            sprintf('Ip = %3.0f | %3.0f[kA]',...
            LY1.Ip/1e3,LY2.Ip/1e3);
          
          % run another case
          LX.Wk = LX.Wk*2;
          LY2 = fbtt(L,LX);

          testCase.myobj = meqplotslider(L,{LY,LY2},...
            'plotfun',@meqplotcontours,... % takes L,LY
            'titlefun',titstr,...
            'parent',hf_);
        otherwise, error('unsupported calltype %s',calltype)
      end

    end
  end

  methods (Test,TestTags={'Integration'})
    function test_slider_callback(testCase)

      slider = testCase.myobj.slider;

      % Simulate slider movement
      val = get(slider,'Value');
      set(slider, 'Value', val+1);

      testCase.verifyEqual(get(slider,'Value'),val+1);
    end

    function test_play_button_callback(testCase)
      % Create the figure and get handles to UI elements
      hf_ = testCase.myobj.hf;
      playButton = findobj(hf_, 'Tag', 'Play');
      slider = testCase.myobj.slider;

      % Simulate button press
      feval(get(playButton, 'Callback'), playButton, []);

      % check that it runs until the end
      testCase.verifyEqual(get(slider,'Value'),slider.Max);
    end

    function test_setting_time(testCase)
      % check that we can programmatically set and retrieve the time slice
      % displayed
      it = 5;
      testCase.myobj.set_time_slice(it);
      testCase.assertEqual(testCase.myobj.get_current_time_slice,it,...
        'could not set time slice displayed')
    end
  end
end
