classdef meqinterp_test < meq_test
  % Test for meqinterp.m
  %
  % [+MEQ MatlabEQuilibrium Toolbox+]

  %    Copyright 2022-2025 Swiss Plasma Center EPFL
  %
  %   Licensed under the Apache License, Version 2.0 (the "License");
  %   you may not use this file except in compliance with the License.
  %   You may obtain a copy of the License at
  %
  %       http://www.apache.org/licenses/LICENSE-2.0
  %
  %   Unless required by applicable law or agreed to in writing, software
  %   distributed under the License is distributed on an "AS IS" BASIS,
  %   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  %   See the License for the specific language governing permissions and
  %   limitations under the License.

  properties (TestParameter)
    method = {'nearest','previous','next','linear'};
    emptydim = {'none','first','second','third','all'}
  end
  
  methods(Test,TestTags={'Unit'})
    function test_ND_interp(testCase,method,emptydim)
      %%
      t = 0:1e-2:0.1;
      nt = numel(t);
      
      m = 4; % default
      n = 3;
      switch emptydim
        case 'first',   m=0;
        case 'second',  n=0;
        case 'third' , nt=0;
        case 'all',  m=0;n=0;nt=0;
      end

      LY = struct('t',t,...
        'field0D',rand(1,nt),...
        'field1D',rand(m,nt),...
        'field2D',rand(m,n,nt));
      
      t_interp = 0:1e-3:0.1;
      LX = meqinterp(LY,t_interp,method);
      
      [~,~,it] = intersect(t,t_interp); % find common time indices
      
      % assert equal on common time points
      testCase.verifyEqual(meqxk(LX,it),LY);
    end
    
    function test_nan_interp(testCase,method)
      % test that attempts at linear interpolation of nan throws an
      % error unless both sides of the interpolation are equal
      t = [0,1];
      LY = struct('t',t,...
        'field0D',[0 nan],...
        'field1D',[0 nan; 0 1],...
        'field2D',cat(3,[0 nan; 0 1],[0 nan; 1 1]));
      
      t_interp = [0,0.5];
      LX = meqinterp(LY,t_interp,method); LX.t = t_interp;
      LX_exp = struct('t',t_interp,...
        'field0D',[0 nan],...
        'field1D',[0 nan; 0,0.5],...
        'field2D',cat(3,[0 nan;0 1],[0,nan;0.5,1]));
      
      switch method
        case 'linear'
          testCase.assertEqual(LX,LX_exp,'unexpected interpolation result');
        case 'nearest'
          it = iround(LY.t,t_interp); LX_exp = meqxk(LY,it); LX_exp.t = t_interp;
          % assert equal on common time points
          testCase.assertEqual(LX,LX_exp);
        case 'previous'
          it = ifloor(LY.t,t_interp);   LX_exp = meqxk(LY,it); LX_exp.t = t_interp;
          % assert equal on common time points
          testCase.assertEqual(LX,LX_exp);
        case 'next'
          it = iceil(LY.t,t_interp); LX_exp = meqxk(LY,it); LX_exp.t = t_interp;
          % assert equal on common time points
          testCase.assertEqual(LX,LX_exp);
      end
    end
    
    function test_inf_interp(testCase,method)
      % Test that inferpolation with inf values works as expected
      t = [0,1];
      LY = struct('t',t,...
        'field0D',[1 Inf],...
        'field1D',[0 Inf; Inf 1]);

      t_interp = [0,0.5];
      % expect 2 because at mid-point of 1/[1 0] we have 1/0.5 
      % expect 1 because at mid-point of [0,Inf] we insert 1
      LX_exp = struct('t',t_interp,...
        'field0D',[1 2],...
        'field1D',[0 1; Inf, 2]);

      % Test that allowInf=false throws expected error if interpolating Infs
      allowInf = false;
      if strcmp(method,'linear')
        testCase.verifyError(@() meqinterp(LY,t_interp,method,allowInf),'meqinterp:NotAllowInf')
      end
      
      % Test interpolation with allowInf
      allowInf = true;
      LX = meqinterp(LY,t_interp,method,allowInf); LX.t = t_interp;
      

      
      switch method
        case 'linear'
          testCase.assertEqual(LX,LX_exp,'unexpected interpolation result');
        case 'nearest'
          it = iround(LY.t,t_interp); LX_exp = meqxk(LY,it); LX_exp.t = t_interp;
          % assert equal on common time points
          testCase.assertEqual(LX,LX_exp);
        case 'previous'
          it = ifloor(LY.t,t_interp);   LX_exp = meqxk(LY,it); LX_exp.t = t_interp;
          % assert equal on common time points
          testCase.assertEqual(LX,LX_exp);
        case 'next'
          it = iceil(LY.t,t_interp); LX_exp = meqxk(LY,it); LX_exp.t = t_interp;
          % assert equal on common time points
          testCase.assertEqual(LX,LX_exp);
      end
    end
    
    function test_logical_interp(testCase,method)
      % assert linear interpolation of logical values gives error, and works otherwise
      t = 0:1e-2:0.1;
      nt = numel(t);
      m = 4;
      n = 3;
      LY = struct('t',t,...
        'field0D',randn(1,nt)>0,...
        'field1D',randn(m,nt)>0,...
        'field2D',randn(m,n,nt)>0);
      
      t_interp = 0:1e-3:0.1;
      if isequal(method,'linear')
        testCase.verifyError(@() meqinterp(LY,t_interp,method),'meqinterp:interpNotAllowed');
      else
        LX = meqinterp(LY,t_interp,method); LX.t = t_interp;
        [~,~,it] = intersect(t,t_interp); % find common time indices
        % assert equal on common time points
        testCase.assertEqual(meqxk(LX,it),LY);
      end
    end
    
    function test_out_of_time_range(testCase,method)
      % check appropriate error for out of range
      t=[0,1]; nt = numel(t); m=2;n=3;
      LY = struct('t',t,...
        'field0D',randn(1,nt),...
        'field1D',randn(m,nt),...
        'field2D',randn(m,n,nt));
      
      for ti=[-1,2]% test extrapolating before first and after last time
        switch method
          case 'linear'
            testCase.verifyError(@() meqinterp(LY,ti,method),'meqinterp:OutOfTimeRange')
          case 'previous' % works for extrapolation after last value
            if ti>max(t), LX_exp = meqxk(LY,'last');
              testCase.verifyEqual(meqinterp(LY,ti,method),LX_exp,'did not return expected value');
            else
              testCase.verifyError(@() meqinterp(LY,ti,method),'meqinterp:OutOfTimeRange')
            end
          case 'next' % works for extrapolation before first value
            if ti<min(t), LX_exp = meqxk(LY,1);
              testCase.verifyEqual(meqinterp(LY,ti,method),LX_exp,'did not return expected value');
            else
              testCase.verifyError(@() meqinterp(LY,ti,method),'meqinterp:OutOfTimeRange')
            end
          case 'nearest' % works for both ends
            if ti>max(t), LX_exp = meqxk(LY,'last');
            else,         LX_exp = meqxk(LY,1     );
            end
            testCase.verifyEqual(meqinterp(LY,ti,method),LX_exp,'did not return expected value');
          otherwise 
            error('unexpected method %s',method); % should not get here
        end
      end
    end
  end
end