classdef meqdFdI_test < meq_jacobian_test
  % Test computation of partial derivatives of post-processing outputs of F operator
  %
  % [+MEQ MatlabEQuilibrium Toolbox+]

  %    Copyright 2022-2025 Swiss Plasma Center EPFL
  %
  %   Licensed under the Apache License, Version 2.0 (the "License");
  %   you may not use this file except in compliance with the License.
  %   You may obtain a copy of the License at
  %
  %       http://www.apache.org/licenses/LICENSE-2.0
  %
  %   Unless required by applicable law or agreed to in writing, software
  %   distributed under the License is distributed on an "AS IS" BASIS,
  %   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  %   See the License for the specific language governing permissions and
  %   limitations under the License.

  properties
    verbosity = 1;
    relTol = 1e-3;
  end

  properties(TestParameter)
    shot = struct('circular',1,'diverted',2,'squashed',5,...
	meq_jacobian_test.doublets_under_test{:});
    usecs = struct('false',false,'true',true);
    algosNL = {'all-nl','all-nl-Fx','Newton-GS'};
  end

  methods (Test,TestTags={'Jacobian'})
    function test_meqdFdI(testCase,shot,usecs,algosNL)
      % Compares results of meqdFdI with and without analytical jacobian

      testCase.assumeFalse(usecs && floor(shot/10) == 8,'Skipping meqdFdI tests for multi-domain cases with icsint=true')

      PP = {'tol',1e-6, ...
            'tolF',1e-10, ...
            'agcon',{'Ip','bp','qA'}, ...
            'algoNL',algosNL, ...
            'algoGMRES','aya', ...
            'anajac',true, ...
            'mkryl',150};
      if strcmpi(algosNL,'Newton-GS')
        PP = [PP,{'algoF','newton','usepreconditioner',true}];
      end
      if usecs
        PP = [PP,{'icsint',true,'ilim',3}];
      else
        PP = [PP,{'icsint',false,'ilim',1}];
      end

      % Setup
      [L,LX] = fge('ana',shot,0,PP{:});

      if usecs
        % Check we aren't too close to cell boundaries
        mindz = min(abs([LX.zA;LX.zB] - L.zx.'),[],[1,2]);
        mindr = min(abs([LX.rA;LX.rB] - L.rx.'),[],[1,2]);
        testCase.assumeTrue(min(mindr,mindz)>1e-4,'Skipping test as axis/boundary point is too close to grid cell boundary');
      end

      % Compute derivatives
      names = {'Iy', 'Fx', 'FA', 'FB', 'rA', 'zA', 'Fn', 'Brn', 'Bzn', 'Ft', 'Ip', 'bp', 'bpD', 'li', 'qA', 'Wk'};
      nout = numel(names);
      values_ana = cell(1,nout);
      values_num = cell(1,nout);

      [values_ana{:}] = meqdFdI(L ,LX);

      L_ = L;
      L_.P.anajac = false;
      [values_num{:}] = meqdFdI(L_,LX);

      % Debug
      if testCase.verbosity
        figure('Name',sprintf('shot=%d algoNL=%s icsint=%d',shot,algosNL,usecs));
        for ii = 1:nout
          num = values_num{ii} ;
          ana = values_ana{ii} ;
          ax = subplot(4,4,ii);
          plot(ax,abs(num - ana)/max(abs(ana)));
          ax.YScale = 'log';
          ax.YLim = [1e-16,1];
          title(ax,names{ii});
        end
      end

      % Compare outputs
      for ii = 1:nout
        num =      values_num{ii} ;
        ana = full(values_ana{ii});

        testCase.verifyEqual(size(num),size(ana),sprintf('Size of output ''d%sdx'' of meqdFdI with and without analytical jacobian do not match',names{ii}));
        if isempty(ana), continue; end % Skip empty fields
        tol = max(testCase.relTol*max(abs(ana(:))),5e-10);
        testCase.verifyEqual(num,ana,'AbsTol',tol,sprintf('Output ''d%sdx'' of meqdFdI with and without analytical jacobian do not match',names{ii}));
      end
    end
  end

end
