classdef meqbdfield_tests < meq_test
% Tests for meqbdfield
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

  properties
    verbosity = 0;
  end

  properties(TestParameter)
    null_angle = { 0, pi/8, +pi/6, +pi/4, +pi/2,  -pi/3 }
    sIp = {-1,1};
  end

  methods(Test,TestTags={'Unit'})
    function test_basic_call(testCase,null_angle,sIp)
      L = fbt('ana',1,[]);
      L.G = meqg(L.G,L.P,'Brxa','Bzxa');

      % control points in a 3x3 square close to the null
      xx = L.P.ac*0.1*(-1:1);
      [rc,zc] = meshgrid(xx,xx);

      % rotate control points to match null orientation
      rc1 =  rc.*cos(null_angle) + zc.*sin(null_angle);
      zc1 = -rc.*sin(null_angle) + zc.*cos(null_angle);

      % radial offset
      rc = rc1+L.P.r0;
      zc = zc1;
        
      L.P.onull = null_angle; % angle (radian)
      L.P.rnull = L.P.r0;
      L.P.znull = 0;
      L.P.gBpnull = 0.01;
      L.P.nullmethod = 'fit';

      [Fc,Brc,Bzc,dnc,Fx,Brx,Bzx] = meqbdfield(L,sIp,rc,zc);

      % check that tangential field above/below null is as expected
      Br_target =  cos(null_angle) * (sIp*xx*L.P.gBpnull)';
      Bz_target = -sin(null_angle) * (sIp*xx*L.P.gBpnull)';

      testCase.verifyEqual(Brc(:,2),Br_target ,'AbsTol',L.P.gBpnull*1e-2)
      testCase.verifyEqual(Bzc(:,2),Bz_target ,'AbsTol',L.P.gBpnull*1e-2)
      testCase.verifyEqual(dnc(2,2),0)

      Fn = Fc(2,2);
      doplot(L,Fx,Brx,Bzx,Fn,rc,zc,sIp,testCase.verbosity>0);
    end

    function test_doublets(testCase,null_angle,sIp)
      L = fbt('ana',81,[]); %#ok<*PROP>
      L.G = meqg(L.G,L.P,'Brxa','Bzxa');

      L.P.onull = null_angle; % angle (radian)
      L.P.rnull = L.P.r0*[1;1];
      L.P.znull = [1;-1]*L.P.ac*0.5;
      L.P.gBpnull = 0.01;
      L.P.nullmethod = 'fit';

      rc = L.P.rnull; zc=L.P.znull;

      tol = sqrt(eps);
      [Fc,Brc,Bzc,dnc,Fx,Brx,Bzx] = meqbdfield(L,sIp,rc,zc);
      testCase.verifyEqual(Brc,[0;0],'AbsTol',tol)
      testCase.verifyEqual(Bzc,[0;0],'AbsTol',tol)
      testCase.verifyEqual(dnc,[0;0],'AbsTol',tol)
      
      doplot(L,Fx,Brx,Bzx,Fc(1,1),rc,zc,sIp,testCase.verbosity>0);
    end
  end
end

function doplot(L,Fx,Brx,Bzx,Fn,rc,zc,sIp,visible)
if visible, vis='on'; else, vis='off'; end
hf = figure('Visible',vis);
% Always close invisible figures
if ~visible, clean = onCleanup(@() close(hf)); end

% debug plots
clf;
Fval = (-10e-3:1e-3:10e-3); % values to plot
contourf(L.rx,L.zx,Fx-Fn,Fval); hold on; colorbar;
Bpx = sqrt(Brx.^2 + Bzx.^2);
contour(L.rx,L.zx,Bpx,0:1e-3:5e-3,'r--')
plot(rc,zc,'ob');
meqgplot(L.G,gca,'vlw');
axis equal;
title(sprintf('\\theta_{null} = %2.1f^o,sIp=%d',180/pi*L.P.onull,sIp))
drawnow;
end
