classdef (TestTags={'Unit','mex'}) meq_mexargs_test < meq_test
% MEQ_MEXARGS_TEST test class for C MEX-files argument checks 
%
% Uses the mexargs_test class to construct the test case and define
% the test methods. Defines the mexfile property with all MEX-file names
% and each corresponding function defining the checks
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

%% begin mexargs_test boilerplate

  properties
    obj
  end

  methods (TestClassSetup)
    function setup_mexargs(testCase,mexfile)
      % Setup paths to mexargs project
      testCase.applyFixture(matlab.unittest.fixtures.PathFixture(testCase.mexargs_path));
      % Initialize mexargs object for the current MEX-file
      % [The following if block is an MEQ customization to avoid creating
      %  individual functions for all combinations of MEX-based bf sets
      %  and mode values]
      [tokens] = regexp(mexfile,'(bf..)mex_(\d)+','tokens','once');
      if ~isempty(tokens)
        S = meq_mexargs_test.bfxx_args(tokens{1},str2double(tokens{2}));
      else
        S = testCase.([mexfile,'_args']);
      end
      % Initialize mexargs_test object with test methods
      testCase.obj = mexargs_test(S);
    end
  end

  methods (Test)
    function check_default(testCase)
      testCase.obj.check_default(testCase);
    end
    function check_nargin(testCase)
      testCase.obj.check_nargin(testCase);
    end
    function check_arguments(testCase)
      testCase.obj.check_arguments(testCase);
    end
    function check_nargout(testCase)
      testCase.obj.check_nargout(testCase);
    end
  end

%% end   mexargs_test boilerplate

  properties
    mexargs_path = fullfile(fileparts(fileparts(mfilename('fullpath'))),'genlib','mexargs');
  end
 
  properties (ClassSetupParameter)
    mexfile = [meq_mexargs_test.bfset_mode_list(),{
      'asxymex'   ,...
      'bavxmex'   ,...
      'bavxmex_8' ,...
      'bboxmex'   ,...
      'bfprmex'   ,...
      'bintmex'   ,...
      'bslvmex'   ,...
      'cholmex'   ,...
      'cholmex_3' ,...
      'cizrmex'   ,...
      'fbndmex'   ,...
      'fl4pmex'   ,...
      'fsgimex'   ,...
      'fsg2mex'   ,...
      'gszrmex'   ,...
      'iatamex'   ,...
      'ipm4mex'   ,...
      'ipm2mex'   ,...
      'ipmhmex'   ,...
      'ipmjmex'   ,...
      'locRmex'   ,...
      'locRmex_4' ,...
      'locRmex_5' ,...
      'locSmex'   ,...
      'locSmex_4' ,...
      'minQmex'   ,...
      'minQmex_4' ,...
      'minQmex_5' ,...
      'nfdbmex'   ,...
      'pdommex'   ,...
      'qintmex'   ,...
      'respmex'   ,...
      'rtcimex'   ,...
      'rtcimex_13',...
      'rtcimex_15',...
      'shapmex'   ,...
      'uatamex'   ,...
      'vizrmex'   ,...
      }];
  end
  
  methods (Static)
    function S = asxymex_args %#ok<*DEFNU>
      nzx = 65; nrx = 31;
      ri =  1; ro =  2;
      zl = -1; zu =  1;
      dzx = (zu-zl)/(nzx-1);
      drx = (ro-ri)/(nrx-1);
      dimw = 20;
      S = mexargs('asxymex',{'eq',10},{'le',15});
      S.addarg(rand(nzx,nrx),{'type','double','numel',[]});
      S.addarg(linspace(zl,zu,nzx).',{'type','double'});
      S.addarg(linspace(ri,ro,nrx).',{'type','double'});
      S.addarg(2,{'type','numeric','scalar',[]});
      S.addarg(dzx,{'type','numeric','scalar',[]});
      S.addarg(drx,{'type','numeric','scalar',[]});
      S.addarg(1/dzx,{'type','numeric','scalar',[]});
      S.addarg(1/drx,{'type','numeric','scalar',[]});
      S.addarg(true(nzx,nrx),{'type','logical','numel',[]});
      S.addarg(dimw,{'type','numeric','scalar',[]});
    end
    
    function S = bavxmex_args
      nX = 3;
      nr = 11; nz = 2;
      S = mexargs('bavxmex',{'one_of',[7,8]},{'le',1});
      S.addarg(rand(nX,1),{'type','double'});
      S.addarg(rand(nX,1),{'type','double','numel',[]});
      S.addarg(rand(nX,1),{'type','double','numel',[]});
      S.addarg(rand(nX,1),{'type','double','numel',[]});
      S.addarg(0,{'scalar_or_empty',[]});
      S.addarg(rand(nr,nz),{'type','double'});
      S.addarg(rand(nr,nz),{'type','double','numel',[]});
    end
    
    function S = bavxmex_8_args
      S = meq_mexargs_test.bavxmex_args;
      S.arg( 7).check(3:4) = []; % Remove size check for 7th argument
      S.addarg([],{});
    end
    
    function S = bboxmex_args
      nzx = 65; nrx = 31;
      ri =  1; ro =  2;
      zl = -1; zu =  1;
      S = mexargs('bboxmex',{'eq',3},{'le',1});
      S.addarg(true(nzx,nrx),{'type','logical'});
      S.addarg(linspace(zl,zu,nzx).',{'type','double','numel',[]});
      S.addarg(linspace(ri,ro,nrx).',{'type','double','numel',[]});
    end

    function C = bfset_mode_list()
      % Generate list of (bfset/mode) values (MEX-based sets only)
      sets = {'3i','3p','ab','ef'};
      modes = [0:8,11:13,15,16,17,91,92];
      nsets = numel(sets);
      nmodes = numel(modes);
      C = cell(1,nmodes*nsets);
      k = 1;
      for set = sets
        for mode = modes
          C{k} = sprintf('bf%smex_%d',set{1},mode);
          k = k+1;
        end
      end
    end
    
    function S = bfxx_args(bf,mode)
      ng = 3;
      switch bf
        case 'bf3i'
          par.gNg = rand(11,ng);
          par.IgNg = bfprmex(par.gNg);
          par.fPg = [1;0;0];
          par.fTg = 1-par.fPg;
        case 'bf3p'
          par = true;
        case 'bfab'
          par = [2,2];
          ng = sum(par);
        case 'bfef'
          par = [3,2];
          ng = sum(par);
        otherwise
          error('Unknown bf MEX-file: %s',bf);
      end
      %
      nzx = 65; nrx = 31;
      nzy = nzx-2; nry = nrx-2;
      nQ = 19;
      nD = 8;
      %
      S = mexargs([bf,'mex'],{'ge',2},{'le',3});
      S.addarg(mode,{'type','numeric','scalar',[]});
      S.addarg(par,{}); % No tests for par values
      %
      switch mode
        case 0
          S.nargin.check(3:4) = {'eq',2};
          S.nargout.check(1:2) = {'le',3};
        case {1,11}
          nargout = ifc(mode==1,3,5);
          S.nargin.check(3:4) = {'eq',8};
          S.nargout.check(1:2) = {'le',nargout};
          S.addarg(rand(nzx,nrx),{'type','double','numel',[]});
          S.addarg(rand,{});
          S.addarg(rand,{});
          S.addarg(randi(2,nzy,nry,'int8')-1,{'type','int8'});
          S.addarg(rand(nry,1),{'type','double','numel',[]});
          S.addarg(rand(nry,1),{'type','double','numel',[]});
        case {2,12}
          nargout = ifc(mode==2,2,6);
          S.nargin.check(3:4) = {'eq',5};
          S.nargout.check(1:2) = {'le',nargout};
          S.addarg(linspace(0,1,nQ).',{'type','double'});
          S.addarg(rand,{});
          S.addarg(rand,{});
        case {3,13}
          nargout = ifc(mode==3,4,8);
          S.nargin.check(3:4) = {'eq',8};
          S.nargout.check(1:2) = {'le',nargout};
          S.addarg(rand(ng,1),{'type','double','numel',[]});
          S.addarg(rand,{});
          S.addarg(rand,{});
          S.addarg(rand(ng,1),{'type','double','numel',[]});
          S.addarg(rand(ng,1),{'type','double','numel',[]});
          S.addarg(rand,{'type','numeric','scalar',[]});
        case 4
          S.nargin.check(3:4) = {'eq',8};
          S.nargout.check(1:2) = {'le',2};
          S.addarg(rand(nzx,nrx),{'type','double'});
          S.addarg(rand,{});
          S.addarg(rand,{});
          S.addarg(randi(2,nzy,nry,'int8')-1,{'type','int8','numel',[]});
          S.addarg(randi(nry,nD,1,'uint32'),{'type','uint32'});
          S.addarg(rand(nD,2),{'type','double','numel',[]});
        case {5,15}
          nargout = ifc(mode==5,2,4);
          S.nargin.check(3:4) = {'eq',5};
          S.nargout.check(1:2) = {'le',nargout};
          S.addarg([],{});
          S.addarg(rand,{});
          S.addarg(rand,{});
        case {6,16}
          nargout = ifc(mode==6,2,4);
          S.nargin.check(3:4) = {'eq',8};
          S.nargout.check(1:2) = {'le',nargout};
          S.addarg([],{});
          S.addarg(rand,{});
          S.addarg(rand,{});
          S.addarg(rand,{'type','numeric','scalar',[]});
          S.addarg(rand,{'type','numeric','scalar',[]});
          S.addarg(rand,{'type','numeric','scalar',[]});
        case {7,17}
          S.nargin.check(3:4) = {'eq',5};
          S.nargout.check(1:2) = {'le',2};
          S.addarg([],{});
          S.addarg(rand,{});
          S.addarg(rand,{});
        case 8
          S.nargin.check(3:4) = {'eq',10};
          S.nargout.check(1:2) = {'le',1};
          S.addarg(rand(nzx,nrx),{'type','double'});
          S.addarg(rand,{});
          S.addarg(rand,{});
          S.addarg(randi(2,nzy,nry,'int8')-1,{'type','int8','numel',[]});
          S.addarg(rand(ng,1),{'type','double','numel',[]});
          S.addarg(rand,{'type','numeric','scalar',[]});
          S.addarg(rand,{'type','numeric','scalar',[]});
          S.addarg(rand(nry,1),{'type','double','numel',[]});
        case 91
          S.nargin.check(3:4) = {'eq',5};
          S.nargout.check(1:2) = {'le',2};
          S.addarg(rand(nQ,1),{'type','double'});
          S.addarg(rand,{});
          S.addarg(rand,{});
        case 92
          S.nargin.check(3:4) = {'eq',5};
          S.nargout.check(1:2) = {'le',6};
          S.addarg(rand(nQ,1),{'type','double'});
          S.addarg(rand,{});
          S.addarg(rand,{});
        otherwise
          error('BF Mode=%d does not exist');
      end
      %
      % Customizations per mode and BF set
      if mode == 6 && strcmp(bf,'bf3i')
        S.arg( 6).check = {};
        S.arg( 7).check = {};
        S.arg( 8).check = {};
      elseif mode == 6 && strcmp(bf,'bf3p')
        S.arg( 6).check = {};
      end
      %
    end
    
    function S = bfprmex_args
      nq = 21; ng = 3;
      S = mexargs('bfprmex',{'eq',1},{'le',1});
      S.addarg(rand(nq,ng),{'type','double','ncols',[]});
    end
    
    function S = bintmex_args
      nzx = 65; nrx = 31;
      ni = 12;
      S = mexargs('bintmex',{'eq',3},{'le',1});
      S.addarg(rand(nzx,nrx),{'type','double'});
      S.addarg(randi(nzx*nrx,ni,1,'int32'),{'type','int32'});
      S.addarg(rand(4,ni),{'type','double','numel',[]});
    end
    
    function S = bslvmex_args
      ne = 12;
      ni = 3;
      iters = 12;
      S = mexargs('bslvmex',{'eq',6},{'le',2});
      S.addarg(rand(ne, 1),{'type','double'});
      S.addarg(rand(ni, 1),{'type','double'});
      S.addarg(rand(ni, 1),{'type','double','numel',[]});
      S.addarg(rand(ne,ni),{'type','double','numel',[]});
      S.addarg(rand(ni,ne),{'type','double','numel',[]});
      S.addarg(iters,{'type','numeric','scalar',[]});
    end
    
    function S = cholmex_args
      n = 3;
      nA = n*(n+1)/2;
      S = mexargs('cholmex',{'one_of',[2,3]},{'le',1});
      S.addarg(n,{'type','numeric','scalar',[]});
      S.addarg(rand(nA,1),{'type','double','numel',[]});
    end
    
    function S = cholmex_3_args
      n = 3;
      nb = 2;
      S = meq_mexargs_test.cholmex_args;
      S.addarg(rand(n,nb),{'type','double','nrows',[]});
    end
    
    function S = cizrmex_args
      nzx = 65; nrx = 31; nV = 1; nQ = 1; nD = 1;
      ri =  1; ro =  2;
      zl = -1; zu =  1;
      nzy = nzx-2; nry = nrx-2;
      ds = (ro-ri)*(zu-zl)/(nry+1)/(nzy+1);
      S = mexargs('cizrmex',{'eq',7},{'le',1});
      S.addarg(rand(nzx,nrx),{'type','double'});
      S.addarg(int8(rand(nzy,nry)>0.5),{'type','int8'});
      S.addarg(rand(nzy*nry,nV),{'type','double','nrows',[]});
      S.addarg(ds,{'type','numeric','scalar',[]});
      S.addarg(sort(rand(nQ,1)),{'type','double'});
      S.addarg(rand(nD,1),{'type','double'});
      S.addarg(rand(nD,1),{'type','double','numel',[]});
    end
    
    function S = fbndmex_args
      nl = 64;
      nX = 3;
      S = mexargs('fbndmex',{'eq',7},{'le',6});
      S.addarg(rand(nl,1),{'type','double'});
      S.addarg(rand(nl,1),{'type','double','numel',[]});
      S.addarg(rand(nl,1),{'type','double','numel',[]});
      S.addarg(rand(nX,1),{'type','double'});
      S.addarg(rand(nX,1),{'type','double','numel',[]});
      S.addarg(rand(nX,1),{'type','double','numel',[]});
      S.addarg(Inf,{'type','numeric','scalar',[]});
    end
    
    function S = fl4pmex_args
      nzx = 65; nrx = 31;
      nl = 12;
      S = mexargs('fl4pmex',{'eq',4},{'le',3});
      S.addarg(rand(nzx,nrx),{'type','double'});
      S.addarg(randi(nzx*nrx,nl,1,'int32'),{'type','int32'});
      S.addarg(rand(4,nl),{'type','double','numel',[]});
      S.addarg(Inf,{'type','numeric','scalar',[]});
    end
    
    function S = fsgimex_args
      noq = 32; npq = 18; nQ = npq+1;
      S = mexargs('fsgimex',{'eq',12},{'le',11});
      S.addarg(rand(noq,npq),{'type','double'});
      S.addarg(rand(noq,npq),{'type','double','numel',[]});
      S.addarg(rand(noq,npq),{'type','double','numel',[]});
      S.addarg(rand(noq,npq),{'type','double','numel',[]});
      S.addarg(1,{'type','numeric','scalar',[]});
      S.addarg(rand,{'type','numeric','scalar',[]});
      S.addarg(rand,{'type','numeric','scalar',[]});
      S.addarg(rand,{'type','numeric','scalar',[]});
      S.addarg(true,{'type','logical','scalar',[]});
      S.addarg(2,{'type','numeric','scalar',[]});
      S.addarg(rand(nQ,1),{'type','double'});
      S.addarg(2*pi/noq,{'type','numeric','scalar',[]});
    end
    
    function S = fsg2mex_args
      noq = 32; npq = 18;
      S = mexargs('fsg2mex',{'eq',4},{'le',2});
      S.addarg(rand(noq,npq),{'type','double'});
      S.addarg(1,{'type','numeric','scalar',[]});
      S.addarg(rand(noq,1),{'type','double'});
      S.addarg(noq/(2*pi),{'type','numeric','scalar',[]});
    end

    function S = gszrmex_args
      nzx = 65; nrx = 31;
      nzy = nzx-2; nry = nrx-2;
      S = mexargs('gszrmex',{'eq',9},{'le',1});
      S.addarg(rand(2*(nrx+nzx-2),1),{'type','double','numel',[]});
      S.addarg(rand(nzy,nry),{'type','double'});
      S.addarg(rand(nry,  1),{'type','double','numel',[]});
      S.addarg(rand(nzy,nry),{'type','double','numel',[]});
      S.addarg(rand(nzy,nry),{'type','double','numel',[]});
      S.addarg(rand(nzy,nry),{'type','double','numel',[]});
      S.addarg(rand,{'type','numeric','scalar',[]});
      S.addarg(rand,{'type','numeric','scalar',[]});
      S.addarg(rand,{'type','numeric','scalar',[]});
    end
    
    function S = iatamex_args
      nd = 72; ne = 18; ni = 4;
      S = mexargs('iatamex',{'eq',2},{'le',1});
      S.addarg(rand(nd,ni),{'type','double'});
      S.addarg(rand(ne,ni),{'type','double','ncols',[]});
    end
    
    function S = ipm4mex_args
      n = 3;
      nH = n*(n+1)/2;
      tol = 1e-4; niter = 20; verb = false;
      S = mexargs('ipm4mex',{'eq',8},{'le',4});
      S.addarg(rand(nH,1),{'type','double','numel',[]});
      S.addarg(rand( n,1),{'type','double'});
      S.addarg(         1,{'type','numeric','scalar',[]});
      S.addarg(rand( n,1),{'type','double','numel',[]});
      S.addarg(rand( n,1),{'type','double','numel',[]});
      S.addarg(       tol,{'type','numeric','scalar',[]});
      S.addarg(     niter,{'type','numeric','scalar',[]});
      S.addarg(      verb,{'type','logical','scalar',[]});
    end
    
    function S = ipm2mex_args
      n = 3;
      nH = n*(n+1)/2;
      ni = 2;
      tol = 1e-4; niter = 20; verb = false;
      S = mexargs('ipm2mex',{'eq',10},{'le',5});
      S.addarg(rand(nH,1),{'type','double'});
      S.addarg(rand( n,1),{'type','double','numel',[]});
      S.addarg(rand(ni,n),{'type','double','ncols',[]});
      S.addarg(rand(ni,1),{'type','double','numel',[]});
      S.addarg(rand( n,1),{'type','double','numel',[]});
      S.addarg(rand(ni,1),{'type','double','numel',[]});
      S.addarg(rand(ni,1),{'type','double','numel',[]});
      S.addarg(       tol,{'type','numeric','scalar',[]});
      S.addarg(     niter,{'type','numeric','scalar',[]});
      S.addarg(      verb,{'type','logical','scalar',[]});
    end
    
    function S = ipmhmex_args
      nh = 12; nd = 72; ne = 19;
      nu = nh*(nh+1)/2;
      tol = 1e-4; niter = 20;
      S = mexargs('ipmhmex',{'eq',12},{'le',3});
      S.addarg(rand(nh,nd),{'type','real'});
      S.addarg(rand(ne,nd),{'type_match',1,'ncols',[]});
      S.addarg(rand(nh,ne),{'type_match',1,'numel',[]});
      S.addarg(rand(ne,nh),{'type_match',1,'numel',[]});
      S.addarg(rand(nh,nh),{'type_match',1,'numel',[]});
      S.addarg(rand(nu, 1),{'type_match',1,'numel',[]});
      S.addarg(rand(nd, 1),{'type_match',1,'numel',[]});
      S.addarg(rand(ne, 1),{'type_match',1,'numel',[]});
      S.addarg(rand(nh, 1),{'type_match',1,'numel',[]});
      S.addarg(          1,{'type','numeric','scalar',[]});
      S.addarg(      niter,{'type','numeric','scalar',[]});
      S.addarg(        tol,{'type','numeric','scalar',[]});
    end
    
    function S = ipmjmex_args
      nj = 4; ne = 19;
      nc = 2;
      nu = nj*(nj+1)/2;
      tol = 1e-4; niter = 20;
      S = mexargs('ipmjmex',{'eq',13},{'le',3});
      S.addarg(rand(nu, 1),{'type','real','numel',[]});
      S.addarg(rand(nj,ne),{'type_match',1});
      S.addarg(rand(ne,nj),{'type_match',1,'numel',[]});
      S.addarg(rand(ne, 1),{'type_match',1,'numel',[]});
      S.addarg(rand(nj, 1),{'type_match',1,'numel',[]});
      S.addarg(rand(ne, 1),{'type_match',1,'numel',[]});
      S.addarg(rand(nj, 1),{'type_match',1,'numel',[]});
      S.addarg(rand(nc,nj),{'type_match',1,'ncols',[]});
      S.addarg(rand(nc, 1),{'type_match',1,'numel',[]});
      S.addarg(rand(nc, 1),{'type_match',1,'numel',[]});
      S.addarg(rand(nc, 1),{'type_match',1,'numel',[]});
      S.addarg(        tol,{'type','numeric','scalar',[]});
      S.addarg(      niter,{'type','numeric','scalar',[]});
    end
    
    function S = locRmex_args
      nQ = 19; nR = 4;
      S = mexargs('locRmex',{'ge',3,'le',5},{'le',1});
      S.addarg(linspace(0,1,nQ).',{'type','double'});
      S.addarg(rand(nQ,1),{'type','double','numel',[]});
      S.addarg(rand(nR,1),{'type','double'});
    end
    
    function S = locRmex_4_args
      S = meq_mexargs_test.locRmex_args;
      S.addarg(1,{'type','numeric','scalar',[]});
    end
    
    function S = locRmex_5_args
      S = meq_mexargs_test.locRmex_4_args;
      S.addarg(NaN,{'type','numeric','scalar',[]});
    end
    
    function S = locSmex_args
      nQ = 19; nS = 4;
      S = mexargs('locSmex',{'ge',3,'le',4},{'le',1});
      S.addarg(linspace(0,1,nQ),{'type','double'});
      S.addarg(rand(nQ,1),{'type','double','numel',[]});
      S.addarg(rand(nS,1),{'type','double'});
    end
    
    function S = locSmex_4_args
      S = meq_mexargs_test.locSmex_args;
      S.addarg(NaN,{'type','numeric','scalar',[]});
    end
    
    function S = minQmex_args
      nQ = 19;
      S = mexargs('minQmex',{'ge',3,'le',5},{'le',2});
      S.addarg(linspace(0,1,nQ).',{'type','double'});
      S.addarg(rand(nQ,1),{'type','double','numel',[]});
      S.addarg(1,{'type','numeric','scalar',[]});
    end
    
    function S = minQmex_4_args
      S = meq_mexargs_test.minQmex_args;
      S.addarg(1,{'type','numeric','scalar',[]});
    end
    
    function S = minQmex_5_args
      S = meq_mexargs_test.minQmex_4_args;
      S.addarg(NaN,{'type','numeric','scalar',[]});
    end
    
    function S = nfdbmex_args
      nzx = 65; nrx = 31;
      S = mexargs('nfdbmex',{'eq',1},{'le',1});
      S.addarg(rand(nzx,nrx),{'type','double'});
    end
    
    function S = pdommex_args
      nzx = 65; nrx = 31;
      nzy = nzx-2; nry = nrx-2;
      S = mexargs('pdommex',{'eq',4},{'le',1});
      S.addarg(rand(nzx,nrx),{'type','double'});
      S.addarg(1,{'type','numeric','scalar',[]});
      S.addarg(1,{'type','numeric','scalar',[]});
      S.addarg(true(nzy,nry),{'type','logical','numel',[]});
    end
    
    function S = qintmex_args
      nzx = 65; nrx = 31;
      ri =  1; ro =  2;
      zl = -1; zu =  1;
      ni = 12;
      n = 9;
      S = mexargs('qintmex',{'eq',6},{'le',7});
      S.addarg(linspace(ri,ro,nrx).',{'type','double'});
      S.addarg(linspace(zl,zu,nzx).',{'type','double'});
      S.addarg(rand(nzx,nrx),{'type','double','numel',[]});
      S.addarg(rand(ni,1),{'type','double'});
      S.addarg(rand(ni,1),{'type','double','numel',[]});
      S.addarg(rand(6,n),{'type','double','nrows',[]});
    end
    
    function S = respmex_args
      mA = 12; nA = 25; nB = 3; nC = 32; nD = 1;
      S = mexargs('respmex',{'eq',4},{'le',1});
      S.addarg(rand(mA,nA),{'type','double'});
      S.addarg(rand(nA,nB),{'type','double','nrows',[]});
      S.addarg(rand(mA,nC),{'type','double','nrows',[]});
      S.addarg(rand(nC,nD),{'type','double','nrows',[]});
    end
    
    function S = rtcimex_args
      nzx = 65; nrx = 31;
      nzy = nzx-2; nry = nrx-2;
      noq = 32; npq = 18; nD = 2;
      S = mexargs('rtcimex',{'one_of',[11,13,15]},{'le',2});
      S.addarg(rand(noq,npq),{'type','double'});
      S.addarg(rand( nD,  1),{'type','double'});
      S.addarg(rand( nD,  1),{'type','double','numel',[]});
      S.addarg(rand(nzx,nrx),{'type','double'});
      S.addarg(rand(npq,  1),{'type','double','numel',[]});
      S.addarg(rand(noq,  1),{'type','double','numel',[]});
      S.addarg(rand(noq,  1),{'type','double','numel',[]});
      S.addarg(randi(nD,nzy,nry,'int8'),{'type','int8','numel',[]});
      S.addarg(rand(nD, 1),{'type','double','numel',[]});
      S.addarg(randi(nD, 1, 1,'int8'),{'type','int8','numel',[]});
      S.addarg(2,{'type','numeric','scalar',[]});
    end
    
    function S = rtcimex_13_args
      noq = 32; nD = 2;
      S = meq_mexargs_test.rtcimex_args;
      S.nargout.check = {'le',3};
      S.addarg(rand( nD,  1),{'type','double','numel',[]});
      S.addarg(rand(noq,  1),{'type','double','numel',[]});
    end
    
    function S = rtcimex_15_args
      noq = 32; nD = 2;
      S = meq_mexargs_test.rtcimex_13_args;
      S.nargout.check = {'le',5};
      S.addarg(rand( nD, 1),{'type','double','numel',[]});
      S.addarg(rand(noq, 1),{'type','double','numel',[]});
    end
    
    function S = shapmex_args
      noq = 32; npq = 18;
      S = mexargs('shapmex',{'eq',10},{'le',16});
      S.addarg(rand(noq,npq),{'type','double'});
      S.addarg(rand(noq,npq),{'type','double','numel',[]});
      S.addarg(rand,{'scalar_or_empty',[]});
      S.addarg(rand,{'numel',[]});
      S.addarg(rand,{'type','numeric','scalar',[]});
      S.addarg(rand,{'type','numeric','scalar',[]});
      S.addarg(rand,{'type','numeric','scalar',[]});
      S.addarg(rand,{'type','numeric','scalar',[]});
    end
    
    function S = uatamex_args
      m = 12; n = 8;
      S = mexargs('uatamex',{'eq',1},{'le',1});
      S.addarg(rand(m,n),{'type','real'});
    end
    
    function S = vizrmex_args
      nzx = 65; nrx = 31;
      ri =  1; ro =  2;
      zl = -1; zu =  1;
      nzy = nzx-2; nry = nrx-2;
      S = mexargs('vizrmex',{'eq',7},{'le',3});
      S.addarg(rand(nzx,nrx),{'type','double'});
      S.addarg(int8(rand(nzy,nry)>0.5),{'type','int8'});
      S.addarg(linspace(ri,ro,nry).',{'type','double','numel',[]});
      S.addarg(1./linspace(ri,ro,nry).',{'type','double','numel',[]});
      S.addarg(1,{'type','numeric','scalar',[]});
      S.addarg((ro-ri)/(nrx-1),{'type','numeric','scalar',[]});
      S.addarg((zu-zl)/(nzx-1),{'type','numeric','scalar',[]});
    end
  end
end
