classdef meqFx_jacobian_test < meq_jacobian_test
  % Test meqFxjac computation of jacobian of meqFx
  %
  % [+MEQ MatlabEQuilibrium Toolbox+]

  %    Copyright 2022-2025 Swiss Plasma Center EPFL
  %
  %   Licensed under the Apache License, Version 2.0 (the "License");
  %   you may not use this file except in compliance with the License.
  %   You may obtain a copy of the License at
  %
  %       http://www.apache.org/licenses/LICENSE-2.0
  %
  %   Unless required by applicable law or agreed to in writing, software
  %   distributed under the License is distributed on an "AS IS" BASIS,
  %   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  %   See the License for the specific language governing permissions and
  %   limitations under the License.

  properties
    ntests = 20;
    tolFx = 1e-10;
    data
  end

  properties(ClassSetupParameter)
    gsxe = {'1','2','3'};
  end

  methods(TestClassSetup)
    function setup_L(testCase,gsxe)
      testCase.data.L = fgs('ana',1,[],'gsxe',str2double(gsxe));
    end
  end

  methods(Test,TestTags={'Jacobian'})
    function test_meqFxjac(testCase)
      % Test that meqFx solution corresponds to linear combination based on derivatives computed in meqFxjac
      L = testCase.data.L;

      % Compute jacobian
      [Mxy,Mxe] = meqFxjac(L);

      % Test on random inputs
      for ii = 1:testCase.ntests
        Iy = rand(L.nzy,L.nry)*5e5/L.ny;
        Ie = rand(L.ne,1)*1e4;

        Fx_ = meqFx(L,Iy,Ie);
        Fx  = reshape(Mxy*Iy(:)+Mxe*Ie,L.nzx,L.nrx);

        testCase.verifyEqual(Fx,Fx_,'AbsTol',testCase.tolFx*max(abs(Fx_(:))),sprintf('Solution of meqFx differs from linear estimate (gsxe=%d)',L.P.gsxe));
      end
    end

    function test_meqFxjac_fd(testCase)
      % Test that outputs of meqFxjac match finite difference derivatives
      L = testCase.data.L;

      % Compute jacobian
      [Mxy0,Mxe0] = meqFxjac(L);

      % Compute finite difference estimate
      Iy0 = zeros(L.nzy,L.nry);
      Ie0 = zeros(L.ne,1);
      x0 = {L,Iy0,Ie0};
      Mxy1 = jacfd(@meqFx,x0,'F0',{zeros(L.nzx,L.nrx)},'szF0',{L.nx},'iargin',2);
      Mxe1 = jacfd(@meqFx,x0,'F0',{zeros(L.nzx,L.nrx)},'szF0',{L.nx},'iargin',3);

      % Compare them
      testCase.verifyEqual(Mxy0,Mxy1,'AbsTol',testCase.tolFx*max(abs(Mxy1(:))),sprintf('Mxy from meqFxjac differ from finite difference estimate for gsxe=%d',L.P.gsxe));
      testCase.verifyEqual(Mxe0,Mxe1,'AbsTol',testCase.tolFx*max(abs(Mxe1(:))),sprintf('Mxe from meqFxjac differ from finite difference estimate for gsxe=%d',L.P.gsxe));
    end

    % Since meqFx is linear in its inputs, it is pointless to do a
    % convergence study
  end

end
