classdef liu_Newton_vs_Picard_test < meq_test
  % Compare Liuqe solution with Picard iteration vs Newton wrapper on liut for singlet and 1 case of droplet with stable Picard
  %
  % [+MEQ MatlabEQuilibrium Toolbox+]

  %    Copyright 2022-2025 Swiss Plasma Center EPFL
  %
  %   Licensed under the Apache License, Version 2.0 (the "License");
  %   you may not use this file except in compliance with the License.
  %   You may obtain a copy of the License at
  %
  %       http://www.apache.org/licenses/LICENSE-2.0
  %
  %   Unless required by applicable law or agreed to in writing, software
  %   distributed under the License is distributed on an "AS IS" BASIS,
  %   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  %   See the License for the specific language governing permissions and
  %   limitations under the License.

  properties (TestParameter)
    tok      = struct('TCV_diverted','tcv','TCV_limited_NT','tcv','TCV_droplet_Picard_stable','tcv');
    shot     = struct('TCV_diverted',69185,'TCV_limited_NT',69179,'TCV_droplet_Picard_stable',68466);
    t        = struct('TCV_diverted',  1.1,'TCV_limited_NT',    1,'TCV_droplet_Picard_stable', 2.43);
    idoublet = struct('TCV_diverted',false,'TCV_limited_NT',false,'TCV_droplet_Picard_stable', true);
  end

  properties
    verbosity = 0;
    psichco = 1e-11; % High convergence request for Picard iteration
    tol = 5e-7; % This should converge to 1e-14 asking for high enough convergence for both Picard and Newton
    toldzA = 1e-2; % tolerance [m] zA LIUQE Picard -  zA LIUQE Newton with dz = 0;
    % The convergence test in Picard is on zFx - Fx , while in Newton on Iy - tildeI(Iy, Ia, Ie, Bm, Ff, ....)
    % The two threshold are therefore not directly comparable
  end

  methods(Test,TestTags={'TCV'},ParameterCombination='sequential')
    function no_dz_stabilization(testCase,tok,shot,t, idoublet)
      % Compare the solution of Picard with stabilization and Newton with no dz stabilization

      PPcommon = {'debug',testCase.verbosity,'gsxe',2,'idoublet',idoublet,...
        'izgrid',true,'stabz',true,'psichco',testCase.psichco,'tolstep',0};

      %% Picard
      inPar = [{'useSQP', false, ...
        'wdz', [0;0;Inf], ... % Disable dz in the mantle (no current there yet)
        'iters', 0, ... % For direct comparison liut Picard vs Newton iters = 0 for liut Picard
        'itera', 500,...
        'iterfrz', Inf},...
        PPcommon];
      if idoublet
        % Get default parameter multiple domains and overload specific requests
        inPar = liup_doublet(inPar{:});
      end
      [LliuP,LXliuP,LYliuP] = liu(tok, shot,t, inPar{:});

      %% Newton
      inPar = [{'useSQP',true},PPcommon];
      if idoublet
        % Get default parameter multiple domains and overload specific requests
        inPar = liup_doublet(inPar{:});
      end
      LliuN = liu(tok, shot,t, inPar{:});
      LYliuN = liut(LliuN,LXliuP);

      %% Check solution
      testCase.assertTrue(meq_test.check_convergence(LliuP,LXliuP.t,LYliuP) ,'LIUQE Picard did not converge')
      testCase.assertTrue(meq_test.check_convergence(LliuN,LXliuP.t,LYliuN) ,'LIUQE Newton did not converge')
      testCase.assertEqual(LYliuN.dz,zeros(LliuN.ndz,1),'LIUQE Newton LY.dz~=0 while dz=0 requested')

      % Check that the solution of LIUQE Picard zA ~ LIUQE Newton zA .
      % One does not expect LIUQE Picard zA - LIUQE Newton zA = dz, since LIUQE Picard is not a proper solution of the GS.
      % See F.Carpanese PhD Thesis 2020 section 4.5.1.
      testCase.verifyEqual(LYliuN.zA(1),LYliuP.zA(1),'AbsTol',testCase.toldzA ,...
        'Significant difference between solution LIUQE Picard and LIUQE Newton')
      if testCase.verbosity
        clf;  figure; meqplotfancy(LliuN,LYliuN); figure; meqplotfancy(LliuP,LYliuP);
      end

      %% FGS residual based on Newton solution
      [Lfgs] = fgs(tok,shot,[],PPcommon{:},'agcon',{'Ip','Wk','qA'},'selu','sx',...
        'icsint',LliuN.P.icsint,'ilim',LliuN.P.ilim); % ensure same icsint/ilim);
      LXfgs = meqxconvert(LliuN,LYliuN,Lfgs);
      x = Lfgs.LX2x(LXfgs);
      LYp = LXfgs;
      res = fgeF(x,Lfgs,LXfgs,LYp);
      testCase.verifyLessThan(norm(res),testCase.psichco,'LIUQE Newton solution has high FGS residual');
    end

  end
end
