classdef lih2obs_test < meq_test
  % Tests for lih2obs, lih2shapobs, lih2qshapobs
  %
  % [+MEQ MatlabEQuilibrium Toolbox+]

  %    Copyright 2022-2025 Swiss Plasma Center EPFL
  %
  %   Licensed under the Apache License, Version 2.0 (the "License");
  %   you may not use this file except in compliance with the License.
  %   You may obtain a copy of the License at
  %
  %       http://www.apache.org/licenses/LICENSE-2.0
  %
  %   Unless required by applicable law or agreed to in writing, software
  %   distributed under the License is distributed on an "AS IS" BASIS,
  %   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  %   See the License for the specific language governing permissions and
  %   limitations under the License.

  properties
    tol = 100*eps();
    L_; LX_; LY_; rc_; zc_; vrc_; vzc_;
  end

  properties(ClassSetupParameter)
    tok            = struct('ana_singlet', 'ana','ana_singlet_no_Ip', 'ana','ana_doublet','ana','tcv_singlet','tcv','tcv_doublet', 'tcv')
    shot           = struct('ana_singlet',     2,'ana_singlet_no_Ip',     2,'ana_doublet',   82,'tcv_singlet',61400,'tcv_doublet', 55331)
    t              = struct('ana_singlet',     0,'ana_singlet_no_Ip',     0,'ana_doublet',   0 ,'tcv_singlet',    1,'tcv_doublet',  0.02)
    doublet        = struct('ana_singlet', false,'ana_singlet_no_Ip', false,'ana_doublet', true,'tcv_singlet',false,'tcv_doublet',  true)
    Ie_constrained = struct('ana_singlet','none','ana_singlet_no_Ip','none','ana_doublet','one','tcv_singlet','all','tcv_doublet','none')
    ivesm          = struct('ana_singlet',  true,'ana_singlet_no_Ip',  true,'ana_doublet', true,'tcv_singlet',false,'tcv_doublet', false)
    Ipmeas         = struct('ana_singlet',  true,'ana_singlet_no_Ip', false,'ana_doublet', true,'tcv_singlet', true,'tcv_doublet',  true)
  end
  
  methods(TestClassSetup, ParameterCombination = 'sequential')
    function setup_LLX(testCase,tok,shot,t,doublet,Ie_constrained,ivesm,Ipmeas)
      testCase.assumeTrue(meq_test.check_tok(tok))
      
      if doublet, ndom=3; else, ndom=1; end

      L = lih(tok,shot,t); % get geometry
      wIa = ones(L.G.na,1); % default weights
      wIu = ones(L.G.nu,1); % default weights

      switch Ie_constrained
        case 'none' % do nothing
        case 'one'
          wIa(3) = Inf; % constrain one Ie
        case 'all'
          wIa(:) = Inf; wIu(:) = Inf; % constrain all Ie
      end

      % run lih
      [L,LX] = lih(tok,shot,t,...
        'wIa',wIa,'wIu',wIu,...
        'idoublet',doublet,'ndom',ndom,...
        'iterh',0,'Ipmeas',true,'ivesm',ivesm,'infct',@qintmex);
      
      % use Ip estimator instead of true Ip for no-Ipmeas test
      if ~Ipmeas
        LX.Ip = [L.G.Ipm,L.Ipa,L.P.ivesm*L.Ipu] * [LX.Bm; LX.Ia; LX.Iu];
      end
      % run LIH
      testCase.LX_ = LX;
      testCase.LY_ = liht(L,LX);
      
      % reinit L with correct Ipmeas setting
      [testCase.L_] = lih(tok,shot,t,...
        'wIa',wIa,'wIu',wIu,...
        'idoublet',doublet,'ndom',ndom,...
        'iterh',0,'Ipmeas',Ipmeas,'ivesm',ivesm,'infct',@qintmex);
      
      % pick contour points to check shape observers on
      nc = 12;
      % parameters for elipse inside center
      a = 0.25 * (max(L.G.rl) - min(L.G.rl));
      b = 0.25 * (max(L.G.zl) - min(L.G.zl));
      r0 = 0.5 * (max(L.G.rl) + min(L.G.rl));
      z0 = 0.5 * (max(L.G.zl) + min(L.G.zl));
      
      th = linspace(0, 2*pi, nc+1); th = th(1:end-1)';
      testCase.rc_ = r0 + a * cos(th);
      testCase.zc_ = z0 + b * sin(th);
      testCase.vrc_ = cos(th);
      testCase.vzc_ = sin(th);
    end
  end
  
  

  methods(Test, TestTags={'Integration'})
    function test_lih2obs_vs_lih(testCase)
      % get L, LX, LY
      L = testCase.L_;
      LX = testCase.LX_;
      LY = testCase.LY_;
      
      % run lih2obs
      [ Ief, Iem, Iea, Ieu, Iep,...
        Ipf, Ipm, Ipa, Ipu, Ipp,...
       zIpf,zIpm,zIpa,zIpu,zIpp,...
       rIpf,rIpm,rIpa,rIpu,rIpp] = lih2obs(L);

      % Check
      Xr = [LX.Ff; LX.Bm]; % magnetic measurements
      if L.P.ivesm
        Xe = [LX.Ia; LX.Iu]; % external currents
      else
        Xe = LX.Ia;
      end
      % integral measurements (nb, Xt is excluded if ~Ipmeas)
      if L.P.Ipmeas
        Xi = [LX.Ip];
        Xd = [Xr;Xe;Xi];
      else
        Xd = [Xr;Xe];
      end

      Ie   = [Ief,Iem,Iea,Ieu,Iep] * Xd;
      IpD  = [Ipf,Ipm,Ipa,Ipu,Ipp] * Xd;
      zIpD = [zIpf,zIpm,zIpa,zIpu,zIpp] * Xd;
      rIpD = [rIpf,rIpm,rIpa,rIpu,rIpp] * Xd;

      Ie_expected   = [LY.Ia;LY.Iu];
      IpD_expected  = LY.IpD;
      zIpD_expected = LY.zIpD;
      rIpD_expected = LY.rIpD;

      testCase.verifyEqual(  Ie,  Ie_expected, 'AbsTol',testCase.tol*max([L.Ia0;L.Iu0]));
      testCase.verifyEqual( IpD, IpD_expected, 'AbsTol',testCase.tol*L.Ip0);
      testCase.verifyEqual(rIpD,rIpD_expected, 'AbsTol',testCase.tol*L.Ip0);
      testCase.verifyEqual(zIpD,zIpD_expected, 'AbsTol',testCase.tol*L.Ip0);
    end
    
    function test_lih2shapobs_vs_lih(testCase)
      L = testCase.L_;
      LX = testCase.LX_;
      LY = testCase.LY_;
      rc = testCase.rc_; zc = testCase.zc_;
      
      [ Mnf,  Mnm,  Mna,  Mnu,  Mnp,...
       Brnf, Brnm, Brna, Brnu, Brnp,...
       Bznf, Bznm, Bzna, Bznu, Bznp] = lih2shapobs(L,rc,zc);

      % Check
      Xr = [LX.Ff; LX.Bm]; % magnetic measurements
      if L.P.ivesm
        Xe = [LX.Ia; LX.Iu]; % external currents
      else
        Xe = LX.Ia;
      end
      % integral measurements (nb, Xt is excluded if ~Ipmeas)
      if L.P.Ipmeas
        Xi = [LX.Ip];
        Xd = [Xr;Xe;Xi];
      else
        Xd = [Xr;Xe];
      end

      Fn  = [ Mnf,  Mnm,  Mna,  Mnu,  Mnp] * Xd;
      Brn = [Brnf, Brnm, Brna, Brnu, Brnp] * Xd;
      Bzn = [Bznf, Bznm, Bzna, Bznu, Bznp] * Xd;
      
      [Fn_expected,Brn_expected,Bzn_expected] = L.P.infct(L.rx,L.zx,LY.Fx,rc,zc,L.inM);
      
      % derivative tolerances are scaled by 1/h
      testCase.verifyEqual( Fn, Fn_expected, 'AbsTol',testCase.tol);
      testCase.verifyEqual(Brn,Brn_expected, 'AbsTol',testCase.tol * sqrt(L.idsx));
      testCase.verifyEqual(Bzn,Bzn_expected, 'AbsTol',testCase.tol * sqrt(L.idsx));
    end
    
    function test_lih2qshapobs_vs_lih(testCase)
      L = testCase.L_;
      LX = testCase.LX_;
      LY = testCase.LY_;
      rc = testCase.rc_; zc = testCase.zc_;
      vrc = testCase.vrc_; vzc = testCase.vzc_; 
      
      [   Mnf,    Mnm,    Mna,    Mnu,    Mnp,...
        dvMnf,  dvMnm,  dvMna,  dvMnu,  dvMnp,...
       dv2Mnf, dv2Mnm, dv2Mna, dv2Mnu, dv2Mnp] = lih2qshapobs(L,rc,zc,vrc,vzc);
     
      % Check
      Xr = [LX.Ff; LX.Bm]; % magnetic measurements
      if L.P.ivesm
        Xe = [LX.Ia; LX.Iu]; % external currents
      else
        Xe = LX.Ia;
      end
      % integral measurements (nb, Xt is excluded if ~Ipmeas)
      if L.P.Ipmeas
        Xi = [LX.Ip];
        Xd = [Xr;Xe;Xi];
      else
        Xd = [Xr;Xe];
      end

      Fn    = [   Mnf,    Mnm,    Mna,    Mnu,    Mnp] * Xd;
      dvFn  = [ dvMnf,  dvMnm,  dvMna,  dvMnu,  dvMnp] * Xd;
      dv2Fn = [dv2Mnf, dv2Mnm, dv2Mna, dv2Mnu, dv2Mnp] * Xd;
      
      [Fn_expected,Brn,Bzn,~,Brzn,Bzrn,Bzzn] = L.P.infct(L.rx,L.zx,LY.Fx,rc,zc,L.inM);
      % convert magnetic fields into flux derivatives
      drMn =   2 * pi * rc .* Bzn;
      dzMn =  -2 * pi * rc .* Brn;
      drrMn =  2 * pi * rc .* Bzrn + 2 * pi .* Bzn;
      drzMn =  2 * pi * rc .* Bzzn;
      dzzMn = -2 * pi * rc .* Brzn;
      
      % compute derivatives along [vrc, vzc]
      dvFn_expected  = drMn .* vrc + dzMn .* vzc;
      dv2Fn_expected = drrMn .* vrc .* vrc + 2 * drzMn .* vrc .* vzc + dzzMn .* vzc .* vzc;
      
      % derivative tolerances are scaled by 1/h
      testCase.verifyEqual(   Fn,   Fn_expected, 'AbsTol',testCase.tol);
      testCase.verifyEqual( dvFn, dvFn_expected, 'AbsTol',testCase.tol * sqrt(L.idsx));
      testCase.verifyEqual(dv2Fn,dv2Fn_expected, 'AbsTol',testCase.tol * L.idsx);
    end
  end
end
