classdef jacfd_test < meq_test
  % Test of Generic finite difference jacobian evaluation
  %
  % [+MEQ MatlabEQuilibrium Toolbox+]

  %    Copyright 2022-2025 Swiss Plasma Center EPFL
  %
  %   Licensed under the Apache License, Version 2.0 (the "License");
  %   you may not use this file except in compliance with the License.
  %   You may obtain a copy of the License at
  %
  %       http://www.apache.org/licenses/LICENSE-2.0
  %
  %   Unless required by applicable law or agreed to in writing, software
  %   distributed under the License is distributed on an "AS IS" BASIS,
  %   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  %   See the License for the specific language governing permissions and
  %   limitations under the License.

  properties
    paravail
  end

  properties (TestParameter)
    workers = {1,2}; % Enables testing of parfor block
  end

  methods (TestClassSetup)
    function setupParallelPool(testCase)

      % check if parallel toolbox is installed, parallel replaced distcomp since R2019b
      testCase.paravail = ~isempty(ver('parallel')) || ~isempty(ver('distcomp')); %#ok<DCRENAME>

      if testCase.paravail && isempty(gcp('nocreate'))
        % If no pool is already available, delete the created one at the end of these tests
        testCase.addTeardown(@() delete(gcp('nocreate')));
      end
    end
  end

  methods (Test, TestTags={'Unit'})
    function test_jacfd_pertgroups(testCase,workers)
      % Test pertgroups option of jacfd function
      testCase.assumeTrue(workers == 1 || testCase.paravail, 'Skipping parallel tests since parallel toolbox was not found');

      n = 201;
      x = linspace(0,2*pi,n).';
      f = @sin;

      % Compute analytical jacobian
      J = cos(x);

      % Compute full numerical jacobian
      J_fd_ = jacfd(f,{x},'szF0',{n},'workers',workers);
      testCase.verifySize(J_fd_,[n,n]);

      % Compute numerical jacobian perturbing all x values at once
      J_fd = jacfd(f,{x},'szF0',{n},'pertgroups',{1:n});
      testCase.verifySize(J_fd,[n,1]);

      testCase.verifyEqual(J_fd,J,'AbsTol',sqrt(eps),'Analytical and numerical jacobians do not match');
      testCase.verifyEqual(diag(J_fd_),J_fd,'diagonal of full jacobian does not match reduced jacobian');

    end
  end

end
