classdef interpolators_test < meq_test
 % Test various flux/field interpolators in liut.m
 %
 % [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

 properties
  tolF = 5e-3; % error tolerance on F interpolation
  infct; % interpolator function handle
  inp; % interpolator parameters
  L,LX; % LIUQE structures
 end
 
 properties(TestParameter)
   rn = struct('empty',[],'three',[1;1;1])
   zn = struct('empty',[],'three',[-0.2;0;0.2]);
 end
 
 properties(MethodSetupParameter)
   % interpolator: function handle for interpolator
   % intp: parameter structure for interpolator
   interpolator = struct('none',[],'quadratic_6p',    'qintmex','quadratic_9p','qintmex'    );
   interpparams = struct('none',[],'quadratic_6p',struct('n',6),'quadratic_9p',struct('n',9));
 end
 
 methods(TestClassSetup)
   function setup_LLX(testCase)
     testCase.assumeFalse(isempty(which('mdsopen')),'Skip test since no MDS found')
     tok = 'ana'; shot = 1;
     [testCase.L,testCase.LX] = liu(tok,shot);
   end
 end
 
 methods(TestMethodSetup)
   
   function set_infct(testCase,interpolator,interpparams)
     % sets up function handle for interpolator to be tested
     if ~isempty(interpolator)
       testCase.infct = str2func(interpolator);
     else
       testCase.infct = []; % also test 'empty' case (no interpolation)
     end
     testCase.inp = interpparams;
   end
 end
 
 methods(Test,TestTags={'Unit'}, ParameterCombination='sequential' )
   
  function test_liuqe(testCase,rn,zn)
   
   L = testCase.L; %#ok<*PROPLC>
   LX = testCase.LX;

   % recompute consolidation with this intepolator
   L = liuc(L.P,L.G,...
     'infct',testCase.infct,'inp',testCase.inp,...
     'rn',rn,'zn',zn); 
   
   % assign on which ri,zi to give outputs
   LY = liut(L,LX);
   
   if isempty(testCase.infct) || isempty(rn)
     nt = numel(LY.t);
     testCase.assertSize(LY.Fn ,[0,nt], 'Fn should be empty')
     testCase.assertSize(LY.Brn,[0,nt],'Brn should be empty')
     testCase.assertSize(LY.Bzn,[0,nt],'Bzn should be empty')
   else
     Fxi = interp2(L.rx,L.zx,LY.Fx,L.P.rn,L.P.zn); % matlab interpolation
     testCase.assertEqual(LY.Fn,Fxi,'RelTol',testCase.tolF)
   end
  end
 end
end
