classdef (SharedTestFixtures={mexm_fixture}) integrals_test < meq_test
  % Tests for surface and volume integrals
  %
  % [+MEQ MatlabEQuilibrium Toolbox+]

  %    Copyright 2022-2025 Swiss Plasma Center EPFL
  %
  %   Licensed under the Apache License, Version 2.0 (the "License");
  %   you may not use this file except in compliance with the License.
  %   You may obtain a copy of the License at
  %
  %       http://www.apache.org/licenses/LICENSE-2.0
  %
  %   Unless required by applicable law or agreed to in writing, software
  %   distributed under the License is distributed on an "AS IS" BASIS,
  %   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  %   See the License for the specific language governing permissions and
  %   limitations under the License.

  properties
    tol = 1e-12;
  end
  
  properties (TestParameter)
    shot = struct('diverted',2,'droplet',81,'doublet',82);
    sIp = {-1,1};
  end
  
  methods (Test,TestTags={'Unit'})
    
    function test_meqintQ(testCase,sIp)
      %%
      L = fbt('ana',1,[],'pql',0.1);
      
      %% Fake circular equilibrium
      r0 = 1.001; z0 = 0;
      FA = sIp; FB = 0;
      rBt = 1; bp=1; qA = 1.5;
      [L,LY] = testCase.getCircularEquilibrium(L,r0,z0,FA,FB,rBt);
      LY.qA = qA;
      LY.Wk = bp*(1.5e-7*pi*L.P.r0*LY.Ip.^2);
      
      % plasma current from basis function coefficients
      [~,TpDg,ITpDg] = L.bfct(1,L.bfp,LY.Fx,LY.FA,LY.FB,LY.Opy,L.ry,L.iry);
      
      %% Coefficient scaling
      ag = zeros(L.ng,1);
      [res, ~,~,dresdag] = ...
        meqagcon(L,LY,LY.FA,LY.FB,LY.rA,LY.dr2FA,LY.dz2FA,LY.drzFA,ag,LY.Fx,LY.Opy,TpDg,ITpDg);
      ag = ag-dresdag\res;
      
      [IpQ,WkQ,WpQ,VpQ,FtQ,Ft0Q] = ...
        meqintQ(L,FA,FB,rBt,ag, LY.Fx, LY.Opy);
      
      [Ip,Wk,Wp,~,~,~,Vp,Ft,Ft0] = ...
        meqint(L.fPg,L.fTg,TpDg,ITpDg,ag,LY.Fx,LY.Opy,L,rBt);
      
      tol_ = testCase.tol;
      testCase.verifyEqual(Ip, IpQ(end), 'RelTol', tol_, 'IpQ final value not equal to Ip');
      testCase.verifyEqual(Wk, WkQ(end), 'RelTol', tol_, 'WkQ final value not equal to Wk');
      testCase.verifyEqual(Wp, WpQ(end), 'RelTol', tol_, 'WpQ final value not equal to Wp');
      testCase.verifyEqual(Vp, VpQ(end), 'RelTol', tol_, 'VpQ final value not equal to Vp');
      testCase.verifyEqual(Ft, FtQ(end), 'RelTol', tol_, 'FtQ final value not equal to Ft');
      testCase.verifyEqual(Ft0,Ft0Q(end),'RelTol', tol_, 'Ft0Q final value not equal to Ft0');
      
      % check expected volume dependency
      a = sqrt((LY.rA-LY.rB).^2 + (LY.zA-LY.zB).^2);
      V = L.pQ'.^2 * (2*pi^2*r0*a.^2);
      testCase.verifyEqual(V/Vp,VpQ/Vp,'AbsTol',1e-2,'Unexpected volume result for analytical case');
    end
    
    function test_meqpost_vs_meqint(testCase,shot)
      % Use directly FBT simulation and results from meqpost
      [L,~,LY] = fbt('ana',shot,[]);

      [~,TpDg,ITpDg] = L.bfct(1,L.bfp,LY.Fx,LY.F0,LY.F1,LY.Opy,L.ry,L.iry);
      [Ip,Wk,Wp,Wt,Wt0,WN,Vp,Ft,Ft0,bp,bt,mu,li,bpli2] = meqint(L.fPg,L.fTg,TpDg,ITpDg,LY.ag,LY.Fx,LY.Opy,L,LY.rBt);
      
      tol_ = testCase.tol;
      testCase.verifyEqual(LY.Ip   , Ip   , 'RelTol', tol_, 'Ip value does not match meqint');
      testCase.verifyEqual(LY.Wk   , Wk   , 'RelTol', tol_, 'Wk value does not match meqint');
      testCase.verifyEqual(LY.Wp   , Wp   , 'RelTol', tol_, 'Wp value does not match meqint');
      testCase.verifyEqual(LY.Wt   , Wt   , 'RelTol', tol_, 'Wt value does not match meqint');
      testCase.verifyEqual(LY.Wt0  , Wt0  , 'RelTol', tol_, 'Wt0 value does not match meqint');
      testCase.verifyEqual(LY.WN   , WN   , 'RelTol', tol_, 'WN value does not match meqint');
      testCase.verifyEqual(LY.Vp   , Vp   , 'RelTol', tol_, 'Vp value does not match meqint');
      testCase.verifyEqual(LY.Ft   , Ft   , 'RelTol', tol_, 'Ft value does not match meqint');
      testCase.verifyEqual(LY.Ft0  , Ft0  , 'RelTol', tol_, 'Ft0 value does not match meqint');
      testCase.verifyEqual(LY.bp   , bp   , 'RelTol', tol_, 'bp value does not match meqint');
      testCase.verifyEqual(LY.bt   , bt   , 'RelTol', tol_, 'bt value does not match meqint');
      testCase.verifyEqual(LY.mu   , mu   , 'RelTol', tol_, 'mu value does not match meqint');
      testCase.verifyEqual(LY.li   , li   , 'RelTol', tol_, 'li value does not match meqint');
      testCase.verifyEqual(LY.bpli2, bpli2, 'RelTol', tol_, 'bpli2 value does not match meqint');
    end
  end
end
