classdef gaps_test < meq_test
  % Tests for gaps calculations
  %
  % [+MEQ MatlabEQuilibrium Toolbox+]

  %    Copyright 2022-2025 Swiss Plasma Center EPFL
  %
  %   Licensed under the Apache License, Version 2.0 (the "License");
  %   you may not use this file except in compliance with the License.
  %   You may obtain a copy of the License at
  %
  %       http://www.apache.org/licenses/LICENSE-2.0
  %
  %   Unless required by applicable law or agreed to in writing, software
  %   distributed under the License is distributed on an "AS IS" BASIS,
  %   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  %   See the License for the specific language governing permissions and
  %   limitations under the License.

  properties
    tok = 'ana';
    t = 0;
    verbose = 0;
    iterq = 100; % Some targets are quite far from the origin 
    tol = 1e-3;
    L,LX;
  end

  properties(ClassSetupParameter)
    config = {'limited','diverted','secondaryX','limited-secondaryX'};
    sIp = {+1,-1}; % Ip sign
  end
  
  properties(TestParameter)
    nFW = {0,1,2,3};
    icsint = {false,true};
  end
  
  methods(TestClassSetup)
    function setup_LLX(testCase,config,sIp)
      switch config
        case 'limited'
          shot = 1;
        case 'diverted'
          shot = 2;
        case 'secondaryX'
          shot = 3;
        case 'limited-secondaryX'
          shot = 6;
        otherwise
          error('undefined config %s\n',config)
      end
      tok = testCase.tok; t=testCase.t;
      
      %FBT
      [Lfbt,LXfbt] = fbt(tok,shot,t,...
        'debug',testCase.verbose);
      LXfbt.Ip  = LXfbt.Ip *sIp; % possible Ip sign flip
      if Lfbt.nD>1
        LXfbt.IpD = LXfbt.IpD*sIp; % possible Ip sign flip
      end
      LXfbt.qA  = LXfbt.qA *sIp; % flip qA for consistency
      LXfbt = fbtx(Lfbt,LXfbt); % Necessary after manual LX update
      LYfbt = fbtt(Lfbt,LXfbt);

      testCase.assertTrue(meq_test.check_convergence(Lfbt,LXfbt.t,LYfbt),'FBT did not converge')
      
      L = liu(tok,shot,t,...
        'iterq',testCase.iterq,...
        'npq',11,...
        'nW',16);
      
      LX = liux(L,meqxconvert(Lfbt,LYfbt,L));
      testCase.L  = L;
      testCase.LX = LX;
      
      if testCase.verbose
        clf;
        meqplott(Lfbt,LYfbt);
        drawnow;
      end
    end
  end
  
  methods(Test,TestTags={'Unit'})
    function test_gaps(testCase,nFW,icsint)
      
      %% LIUQE
      L = testCase.L; %#ok<*PROPLC>
      LX = testCase.LX;
      % Test with given amount of fluxes to track
      L.P.nFW = nFW;
      L.P.icsint = icsint;
      if icsint
        L.P.ilim=3;
      end
      L = liuc(L.P,L.G);
      LY = liut(L,LX,'debug',testCase.verbose);
      
       %% Optional plotting
      if testCase.verbose
        np = ceil(numel(testCase.nFW)/2);
        subplot(2,np,nFW+1)
        meqplott(L,LY);
        [~,~,~,~,~,~,~,~,FX,~,~,~,...
          ~,~,FB,~,~,~] = meqpdom(LY.Fx,LY.Ip,L.P.isaddl,L);
        hold on;
        if numel(FX)==1, F=FX*[1 1]; else, F=FX; end
        contour(L.rx,L.zx,LY.Fx,F,'w:','linewidth',1.5)
        contour(L.rx,L.zx,LY.Fx,[FB FB],'k');
        title(sprintf('%d surfaces',nFW));
        drawnow
      end
      
      % Verifications
      testCase.assertNotEmpty(LY,'Returned empty LY');

      testCase.verifySize(LY.aW,[L.G.nW,nFW],sprintf('aW has unexpected size [%d,%d]',size(LY.aW)))
      
      if nFW>0 
        
        r = L.G.rW-LY.aW.*cos(L.G.oW);
        z = L.G.zW+LY.aW.*sin(L.G.oW);
        
        % interpolate fluxes on these points
        [rrx,zzx] = meshgrid(L.rx,L.zx);
        if icsint
          % Compute spline interpolation
          rk = L.taur;
          zk = L.tauz;
          c = L.Mr*(L.Mz*LY.Fx).';
          Fi = cs2devalmex(rk,zk,c,r,z,0);
        else
          % Compute linear interpolation
          Fi = interp2(rrx,zzx,LY.Fx,r,z);
        end
        testCase.verifyEqual(Fi(:,1),repmat(LY.FW(1),L.G.nW,1),'AbsTol',1e-8*abs(LY.FA-LY.FB)); % Actual separatrix
        testCase.verifyEqual(Fi     ,repmat(LY.FW   ,L.G.nW,1),'AbsTol',1e-8*abs(LY.FA-LY.FB),...
          sprintf('Secondary separatrices were not reached for all gap specs\n\tsuccess fraction: [%s]/%d',...
          num2str(sum(abs(Fi - LY.FW )<1e-8*abs(LY.FA-LY.FB),1),'%d '),L.G.nW));
      end
    end
    
    function test_strikes(testCase)
      
      L = testCase.L; LX = testCase.LX; %#ok<*PROP>
      testCase.assumeTrue(L.P.shot == 2,'ignore test except for ''diverted'' config (shot=2)')
      
      %%
      PP=[fieldnames(L.P),struct2cell(L.P)]';
      L=liu(testCase.tok,L.P.shot,[],PP{:},...
        'aW',[  0.3  ; 1    ],...
        'rW',[  0.73 ; 1.24 ],...
        'zW',[ -0.4  ;-0.42 ],...
        'oW',[pi;0]);
      
      LY = liut(L,LX,'debug',testCase.verbose);
      if testCase.verbose
        clf;
        meqplott(L,LY);
        drawnow;
      end
      
      r = L.G.rW-LY.aW.*cos(L.G.oW);
      z = L.G.zW+LY.aW.*sin(L.G.oW);
      
      % interpolate fluxes on these points
      [rrx,zzx] = meshgrid(L.rx,L.zx);
      [Fi] = interp2(rrx,zzx,LY.Fx,r,z);
      testCase.verifyEqual(Fi,repmat(LY.FW,L.G.nW,1),'AbsTol',1e-8*abs(LY.FA-LY.FB));
      
    end

    function test_limits(testCase)
     % test that limits in aW are upheld
     L = testCase.L; LX = testCase.LX; %#ok<*PROP>
     L.G.aW(:) = 0.1;

     L = liuc(L.P,L.G);
     LY = liut(L,LX,'debugplot',1); %testCase.verbose);
     testCase.verifyLessThanOrEqual(LY.aW, L.G.aW); % check maximum
     testCase.verifyGreaterThanOrEqual(LY.aW,0); % check minimum
    end
  end
end
