classdef fsgi_mantle_test < meq_test
  % Comparison tests between fsgi algorithm for mantle and axis domains
  % The fsgi algorithm for mantle domains uses different expressions for
  % flux surface integrals and should give similar results when applied to
  % axis domains
  %
  % [+MEQ MatlabEQuilibrium Toolbox+]

  %    Copyright 2022-2025 Swiss Plasma Center EPFL
  %
  %   Licensed under the Apache License, Version 2.0 (the "License");
  %   you may not use this file except in compliance with the License.
  %   You may obtain a copy of the License at
  %
  %       http://www.apache.org/licenses/LICENSE-2.0
  %
  %   Unless required by applicable law or agreed to in writing, software
  %   distributed under the License is distributed on an "AS IS" BASIS,
  %   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  %   See the License for the specific language governing permissions and
  %   limitations under the License.

  properties
    data;
    relTol = 6e-3;
  end
  
  properties(ClassSetupParameter)
    npq = {32  64};
  end

  methods(TestClassSetup)
    function setup(testCase,npq)
      % Run FBT for doublet
      [L,~,LY] = fbt('ana',82,0,...
                      'iterq',20,'noq',128,'pq',linspace(0,1,npq),... % Parameters for meqpostq
                      'icsint',true,'ilim',3);                        % Use cubic spline interpolation

      % Store them
      testCase.data = {L,LY};
    end
  end

  methods(Test,TestTags={'Unit'})
    function compare_fsgi_mantle(testCase)
      [L,LY] = deal(testCase.data{:});
      
      % Copy first domain information in LY into mantle domain
      fnames = fieldnames(LY);
      for ii=1:numel(fnames)
        fname = fnames{ii};
        val = LY.(fname);
        val_size = size(val);
        % if last dimension or only dimension is domain dimension
        if val_size(end) == L.nD || (numel(val_size) == 2 && all(val_size == [L.nD, 1]))
          val = reshape(val, [], L.nD);
          val(:, 3) = val(:, 1); % copy from first domain
          LY.(fname) = reshape(val, val_size);
        end
      end
      
      % set ray origin points to first axis
      rO = repmat(LY.rA(1), L.noq, 3);
      zO = repmat(LY.zA(1), L.noq, 3);
      % run fsgi
      [Q0Q,Q1Q,Q2Q,Q3Q,Q4Q,iqQ,ItQ,LpQ,rbQ,Q5Q,SlQ,...
       VQ,AQ,IpVQ,FtPVQ,...
       rgeom,zgeom,aminor,epsilon,kappa,delta,deltal,deltau,...
       rrmax,zrmax,rrmin,zrmin,rzmax,zzmax,rzmin,zzmin] = fsgi(L, LY, LY.rq, LY.zq, LY.aq, rO, zO);
      % special treatment for IpVQ and FtPVQ
      IpVQ = IpVQ - sum(LY.IpVQ(end, [1,2])); % subtract separatrix value
      FtPVQ = FtPVQ - sum(LY.FtPVQ(end, [1,2]));
      
      % Equality is not checked for first entry as fsavgmantle always assumes an X-point at rho=0
      msk = 2:L.nQ;
      
      % Verify
      tol = testCase.relTol;
      testCase.verifyEqual(LY.Q0Q(msk,1),Q0Q(msk,3),'RelTol',tol,'mantle fsgi computation differs for Q0Q');
      testCase.verifyEqual(LY.Q1Q(msk,1),Q1Q(msk,3),'RelTol',tol,'mantle fsgi computation differs for Q1Q');
      testCase.verifyEqual(LY.Q2Q(msk,1),Q2Q(msk,3),'RelTol',tol,'mantle fsgi computation differs for Q2Q');
      testCase.verifyEqual(LY.Q3Q(msk,1),Q3Q(msk,3),'RelTol',tol,'mantle fsgi computation differs for Q3Q');
      testCase.verifyEqual(LY.Q4Q(msk,1),Q4Q(msk,3),'RelTol',tol,'mantle fsgi computation differs for Q4Q');
      testCase.verifyEqual(LY.iqQ(msk,1),iqQ(msk,3),'RelTol',tol,'mantle fsgi computation differs for iqQ');
      testCase.verifyEqual(LY.ItQ(msk,1),ItQ(msk,3),'RelTol',tol,'mantle fsgi computation differs for ItQ');
      testCase.verifyEqual(LY.LpQ(msk,1),LpQ(msk,3),'RelTol',tol,'mantle fsgi computation differs for LpQ');
      testCase.verifyEqual(LY.rbQ(msk,1),rbQ(msk,3),'RelTol',tol,'mantle fsgi computation differs for rbQ');
      testCase.verifyEqual(LY.Q5Q(msk,1),Q5Q(msk,3),'RelTol',tol,'mantle fsgi computation differs for Q5Q');
      testCase.verifyEqual(LY.SlQ(msk,1),SlQ(msk,3),'RelTol',tol,'mantle fsgi computation differs for SlQ');
      testCase.verifyEqual(LY.VQ(msk,1),VQ(msk,3),'RelTol',tol,'mantle fsgi computation differs for VQ');
      testCase.verifyEqual(LY.AQ(msk,1),AQ(msk,3),'RelTol',tol,'mantle fsgi computation differs for AQ');
      testCase.verifyEqual(LY.rgeom(msk,1),rgeom(msk,3),'RelTol',tol,'mantle fsgi computation differs for rgeom');
      testCase.verifyEqual(LY.zgeom(msk,1),zgeom(msk,3),'RelTol',tol,'mantle fsgi computation differs for zgeom');
      testCase.verifyEqual(LY.aminor(msk,1),aminor(msk,3),'RelTol',tol,'mantle fsgi computation differs for aminor');
      testCase.verifyEqual(LY.epsilon(msk,1),epsilon(msk,3),'RelTol',tol,'mantle fsgi computation differs for epsilon');
      testCase.verifyEqual(LY.kappa(msk,1),kappa(msk,3),'RelTol',tol,'mantle fsgi computation differs for kappa');
      testCase.verifyEqual(LY.delta(msk,1),delta(msk,3),'RelTol',tol,'mantle fsgi computation differs for delta');
      testCase.verifyEqual(LY.deltal(msk,1),deltal(msk,3),'RelTol',tol,'mantle fsgi computation differs for deltal');
      testCase.verifyEqual(LY.deltau(msk,1),deltau(msk,3),'RelTol',tol,'mantle fsgi computation differs for deltau');
      testCase.verifyEqual(LY.rrmax(msk,1),rrmax(msk,3),'RelTol',tol,'mantle fsgi computation differs for rrmax');
      testCase.verifyEqual(LY.zrmax(msk,1),zrmax(msk,3),'RelTol',tol,'mantle fsgi computation differs for zrmax');
      testCase.verifyEqual(LY.rrmin(msk,1),rrmin(msk,3),'RelTol',tol,'mantle fsgi computation differs for rrmin');
      testCase.verifyEqual(LY.zrmin(msk,1),zrmin(msk,3),'RelTol',tol,'mantle fsgi computation differs for zrmin');
      testCase.verifyEqual(LY.rzmax(msk,1),rzmax(msk,3),'RelTol',tol,'mantle fsgi computation differs for rzmax');
      testCase.verifyEqual(LY.zzmax(msk,1),zzmax(msk,3),'RelTol',tol,'mantle fsgi computation differs for zzmax');
      testCase.verifyEqual(LY.rzmin(msk,1),rzmin(msk,3),'RelTol',tol,'mantle fsgi computation differs for rzmin');
      testCase.verifyEqual(LY.zzmin(msk,1),zzmin(msk,3),'RelTol',tol,'mantle fsgi computation differs for zzmin');
      % higher tolerances for IpVQ and FtPVQ as they are integrals against VQ
      testCase.verifyEqual(LY.IpVQ(msk,1),IpVQ(msk,3),'RelTol',5*tol,'mantle fsgi computation differs for IpVQ');
      testCase.verifyEqual(LY.FtPVQ(msk,1),FtPVQ(msk,3),'RelTol',5*tol,'mantle fsgi computation differs for FtPVQ');
    end
  end

end
