classdef fgs_test < meq_test
  % Basic tests for fgs
  %
  % [+MEQ MatlabEQuilibrium Toolbox+]

  %    Copyright 2022-2025 Swiss Plasma Center EPFL
  %
  %   Licensed under the Apache License, Version 2.0 (the "License");
  %   you may not use this file except in compliance with the License.
  %   You may obtain a copy of the License at
  %
  %       http://www.apache.org/licenses/LICENSE-2.0
  %
  %   Unless required by applicable law or agreed to in writing, software
  %   distributed under the License is distributed on an "AS IS" BASIS,
  %   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  %   See the License for the specific language governing permissions and
  %   limitations under the License.

  properties
    tok = 'ana';
    shot = 1;
    t = 0;
    verbosity = 0;
  end

  properties(TestParameter)
    debug = {0,1,2,3,4,5}
    algoNL = {'all-nl','all-nl-Fx'}
  end

  methods(Test,TestTags = {'fgs'})
    function test_debuglevel(testCase,debug)
      % run fgs with various debug levels
      [L,LX] = fgs(testCase.tok,testCase.shot,testCase.t);

      L.P.debug = debug;
      L.P.debugplot = debug;
      LY = fgst(L,LX);
      testCase.assertTrue(meq_test.check_convergence(L,LX.t,LY),'FGS did not converge');
    end

    function test_fgeF_debug_display(testCase)
      % Test display and plotting during fgeF internal (meqpdom) error
      [L,~,LY] = fgs(testCase.tok,testCase.shot,testCase.t);

      LY.Iy = 1e-6*LY.Iy; % zero Iy, should cause failure to find LCFS
      x = L.LX2x(LY);
      opts = optsF; opts.dodisp = 1; opts.doplot = 1; opts.dopost=1;
      [~,LYout] = fgeF(x,L,LY,LY,opts);

      % check that post-processing was triggered
      testCase.assertTrue(isfield(LYout,'FB'),'FB not present in LY')
      % check that plasma domain was not found
      testCase.assertFalse(LYout.lB)
      % check that slice is marked as non-converged
      testCase.assertFalse(LYout.isconverged)
    end

    function test_restart(testCase)
      % non-converging example to test debugging and restarting
      [L,LX] = fgs('ana',1,0,'nrestarts',1,'restart_random',false);
      LX.bp = LX.bp*1.4;
      L.P.debug=2; L.P.debugplot = 2;
      LY = fgst(L,LX);
      testCase.assertFalse(LY.isconverged,'expected non-convergence')
    end
    
    function test_bp_scans(testCase,algoNL)
      % Check that bp scans on a fixed Ia still converge
      
      % nominal case
      [L,~,LY] = fgs('ana',2,0,...
        'nrestarts',0,... % No random restart allowed
        'kmax',60,'ilim',3,'icsint',true,...
        'agcon',{'Ip','bp','qA'},...
        'debug',1,'xscal',1,...
        'algoNL',algoNL);
      testCase.assertTrue(LY.isconverged,'nominal FGS case did not converge');
      LX = LY; % use nominal case as starting point for perturbed case
      
      % verbosity
      if testCase.verbosity
        L.P.debugplot = 3; L.P.debug=3; 
      else, L.P.debug = 2;
      end

      % bp scale factor to scan over, remove scale = 1
      bp_scale_grid = 0:0.25:1.8; 
      bp_scale_grid = bp_scale_grid(bp_scale_grid~=1);
      nbg = numel(bp_scale_grid);
       
      L = fgsc(L.P,L.G);
      niter = repmat(L.P.kmax,1,nbg); % defaults
      
      for ii = 1:numel(bp_scale_grid)
        LX.bp = LY.bp*bp_scale_grid(ii); % scale bp
        
        LYnew = fgst(L,LX); % rerun fgs
        testCase.verifyTrue(LYnew.isconverged,sprintf('no convergence for bp=%4.2f',bp_scale_grid(ii)))

        if testCase.verbosity, drawnow; end
        if LYnew.isconverged
          niter(ii) = LYnew.niter;
        end
      end
      
      visible = testCase.verbosity>0;
      hf = figure('Visible',visible);
      % Always close invisible figures
      if ~visible, clean = onCleanup(@() close(hf)); end
      set(hf,'position',[10 10 1300 600])
      if strcmp(algoNL,'all-nl'), jj=1; else jj=2; end % plot index
      hax=subplot(1,2,jj,'parent',hf);
      plot(hax,bp_scale_grid,niter)
      xlabel('LX.bp/LX.bp(original)')
      title(sprintf('FGS %s shot=%d t=%.2f algoNL=%s',L.P.tok,LY.shot,LY.t,L.P.algoNL))
      set(hax,'Ylim',[0,L.P.kmax])
      drawnow
    end
  end
end