classdef fgs_fge_jacobian_test < meq_jacobian_test
  % Tests Forward-Grad-Shafranov solvers FGS and FGE using semi-analytical Jacobian
  %
  % [+MEQ MatlabEQuilibrium Toolbox+]

  %    Copyright 2022-2025 Swiss Plasma Center EPFL
  %
  %   Licensed under the Apache License, Version 2.0 (the "License");
  %   you may not use this file except in compliance with the License.
  %   You may obtain a copy of the License at
  %
  %       http://www.apache.org/licenses/LICENSE-2.0
  %
  %   Unless required by applicable law or agreed to in writing, software
  %   distributed under the License is distributed on an "AS IS" BASIS,
  %   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  %   See the License for the specific language governing permissions and
  %   limitations under the License.

  properties
    tok = 'ana';
    t = 0:5e-3:1e-2;
    verbosity = 0;
    agcon = {'bp','li','qA'};
  end

  properties(ClassSetupParameter)
    codes      = {'fgs','fge'};
    algosNL    = {'all-nl','all-nl-Fx','Newton-GS'};
    algosGMRES = {'direct_inversion','aya','matlab_gmres'};
  end

  properties(TestParameter)
    shot = struct('diverted',2,'double_null',4,'squashed',5,'doublet',82);
    % NOTE : if the grid is too fine, the simulation gets very computationally 
    % expensive since the memory requirement goes as nr*nr*nz*nz*ng
    nr = struct('coarse',20,'fine',45); 
    nz = struct('coarse',8,'fine',32);
  end

  methods(TestClassSetup)

    function set_code_solver(testCase,codes,algosNL,algosGMRES)
      testCase.code      = codes;
      testCase.algoGMRES = algosGMRES;
      testCase.algoNL    = algosNL;
    end

  end

  methods(Test,TestTags={'Jacobian-Integration'})

    function test_code_with_jacobian(testCase,shot)
      
      if testCase.verbosity
        fprintf('Testing code=%s, tok=%s shot=%d, algoGMRES=%s using the analytical Jacobian\n',upper(testCase.code),testCase.tok,shot,testCase.algoGMRES)
      end

      % Run on shot with newton and analytical jacobian
      [L,LX] = get_L_LX(testCase,shot,testCase.t,'algoF','newton');
      LY = get_LY(testCase,L,LX);

      % Verify run succeeded
      testCase.verifyTrue(meq_test.check_convergence(L,LX.t,LY),sprintf('shot %d did not converge',shot));
    end

  end

  methods(Test,TestTags={'Jacobian-Integration'},ParameterCombination='sequential')

    function code_newton_numerical_analytical_Jacobian(testCase)
      for anajac=[false,true]
        if anajac
          results_prefix = 'analytical';
        else
          results_prefix = 'numerical';
        end
        if strcmp(testCase.algoGMRES,'direct_inversion') || anajac
          algoF = 'newton';
        else
          algoF = 'jfnk';
        end
        if testCase.verbosity
          fprintf('Testing tok=%s, algoGMRES=%s with the %s Jacobian\n',testCase.tok,testCase.algoGMRES,results_prefix)
        end

        start_time = tic;
        % Run on diverted case
        [L,LX] = get_L_LX(testCase,2,testCase.t,'algoF',algoF);
        LY = get_LY(testCase,L,LX);

        results.(results_prefix).t = toc(start_time);
        results.(results_prefix).LY = LY;
      end

      % dispatch results
      njac = results.numerical ; njac.niter = sum(njac.LY.niter); njac.nfeval = sum(njac.LY.nfeval);
      ajac = results.analytical; ajac.niter = sum(ajac.LY.niter); ajac.nfeval = sum(ajac.LY.nfeval);

      % test validity of results
      testCase.verifyTrue(meq_test.check_convergence(L,LX.t,njac.LY),'Simulation with numerical jacobian did not converge');
      testCase.verifyTrue(meq_test.check_convergence(L,LX.t,ajac.LY),'Simulation with analytical jacobian did not converge');
      % tol = sqrt(eps);
      tol = 1e-10;
      testCase.verifyEqual(njac.LY.rB, ajac.LY.rB, 'AbsTol', tol);
      testCase.verifyEqual(njac.LY.zB, ajac.LY.zB, 'AbsTol', tol);
      testCase.verifyEqual(njac.LY.FB, ajac.LY.FB, 'AbsTol', tol);
      testCase.verifyEqual(njac.LY.rA, ajac.LY.rA, 'AbsTol', tol);
      testCase.verifyEqual(njac.LY.zA, ajac.LY.zA, 'AbsTol', tol);
      testCase.verifyEqual(njac.LY.FA, ajac.LY.FA, 'AbsTol', tol);
      testCase.verifyEqual(njac.LY.Fx, ajac.LY.Fx, 'AbsTol', tol);

      % analyze and display results
      MethodName = testCase.algoGMRES;
      if ~ismember(MethodName,{'direct_inversion','matlab_gmres'})
        MethodName = ['GMRES-',MethodName];
      end
      MethodName = [upper(testCase.code),' ', MethodName];
      Methods   = {[MethodName,' with numerical Jacobian (num)']; [MethodName,' with analytical Jacobian (ana)']; 'Relative (num/ana)'};
      ComputationTime = [ njac.t     ; ajac.t     ; njac.t      / ajac.t      ];
      TotalNiter      = [ njac.niter ; ajac.niter ; njac.niter  / ajac.niter  ];
      TotalNfeval     = [ njac.nfeval; ajac.nfeval; njac.nfeval / ajac.nfeval ];
      ComparisonTable = table(Methods,ComputationTime,TotalNiter,TotalNfeval);
      disp(ComparisonTable);
    end

    function coarser_finer_grid_test(testCase,nr,nz)
      if testCase.verbosity
        fprintf('Testing %s with analytical Jacobian on a (%u,%u) grid\n',upper(testCase.code),nr,nz)
      end

      % Run diverted case with newton and analytical jacobian
      [L,LX] = get_L_LX(testCase,2,testCase.t,'algoF','newton');
      LY = get_LY(testCase,L,LX);

      % Verify run succeeded
      testCase.verifyTrue(meq_test.check_convergence(L,LX.t,LY),sprintf('Grid (%u,%u) did not converge',nr,nz));
    end
  end

end
