classdef fgs_cases_test < meq_test
  % Test various solver methods and ilackner options for FGS for various shots
  %
  % [+MEQ MatlabEQuilibrium Toolbox+]

  %    Copyright 2022-2025 Swiss Plasma Center EPFL
  %
  %   Licensed under the Apache License, Version 2.0 (the "License");
  %   you may not use this file except in compliance with the License.
  %   You may obtain a copy of the License at
  %
  %       http://www.apache.org/licenses/LICENSE-2.0
  %
  %   Unless required by applicable law or agreed to in writing, software
  %   distributed under the License is distributed on an "AS IS" BASIS,
  %   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  %   See the License for the specific language governing permissions and
  %   limitations under the License.

  properties
    fname
    tok = 'ana'
    L_baseline;
    LX_baseline
    LY_baseline;
    shot_baseline;
    dosave = false;
    verbosity = 0;
    insrc
  end
  
  properties (ClassSetupParameter)
    shots = struct('circular',1,'diverted',2,'diverted2',3,...
      'DN',4,'squashed',5,'NT',6,'elongated',11,'doublet',82);
  end
  
  properties(TestParameter)
    method = {'all-nl','all-nl-Fx','Picard','Newton-GS'};
    lackner_option = struct('green',0, 'default', 1, 'precomputed',2);
  end
  
  methods(TestClassSetup)
    function testSetup(testCase, shots)
      [testCase.L_baseline,testCase.LX_baseline,testCase.LY_baseline] = ...
        fgs('ana',shots,0,'algoNL', 'all-nl');
      testCase.shot_baseline = shots;
      testCase.verifyTrue(meq_test.check_convergence(testCase.L_baseline,0,testCase.LY_baseline),sprintf('shot %d did not converge',testCase.shot_baseline));
    end
  end
  
  methods(Test,TestTags={'fgs'})
    
    function test_algoNL_methods(testCase, method)
      shot = testCase.shot_baseline;
      LY_ = testCase.LY_baseline;
      LX_ = testCase.LX_baseline;
      
      % exclude these cases
      testCase.assumeFalse(strcmp(method, 'Picard') && shot==82,...
        'Picard method does not yet work for doublets, skipping test')
      
      [algoF, anajac] = adapt_algoF(method);
      
      L = fgs('ana',shot,0,'algoNL',method,...
        'usepreconditioner',~isequal(method,'Picard'),...
        'kpic', 200, ...
        'algoF', algoF, 'anajac', anajac);
      LY = fgst(L,LX_);

      testCase.verifyTrue(meq_test.check_convergence(L,0,LY),sprintf('shot %d did not converge',shot));
      testCase.verifyEqual(LY.Fx,LY_.Fx,'AbsTol',L.Fx0*200*L.P.tolF)
      testCase.verifyEqual(LY.Iy,LY_.Iy,'AbsTol',L.Iy0*200*L.P.tolF)
      testCase.verifyEqual(LY.bp,LY_.bp,'AbsTol',      200*L.P.tolF)
      testCase.verifyEqual(LY.li,LY_.li,'AbsTol',      200*L.P.tolF)
      testCase.verifyEqual(LY.Ip,LY_.Ip,'AbsTol',L.Ip0*200*L.P.tolF)
    end
    
    function test_lackner_options(testCase, lackner_option)
      LX = testCase.LX_baseline;
      LY_target = testCase.LY_baseline;
      myshot = testCase.shot_baseline;
      
      L = fgs('ana',myshot,0,'algoNL', 'all-nl', 'ilackner', lackner_option);
      LY = fgst(L,LX);

      % cases with ilackner=0 are harder to converge
      testCase.verifyTrue(meq_test.check_convergence(L,0,LY) || (L.P.ilackner == 0),sprintf('shot %d did not converge',myshot));
      testCase.verifyEqual(LY_target.Fx,LY.Fx,'AbsTol',L.Fx0*2e-2);
      testCase.verifyEqual(LY_target.Iy,LY.Iy,'AbsTol',L.Ip0*2e-2);
    end
    
    function test_fgs(testCase, method)
      shot = testCase.shot_baseline;
      testCase.assumeFalse(isequal(method,'Picard') && shot==82,...
        'Picard method does not yet work for doublets, skipping test')
      
      if testCase.verbosity
        fprintf('Testing %s %s, %s\n',testCase.tok,shot,method)
      end
      
      [algoF, anajac] = adapt_algoF(method);
      
      t = 0:5e-3:1e-2; % test retrieving multiple slices for anamak too
      [L,~,LY] = fgs(testCase.tok,shot,t,...
            'debug',testCase.verbosity,'algoNL',method,'tol',1e-8,...
            'usepreconditioner', ~isequal(method,'Picard'), ...
            'algoF', algoF, 'anajac', anajac);

      testCase.verifyTrue(meq_test.check_convergence(L,t,LY),sprintf('shot %d did not converge',shot));
    end
  end
end

function [algoF, anajac] = adapt_algoF(method)
% Adapt algoF and anajac to algoNL setting
switch method
  case 'Newton-GS'
    algoF = 'newton';
    anajac = true;
  case 'Picard'
    algoF = 'jfnk';
    anajac = false;
  case {'all-nl','all-nl-Fx'}
    algoF = 'jfnk';
    anajac = true;
end
end
