classdef fge_CDE_1D_analytical_test < meq_test
  % Tests of 1D current diffusion equations against analytical examples
  %
  % [+MEQ MatlabEQuilibrium Toolbox+]

  %    Copyright 2022-2025 Swiss Plasma Center EPFL
  %
  %   Licensed under the Apache License, Version 2.0 (the "License");
  %   you may not use this file except in compliance with the License.
  %   You may obtain a copy of the License at
  %
  %       http://www.apache.org/licenses/LICENSE-2.0
  %
  %   Unless required by applicable law or agreed to in writing, software
  %   distributed under the License is distributed on an "AS IS" BASIS,
  %   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  %   See the License for the specific language governing permissions and
  %   limitations under the License.

  properties (ClassSetupParameter)
   nx = {32, 64, 128, 256, 512, 1024}; % pow 2 increase in grid size
   dt = {0.5e-4, 0.125e-4, 3.125e-6, 7.8125e-7, 1.9531e-7, 4.8828e-8}; % pow 4 decrease in dt
   hs = num2cell(2.^(-(0:5)), 1); % scaling parameter
  end
  
  
  properties (MethodSetupParameter)
    steady_state = {0, 1};
    cdetype = {'cde_1D', 'cde_OhmTor_1D', 'static_cde_1D'};
  end
  
  properties (Access = protected)
    Ls; tol; % save screw pinch setup in Ls
    flag_ss; % flag if steady state formulation of analytical case is used
    cdefun;
    h;
    beta; % factor deciding relative influence of p' and TT' to the current distribution
  end
  
  methods (TestClassSetup, ParameterCombination = 'sequential')
    function setup_L(testCase, nx, dt, hs)
      L = struct();
      
      % grid
      [L.xx, L.yx] = meshgrid(linspace(-1, 1, nx)); % 2mx2m space for plasma
      L.nx = nx * nx; L.ny = (nx - 2) * (nx - 2);
      L.nzy = nx - 2; L.nry = nx - 2;
      L.idrx = nx * 0.5; L.idzx = nx * 0.5;
      L.rry = ones(nx-2, nx-2);
      L.dt = dt;
      L.nD = 1;
      L.dsx = 4 / (nx * nx);
      L.idsx = (nx * nx) * 0.25;
      L.rB = 0.8; % plasma does not quite go to the edge
      
      % Q - grid
      %   Use large number of points to avoid interpolation errors
      %   interfering with convergence of residual for analytic case
      L.pQ = linspace(0, 1, L.nx);
      
      % basis functions 2 bfef basis function are able to exactly represent
      % p' and TT' profiles from the analytical examples
      n_basis = 2;
      L.bfct = @bfefmex;
      L.bfp = [n_basis,n_basis];
      L.igD = [false(n_basis, 1); true(n_basis, 1)];
      L.fTg = double(L.igD); L.fPg = double(~L.igD);
      L.TDg = true(1, 2 * n_basis);
      L.ng = 2 * n_basis;
      L.intG0 = ones(L.ng, 1); % no scaling, tolerances are adjusted
      
      L.sigma0 = 1e+6; % base sigma used
      L.Ip0 = 1e+6; % rough base Ip
      
      testCase.Ls = L;
      testCase.h = hs;
    end
  end
  
  methods (TestMethodSetup)
    function set_steady_state_flag(testCase, steady_state, cdetype)
      % static_cde_1D is only valid in steady state
      testCase.assumeTrue(~strcmp(cdetype, 'static_cde_1D') || steady_state);
      
      testCase.flag_ss = steady_state;
      S = meqcdefun();
      testCase.cdefun = S.(cdetype);
      % The cde_OhmTor_1D residual is only true in the small diamagnetic
      % approximation, hence TT' = 0;
      % Also, cde_1D residual decays only with h and not h^2 because of
      % |grad psi|^2 term
      if strcmp(cdetype, 'cde_OhmTor_1D')
        testCase.beta = 0;
        testCase.tol = testCase.h^2 * 0.002;
      else
        testCase.beta = 0.1;
        testCase.tol = testCase.h * 0.02;
      end
    end
  end
  
  methods (Test, TestTags = {'fge'})
    
    function test_1D_cdes(testCase)
      % tests the 1D cdes against a screw pinch plasma case
      L = testCase.Ls;
      
      cde_func = testCase.cdefun;
      
      [L,LX,LY,Fx,ag,Iy,F0,F1,Ie,Opy,idt,igD] = ...
        generate_analytical_test_case(L, testCase.flag_ss, testCase.beta);
      
      res = cde_func(L,LX,LY,Fx,ag,Iy,F0,F1,Ie,Opy,idt,igD);
      
      testCase.verifyTrue(norm(res) < testCase.tol);
    end

  end
end


function [L,LX,LY,Fx,ag,Iy,F0,F1,Ie,Opy,idt,igD] = generate_analytical_test_case(L, steady_state, beta)
  % Returns parameters of steady state screw-pinch equilibrium with Ohms law
  % CDE is "hacked" to be screw-pinch formulation by setting L.rry === 1
  
  % basic inputs to CDE
  igD = L.igD;
  idt = 1 / L.dt;
  Ie = [];
  rBt = 1;
  
  % Define circular plasma domain
  r = sqrt(L.xx.^2 + L.yx.^2);
  Opx = int8(r <= L.rB);
  Opy = Opx(2:end-1, 2:end-1);
  
  % parameters for the bessel function formulation
  j0 = 2.40482555769577276862163188; % bessel function root
  alpha = j0 / L.rB; % analytical solution coefficient
  vloop = -1 / (mu0 * L.sigma0); % vloop for steadt_state==true
  kappa = -alpha^2 / (mu0 * L.sigma0); % current decay exponent for steady_state==false
  gamma = alpha^2 / (4 * pi^2 * mu0) - beta^2 / mu0; % pressure component parameter
  
  % flux definition
  Fx = besselj(0, alpha * r) .* (Opx==1) + ... % bessel in plasma domain
       -alpha * L.rB * besselj(1, L.rB) * log(r / L.rB) .* (Opx==0); % log decay outside
  F0 = 1;
  F1 = 0;
  
  % Iy and ags
  Iy = (L.dsx * alpha^2 ./ (2 * pi * mu0)) * Fx(2:end-1, 2:end-1) .* (Opy==1);
  PpQ = (gamma * (L.pQ.^2 * (F1 - F0) + F0)).' - beta / mu0 * rBt;
  TTpQ = (beta^2 * (L.pQ.^2 * (F1 - F0) + F0)).' + beta * rBt;
  ag = meqfitprof(L,L.pQ.^2,F0,F1,PpQ,TTpQ);
  
  % signeo profile, different for steady-state or current decay mode
  if steady_state
    % small regularization to not get zero on boundary
    rr = linspace(0, L.rB, L.nx);
    FQ = L.pQ.^2 * (F1 - F0) + F0;
    sigma_of_rr = L.sigma0 * alpha^2 * (besselj(0, alpha * rr) + beta * besselj(1, alpha * rr).^2 ./ (beta .* besselj(0, alpha * rr) + rBt));
    signeo = interp1(besselj(0, alpha * rr), sigma_of_rr, FQ)' + 1e-3;
  else
    signeo = L.sigma0 * ones(numel(L.pQ), 1);
  end
    
  % set up LX
  LX = struct();
  LX.Opy = Opx(2:end-1, 2:end-1); % same plasma area
  LX.Fx = Fx;
  LX.F0 = F0;
  LX.F1 = F1;
  LX.Iy = Iy;
  LX.ag = ag;
  LX.t = L.dt;
  
  % set up LY
  LY = LX;
  LY.t = 0;
  if steady_state
    LY.Fx = Fx - vloop * L.dt; % ohmic drive
    LY.Iy = Iy;
    LY.F0 = F0 - vloop * L.dt;
    LY.F1 = F1 - vloop * L.dt;
  else
    LY.Fx = Fx * exp(kappa * -L.dt); % scale for previous timestep
    LY.Iy = Iy * exp(kappa * -L.dt);
    LY.F0 = F0 * exp(kappa * -L.dt);
    LY.F1 = F1; % F1 stays the same
  end
  LY.ag = ag; % ag stay the same
  
  % LX specific stuff
  LX.rBt = rBt;
  LX.signeo = signeo;
  LX.Ip = sum(Iy(:));
  LX.Ini = 0;
end
