classdef fgeFJac_picard_tests < meq_test
  % Tests for FGEF Picard preconditioners
  %
  % [+MEQ MatlabEQuilibrium Toolbox+]

  %    Copyright 2022-2025 Swiss Plasma Center EPFL
  %
  %   Licensed under the Apache License, Version 2.0 (the "License");
  %   you may not use this file except in compliance with the License.
  %   You may obtain a copy of the License at
  %
  %       http://www.apache.org/licenses/LICENSE-2.0
  %
  %   Unless required by applicable law or agreed to in writing, software
  %   distributed under the License is distributed on an "AS IS" BASIS,
  %   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  %   See the License for the specific language governing permissions and
  %   limitations under the License.

  properties
    verbosity = 0;
    params = {'selu','e','nu',20}
  end
  
  properties(TestParameter)
    algoNL = {'all-nl','all-nl-Fx','Newton-GS'};
    dt = {0,1e-4,1e-3};
  end
  
  methods(Test,TestTags = {'Jacobian'})
    function test_fge_picard_precs(testCase,algoNL,dt)
      % Test computation of picard preconditioner for FGE

      [L,LX] = fge('ana',1,0,'algoNL',algoNL,testCase.params{:});
      
      % Set up inputs
      LYp = LX;
      LYp.t = LX.t - dt;
      if dt==0, idt=0; else, idt = 1/dt;end

      % Call helper function
      testCase.picard_approx_checks(L,LX,LYp,idt)
    end

    function test_fgs_picard_precs(testCase,algoNL)
      % Test computation of picard preconditioner for FGS

      [L,LX] = fgs('ana',1,0,'algoNL',algoNL,testCase.params{:});
      
      % Set up inputs
      LYp = LX;
      idt = 0;

      % Call helper function
      testCase.picard_approx_checks(L,LX,LYp,idt)
    end
  end

  methods
    function picard_approx_checks(testCase,L,LX,LYp,idt)
      % Helper function to run tests for FGS and FGE
      
      % Set up state
      x = L.LX2x(LX);

      % In the Picard approximation (dIydFx=0, dCodFx=0), the jacobians are
      % actually equal to the vacuum ones (with IpD=0) except for the
      % dependence on the ag coefficient.
      
      % Compute jacobians
      opts = optsF('dojacx',true);
      [~,~,JxP] = fgeF(x,L,LX,LYp,opts); % P for plasma
      % Set up vacuum case
      LX.Ip = 0;
      LX.Iy = zeros(L.nzy,L.nry);
      LX.ag = zeros(L.ng,1);
      if ~strcmpi(L.P.algoNL,'all-nl')
        LX.Fx = meqFx(L,LX.Iy,[LX.Ia;LX.Iu]);
      end
      % No need to update LYp as jacobians only depend on LYp.t through dt
      x = L.LX2x(LX);
      opts = optsF('dojacx',true,'dojacu',true,'dojacxdot',true);
      [~,~,JxV,JuV,JxdotV] = fgeF(x,L,LX,LYp,opts); % V for vacuum
      % Assemble Picard approximation jacobians
      Jx = JxV;
      Jx(:,L.ind.ixg) = JxP(:,L.ind.ixg);
      Ju = JuV;
      Jxdot = JxdotV;

      % Check Jx is not singular
      lastwarn(''); % Reset
      [~] = inv(Jx);
      [~,wid] = lastwarn;
      if strcmp(wid,'MATLAB:singularMatrix')
        error('fgeF jacobian is singular')
      end
      %
      if L.isEvolutive
        dxdu = -Jx\Ju(:,L.ind.iua); % dxdVa
      else
        dxdu = -Jx\Ju(:,[L.ind.iua,L.ind.iuu]); % dxdIe
      end
      dxdxp = Jx\[zeros(L.nN-L.nrD,L.nN);idt*Jxdot];

      % Compute Picard approximation
      [Ps,dxdus,dxdxps] = fgeFJac_inverse_picard_approx(L,idt);
      P = Ps{1}(Jx(:,L.ind.ixg));
      % Compute inverse of Jx based on P
      iJx = P(eye(L.nN));

      % Debug plots
      if testCase.verbosity
        fig = gcf;
      else
        fig = figure;
        cleanupObj = onCleanup(@() close(fig));
      end
      fig.Visible = testCase.verbosity;
      %
      subplot(1,3,1);
      imagesc(log10(abs(iJx*Jx-eye(L.nN))));
      colorbar;
      caxis([-16,0]);
      %
      subplot(1,3,2);
      imagesc(log10(abs(dxdus{1}-dxdu)));
      colorbar;
      caxis([-16,0]);
      %
      subplot(1,3,3);
      imagesc(log10(abs(dxdxps{1}-dxdxp)));
      colorbar;
      caxis([-16,0]);
      %
      if testCase.verbosity, drawnow; end

      % Checks
      rtol = 1e-10;
      testCase.verifyEqual(iJx*Jx,eye(L.nN),'AbsTol',rtol,'Check of Jx inverse failed for picard approximation');
      testCase.verifyEqual(dxdus{1},dxdu,'AbsTol',rtol*max(abs(dxdu(:))),'Check of dxdu failed for picard approximation');
      testCase.verifyEqual(dxdxps{1},dxdxp,'AbsTol',rtol*max(abs(dxdxp(:))),'Check of dxdxp failed for picard approximation');
    end
  end
end
