classdef fbttcv_test < meq_test
  % Tests of FBT for TCV
  %
  % [+MEQ MatlabEQuilibrium Toolbox+]

  %    Copyright 2022-2025 Swiss Plasma Center EPFL
  %
  %   Licensed under the Apache License, Version 2.0 (the "License");
  %   you may not use this file except in compliance with the License.
  %   You may obtain a copy of the License at
  %
  %       http://www.apache.org/licenses/LICENSE-2.0
  %
  %   Unless required by applicable law or agreed to in writing, software
  %   distributed under the License is distributed on an "AS IS" BASIS,
  %   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  %   See the License for the specific language governing permissions and
  %   limitations under the License.

  properties
    % Should be enough to account for differences in models
    standard_shot = 9e5;
    tolI = 5e-2;
    tolF = 5e-2;
    tolq = 5e-2;
    verbosity = 0; % optional plots
  end

  properties (TestParameter)
    shot = {64988};
    sip = struct('positive',1,'negative',-1);
    sb0 = struct('positive',1,'negative',-1); % Check qA constraint works with all b0 signs
    
    % For vacuum field constraints
    vrz = struct('none',0,'pos',0.1,'neg',-0.05); % value for dBrdz
    ve  = struct('none',Inf,'pos',0.001,'neg',0.001); % weight for dBrdz constraint
    
    % for qA constrained runs
    qAref = {1,[1,1.5]}; 
    
    % doublet or no doublet for fbt2pcs
    bfct = {'bfgenD','bfabmex'}
    
    % Comparison with stored runs
    mds_shot={70153,65414,68225}; % Only problematic runs so far

    % Require disabling inequalities in initial step
    init_shot={44176,53362};

    % Shots for coil combination / passive currents / evolutive tests
    test_shot={70153,65414,68225};
  end
  
  methods(TestClassSetup)
    function check_tcv(testCase)
      [ok,msg] = testCase.check_tok('tcv');
      testCase.assertTrue(ok,msg);
    end
  end
  
  methods(Test, TestTags = {'TCV'})
    function test_fbt_shot(testCase,shot,sip,sb0)
      %% Tests for FBT with different field and current orientations
      [L,LX,LY] = fbt('TCV',shot,[],'iohfb',sip,'if36fb',sb0,'useSQP',false);
      % Use Picard method to compare to MDS results of old shots
      
      % Check fbtx displaying option for all time slices (output to void)
      for it=1:numel(LX.t)
        [~] = fbtxdisp(L,meqxk(LX,it));
      end
      
      % Check qA constraint is verified
      qA = 2*pi./LY.rA./sign(LY.FA)./sqrt(LY.dr2FA.*LY.dz2FA-LY.drzFA.^2) .* LY.TQ(1,:);
      testCase.verifyEqual(qA,L.P.qA,'relTol',testCase.tolq);

      % compare with results of FORTRAN version for existing TCV shots
      mdsopen('pcs',shot);
      sip0 = mdsvalue('\mgams.data:iohfb');
      % "RAMP" always has as many times as LY.t
      Ia  = sip*sip0*mdsvalue('tcv_eq("i_pol"   ,"ramp")'); % FBT  nodes, with correct signs through tcv_eq
      FAB = sip*sip0*mdsvalue('tcv_eq("psi_axis","ramp")'); % FBTE nodes, has correct signs
      Ia  = Ia(1:16,:);
      FAB = reshape(FAB,1,[]);
      
      testCase.verifyEqual(LY.Ia        , Ia  ,'absTol',testCase.tolI*repmat(max(abs(Ia),[],1),size(Ia,1),1));
      testCase.verifyEqual(LY.FA - LY.FB, FAB ,'relTol',testCase.tolF);
    end
    
    function test_single_slice(testCase,shot,qAref)
      % Test ability to run select timeslices while customizing a constraint
      t = linspace(0.5,1,numel(qAref));
      
      % test that trying directly via arguments gives an error
      testCase.verifyError(@()fbt('tcv',shot,t,'qA',qAref),'FBT:legacyP');
      
      % test that changing it via LX works
      [L,LX] = fbt('tcv',shot,t);
      LX.qA = qAref;
      % check that all LX fields have the expected size
      for field = fieldnames(LX)'
        if isequal(field{:},'shot'), continue; end
        sizeok = ~isnumeric(LX.(field{:})) || (any(size(LX.(field{:}),2) == [numel(qAref),0]));
        testCase.verifyTrue(sizeok,sprintf('expected %d time slices in all LX entries, error in %s',numel(t),field{:}))
      end
      LX = fbtx(L,LX); % Necessary after manual LX update
      % solve with this LX
      LY = fbtt(L,LX);

      % Check that constraint equation is satisfied
      testCase.verifyTrue(meq_test.check_convergence(L,LX.t,LY),'some slices did not converge');
      testCase.verifyEqual(LY.qA,qAref,'AbsTol',LX.tol,'qA not equal')
    end
    
    function test_fbt_grid(testCase,shot)
      %% Tests for different grids in FBTE
      % Use Picard scheme as fully NL schemes fails to converge except with
      % icsint=true and ilim=3
      [~,~,LY1] = fbt('TCV',shot,[],'useSQP',false);
      [~,~,LY2] = fbt('TCV',shot,[],'useSQP',false,'selx','X');
      
      testCase.verifyEqual(LY2.Ia, LY1.Ia, 'absTol', testCase.tolI*repmat(max(abs(LY1.Ia),[],1),size(LY1.Ia,1),1));
    end

    function test_ia_constraint(testCase,shot)
      %% Tests that Ia constraint using gpia is working
      P = mgp(shot);
      P.gpia(16,20) = -6e3;
      P.gpie(:,20) = [Inf*ones(15,1);0];
      P.gpid(20) = 1;
      [~,~,LY] = fbt('TCV',shot,[],'gpia',P.gpia,'gpie',P.gpie,'gpid',P.gpid);
      % Verify that exact constraint is enforced
      testCase.verifyEqual(P.iohfb*P.gpia(16,20),LY.Ia(16,20),'absTol',1);
    end
    
    function test_ia_conflict(testCase,shot)      
      %% Tests that conflicts between gpia and icoilon are identified
      P = mgp(shot);
      P.gpia(16,end) = -6e3;
      P.gpie(:,end) = [Inf*ones(15,1);0];
      P.gpid(end) = 1;
      % There is no way to specify the number of return arguments for the call in R2017a (fine for this test)
      testCase.verifyError(@() fbt('TCV',shot,[],'gpia',P.gpia,'gpie',P.gpie,'gpid',P.gpid),'fbtptcv:IaConflict');
    end
    
    function test_LX_interpolation(testCase,shot)
      %% Test time interpolation
      [L,LX] = fbt('tcv',shot);
      
      % select compatible slices
      iok = ~any(diff(isnan(LX.gpr),[],2));
      i1 = 1; i2 = find(diff(iok)==-1,1,'first')+1; % last compatible timeslice in initial sequence
      tt = linspace(LX.t(i1),LX.t(i2),11);
      LX = fbtxinterp(LX,tt,'linear',true);
      LY = fbtt(L,LX);
      testCase.assertTrue(meq_test.check_convergence(L,LX.t,LY),'not all time slices converged')
      testCase.assertEqual(LY.t,tt,'AbsTol',1e-16, 'LY.t is not equal to interpolation time')
    end

    function test_writing_fbt_to_tcv_trees(testCase)
      %% Test full FBTE run for TCV including writing to trees

      % Note setup and teardown description will only be displayed when used as a shared fixture
      testCase.applyFixture(tcv_temp_tree_fixture('pcs'));

      testshot = mgu.mytestshot();
      MG = mgp(testCase.standard_shot); % Load standard shot
      MG.save(testshot);

      [~,~,~] = fbte(testshot);
    end
  end
  
  methods(Test, TestTags = {'TCV'},ParameterCombination='sequential')

    function test_vrz(testCase,vrz,ve)
      %% Tests enforcing vacuum field curvature
      shot_ = testCase.standard_shot;
      [L,LX] = fbt('tcv',shot_,0.01);

      rr = L.P.rmajo1(1);
      zz = L.P.zmajo1(1);
      vrz = L.P.iohfb*vrz;
      %    fbtgp(LX,r ,z ,b,fa,fb,fe,br,bz,ba,be,cr,cz,ca,ce,vrr,vrz,vzz,ve)
      LX = fbtgp(LX,rr,zz,0,[],[],[],[],[],[],[],[],[],[],[],NaN,vrz,NaN,ve);
      LX.gpvd = 1;
      LX = fbtx(L,LX); % Necessary after manual LX update

      LY = fbtt(L,LX);
      testCase.verifyTrue(meq_test.check_convergence(L,LX.t,LY),'FBT has not converged');
      
      % Compute vacuum field derivatives using interpolation
      Fxa = reshape(L.G.Mxa*LY.Ia,L.nzx,L.nrx);
      inM = qintc([],L.drx,L.dzx);
      [~,~,~,~,Brzi,~,~] = qintmex(L.G.rx,L.G.zx,Fxa,rr,zz,inM);
      
      if ~isinf(ve)
        testCase.verifyEqual(Brzi,vrz,'relTol',0.1); % Cannot ask much better agreement with qint.
      end
      
      if testCase.verbosity
        %%
        clf;
        contour(L.rx,L.zx,LY.Fx);
        hold on
        contour(L.rx,L.zx,Fxa,'k--'); 
        axis equal;
        title(sprintf('Brz=%2.2f',Brzi));
        drawnow;
        %%
      end
    end
    
    function compare_fbt_vs_mds(testCase,mds_shot)
      % Compare to stored FBT-FORTRAN runs
      [L,LX,LY] = fbt('TCV',mds_shot,[],'useSQP',false);
      % Use Picard method to compare to MDS results of old shots
      testCase.verifyTrue(meq_test.check_convergence(L,LX.t,LY),'FBT has not converged');
      
      % compare with results of FORTRAN version for existing TCV shots
      mdsopen('pcs',mds_shot);
      % "RAMP" always has as many times as LY.t
      Ia  = mdsvalue('tcv_eq("i_pol"   ,"ramp")');
      FAB = mdsvalue('tcv_eq("psi_axis","ramp")');
      Ia  = Ia(1:16,:);
      FAB = reshape(FAB,1,[]);
      
      testCase.verifyEqual(LY.Ia        , Ia  ,'absTol',testCase.tolI*repmat(max(abs(Ia),[],1),size(Ia,1),1));
      testCase.verifyEqual(LY.FA - LY.FB, FAB ,'relTol',testCase.tolF);     
    end
    
    function compare_merge_nomerge(testCase,test_shot)
      %% Tests for FBTE
      tol = 1e-6;
      [L,LX] = fbt('TCV',test_shot,[],'mergex',false,'debug',0);
      LX.tol(:) = tol;
      LY = fbtt(L,LX);
      % reprocess with merge
      L.P.mergex = true;
      LX2 = fbtx(L,LX );
      LY2 = fbtt(L,LX2);
      
      ignored = {'res','resy','resFx','resC','rese','chie'}; % Ignore residuals in comparison
      testCase.verifyTrue(structcmp(LY,LY2,1e-12,ignored),'LX with merged constraints yields different solution');
    end
    
    function test_passive(testCase,test_shot)
      % regular run with vessel specification
      [L,LX,LY] = fbt('tcv',test_shot,[],'selu','e','nu',30,'selx','X',...
        'circuit',false,'debug',testCase.verbosity);
      % run with passive structure effect simulation - only a few times
      [L2,LX2,LY2] = fbt('tcv',test_shot,LX.t(1:10),'selu','e','nu',30,'selx','X',...
        'circuit',true,'debug',testCase.verbosity);
      
      testCase.assertTrue(meq_test.check_convergence(L,LX.t,LY),'FBT without circuit equations did not converge');
      testCase.assertTrue(meq_test.check_convergence(L2,LX2.t,LY2),'FBT with circuit equations did not converge');
      testCase.assertTrue(all(LY.Iv(:)==0),'Expected Iv=0 when not solving circuit equations')
      testCase.assertTrue(norm(LY2.Iv)>0,'Expected Iv~=0 when solving circuit equations')
      
      if testCase.verbosity
        %%
        clf
        iE = contains(L.G.dima,'E');
        iF = contains(L.G.dima,'F');
        subplot(211)
        plot(LY.t,LY.Ia(iE,:)); hold on;
        plot(LY2.t,LY2.Ia(iE,:),'k');
        subplot(212)
        plot(LY.t,LY.Ia(iF,:)); hold on;
        plot(LY2.t,LY2.Ia(iF,:),'k');
        
      end
    end
    
    function test_voltage(testCase,test_shot)
      [L,LX] = fbt('tcv',test_shot,[],...
        'voltlim',true,'circuit',true,...
        'selu','e','nu',30,...
        'icsint',true,'ilim',3,'iterfrz',30,... % Necessary for SQP method
        'izgrid',true,'debug',testCase.verbosity,...
        'tol',1e-4);
      LX = meqxk(LX,LX.t>0 & LX.t<0.2);
      LY = fbtt(L,LX);
      
      testCase.assertTrue(meq_test.check_convergence(L,LX.t,LY),'FBT with circuit equations did not converge');

      %% Check satisfaction of circuit equation
      Mee = [L.G.Maa,L.G.Mau;L.G.Mau',L.G.Muu];
      Ree = diag([L.G.Ra;L.G.Ru]);

      Ie = [LY.Ia;LY.Iu];
      dIy = diff(LY.Iy,[],3);
      Va = LY.Va;

      nt = numel(LY.t);
      res = zeros(L.ne,nt-1); % init
      for it = 2:nt
        dt = LX.t(it)-LX.t(it-1);
        res(:,it) = Mee*(Ie(:,it)-Ie(:,it-1))/dt + Ree*Ie(:,it) + L.Mey*reshape(dIy(:,:,it-1),L.ny,1)/dt - eye(L.ne,L.G.na)*Va(:,it);
      end
      testCase.assertLessThan(abs(res),1e-5*[L.Va0;L.Vu0],sprintf('Circuit equation residual is too high at time step %d',it))
    end
    
    function test_fbt_shuffle_coils(testCase)
      shot_ = 61400; % one shot is enough for this test
      
      dima = {'E','F','OH','G','TOR'}; % regular order
      [L ,~ ,LY ] = fbt('tcv',shot_,[],'sela',dima);

      dima_shuffled = {'TOR','F','OH','G','E'}; % shuffled
      [Ls,~,LYs] = fbt('tcv',shot_,[],'sela',dima_shuffled);
      
      % Check that the coil currents are the same irrespective of shuffling
      [~,io1,io2] = intersect(L.G.dima,Ls.G.dima);
      testCase.assertEqual(LY.Ia(io1,:),LYs.Ia(io2,:),'AbsTol',5e-8,...
        'different solution after shuffling')
    end

    function test_fbt_only_Fcoils(testCase)
      % Test a run with only F coils (why not)
      shot_ = 61400; t_ = 0.1; % one slice is enough, choose early limited plasma
      dima = {'F'};
      [L ,LX ,LY ] = fbt('tcv',shot_,t_,'sela',dima);
      testCase.assertTrue(meq_test.check_convergence(L,LX.t,LY),'did not converge with only F coils')
      testCase.assertTrue(all(contains(L.G.dima,'F')),'expected only F coils');
    end
    
    function test_fbt_allcoils(testCase,test_shot)
      % Test ability to run fbt with all coils
      [L,LX,LY] = fbt('tcv',test_shot,[],'sela',{'E','F','OH','G','TOR'});

      % Check that OH coil currents are equal
      iOH = contains(L.G.dima,'OH');
      IOH = LY.Ia(iOH,:);
      testCase.verifyEqual(IOH(1,:),IOH(2,:),'AbsTol',1e-8);

      % Check that TOR current is proportional to rBt
      ITOR_expected = 2*pi*LY.rBt/(4e-7*pi)/(6*16); % expected value
      testCase.verifyEqual(LY.Ia(contains(L.G.dima,'TOR'),:),ITOR_expected,...
        'AbsTol',1e-10,'TOR current does not match expected value');
      
      % Check that G coil current is nonzero 
      testCase.verifyTrue(~all(LY.Ia(contains(L.G.dima,'G'),:)==0),'expected nonzero G coil current');

      % Check that equality constraint is displayed correctly
      testCase.verifyWarningFree(@() fbtxdisp(L,meqxk(LX,1),{'currents','limits'}));
      
      % Check that the cost-function is displayed correctly
      LX.gpoe(:) = 1; % not a constraint
      testCase.verifyWarningFree(@() fbtxdisp(L,meqxk(LX,1),{'currents','limits'}));
    end

    function test_fbtinit(testCase,init_shot)
      % Test shots where initialization fails with inequalities
      %  Check that keeping inequalities fails
      [L,LX] = fbt('tcv',init_shot,[]);
      LY = fbtt(L,LX,'initineq',true);
      testCase.assumeFalse(meq_test.check_convergence(L,LX.t,LY),'FBT run succeeded with inequalities for initial guess. Consider removing this test case.');
      %  Check that removing inequalities succeds
      LY = fbtt(L,LX,'initineq',false);
      testCase.verifyTrue(meq_test.check_convergence(L,LX.t,LY),'FBT run failed with inequalities disabled for initial guess');
    end

    function test_fbt2pcs(testCase,bfct)
      % Test fbt2pcs function runs
      
      % Set temporary space for MDSplus trees
      testCase.applyFixture(tcv_temp_tree_fixture('pcs'));
      
      switch bfct
        case 'bfgenD'
          % Need access to the scripts folder
          testCase.applyFixture(meqscripts_fixture());
          
          % Generate default doublet configuration
          [L,LX,~,~] = generate_doubletTCV(...
            'meq_parameters',{'sela',{'E','F'}});
        case 'bfabmex'
          [L,LX] = fbt('tcv',900001,1,...
            'bfct',@bfabmex,'bfp',[1 2]); % a Ip>0, BT>0 PdJ case
        otherwise
          error('not implemented case %s',bfct);
      end
      
      % Check signs of Ip/B0
      testCase.assertGreaterThan(LX.Ip ,0,'fbt2pcs needs a positive Ip case');
      testCase.assertGreaterThan(LX.rBt,0,'fbt2pcs needs a positive B0 case');

      for sIp = [1,-1]
        for sBt = [1,-1]
          LX.Ip  = sIp*abs(LX.Ip );
          LX.rBt = sBt*abs(LX.rBt);
          if (sIp<0 || sBt<0)
            % Verify fbt2pcs throws error
            testCase.verifyError(@() fbt2pcs(L,LX),'fbt2pcs:IpBtsigns');
          else
            % Convert
            MG = fbt2pcs(L,LX);
            testshot = mgu.mytestshot();
            MG.save(testshot);

            % Get new LX
            [~,LXm] = fbt('tcv',testshot);
            % Compare them
            testCase.verifyTrue(structcmp(LX,LXm,1e-6,{'tokamak','shot','t','Iy'}));
          end
        end
      end
    end
  end
end
