classdef fbt_test<meq_test
  % Tests of FBT  
  %
  % [+MEQ MatlabEQuilibrium Toolbox+]

  %    Copyright 2022-2025 Swiss Plasma Center EPFL
  %
  %   Licensed under the Apache License, Version 2.0 (the "License");
  %   you may not use this file except in compliance with the License.
  %   You may obtain a copy of the License at
  %
  %       http://www.apache.org/licenses/LICENSE-2.0
  %
  %   Unless required by applicable law or agreed to in writing, software
  %   distributed under the License is distributed on an "AS IS" BASIS,
  %   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  %   See the License for the specific language governing permissions and
  %   limitations under the License.

  properties
    verbosity = 0;
    tokamak = 'ana';
    tol = sqrt(eps);
  end
  
  properties (TestParameter)
    shot = struct('circular',1,'diverted',2,'doublenull',4);
    bp = {0.1 1 2};
    rBt = {1}; % adding -1 makes convergence more difficult, to be investigated
    qA = {0.8 2};
    Ip = {50e3,-150e3};
    bfct = {'bffbt','bf3pmex'};
    margin = {'default','margin','margin_single'};
    itert = {0 50};
    useSQP = {false, true};
    algoNL = {'all-nl','all-nl-Fx','Newton-GS'};
    algoGMRES = {'direct_inversion','qr'};
    usepreconditioner = {true, false};
  end
  
  methods(Test,TestTags={'fbt'})
    function test_scan(testCase,shot,rBt,Ip,bp,qA,bfct,useSQP)
      
      tok = testCase.tokamak;
      qA = sign(Ip)*sign(rBt)*qA;

      if useSQP
        niter =  25; iterfrz = 20;
      else
        niter = 100; iterfrz = 50;
      end
      
      [L] = fbt(tok,shot,[],...
        'bfct',str2func(bfct),'bfp',[],...
        'fbtagcon',{'Ip','Wk','qA'},...
        'debug',0,...
        'useSQP',useSQP,...
        'iterfrz',iterfrz,'niter',niter,...
        'iterq',50,'izgrid',testCase.verbosity>0,...
        'pql',0.2,...
        'ihole',false); % Necessary at high betap for FBT initial iterations in these configurations

      fbtxtok = str2func(['fbtx',tok]);
      LX = fbtxtok([],L);
     
      % Set values from test parameter values
      LX.Ip  = Ip;
      LX.IpD = Ip;
      if L.nD > 1, LX.bpD = bp;
      else,        LX.bp  = bp;
      end
      LX.qA  = qA;
      LX.rBt = rBt;

      LX = fbtx(L,LX); % Necessary after manual LX update
      % Solve FBT
      LY = fbtt(L,LX);
      
      testCase.assertTrue(meq_test.check_convergence(L,LX.t,LY),'FBT did not converge')
      testCase.verifyEqual(LY.bp,bp,'RelTol',10*L.P.tol*(L.Ip0/Ip).^2); % Residual for bp is (bp(x)-bp)*WN/WN0=(bp(x)-bp)*(Ip/Ip0).^2
      testCase.verifyEqual(LY.qA,qA,'RelTol',10*L.P.tol);               % Residual for qA is a complex expression, use relative tolerance with increased headroom
      testCase.verifyEqual(LY.Ip,Ip,'AbsTol',10*L.P.tol*L.Ip0);         % Residual for Ip is (Ip(x)-Ip)/Ip0
      
      if testCase.verbosity
        subplot(211)
        meqplotfancy(L,LY);
        title(sprintf('bp=%2.1f,qA=%2.1f,Ip=%2.2fkA,rBt=%2.1fTm',LY.bp,LY.qA,LY.Ip/1e3,LY.rBt))
        subplot(223)
        plot(L.pQ.^2,LY.PQ/1e3);
        title('P [kPa]'); xlabel('\psi_N');
        ylim([0 20]);

        subplot(224)
        plot(L.pQ.^2,1./LY.iqQ);
        title('q'); xlabel('\psi_N');
        if sign(Ip)*sign(rBt)>0, ylim([0 7]); else, ylim([-7 0]); end
        drawnow;
      end
    end
        
    function test_fbtxdisp(testCase,shot)
      [L,LX] = fbt('ana',shot);
      
      % Call without output
      testCase.assertWarningFree(@()fbtxdisp(L,LX));
      
      % Call with output
      s = fbtxdisp(L,LX);
      testCase.assertClass(s,'cell');
    end

    function test_fbt_empty(testCase)
      % Test FBT case without any constraints, checking it returns Ia=0 and Fx=0
      [L,~,LY] = fbt('ana',0);
      testCase.assertEqual(LY.Fx,zeros(L.nzx,L.nrx));
      testCase.assertEqual(LY.Ia,zeros(L.G.na,1));
      testCase.assertEqual(LY.Ip,0);
    end

    function test_fbt_vacuum(testCase)
      [~,~,LY] = fbt('ana',91);
      testCase.assertTrue(LY.isconverged);
      testCase.assertTrue(LY.Ip==0)
    end

    function test_assign_Iy(testCase)
      % Test direct assignment of Iy
      tt = 0.01:10e-3:0.1; nt = numel(tt);
      [L,LX] = fbt('ana',1,tt);
      LX.t = tt;
      LY0 = fbtt(L,meqxk(LX,1)); % one slice

      LX.Ip = linspace(0.9,1,nt).*LY0.Ip;
      LX.Iy = LY0.Iy.*reshape((LX.Ip./LY0.Ip),1,1,nt);

      L.P.Ipmin = LX.Ip(end);

      % Check that Iy are as prescribed when not solving for it
      LY = fbtt(L,LX);
      testCase.assertEqual(LY.Iy,LX.Iy,'AbsTol',L.P.tol*L.Iy0,...
        'expected equal Iy when assigning')

      % Check that Iy are not equal when solving each slice
      L.P.Ipmin = 0;
      LY2 = fbtt(L,LX);
      testCase.assertGreaterThan(norm(LY2.Iy(:,:,1)-LX.Iy(:,:,1)),100,...
        'expected different Iy when solving')

    end
    
    function test_fbtxdisp_circuit(testCase)
      [L,LX] = fbt('ana',101,[],'circuit',true,'voltlim',true);
      LX = meqxk(LX,2);
      LX.gpad = 150;
      L.G.Vamax = repmat(200,L.G.na,1);
      L.G.Vamin = -L.G.Vamax;
      
      % Call without output
      testCase.assertWarningFree(@()fbtxdisp(L,LX));
      
      % Call with output
      s = fbtxdisp(L,LX);
      testCase.assertClass(s,'cell');
    end

    function test_limm(testCase,shot,margin,itert)
      % Test margin factor parameter limm for coil current combinations
      % test this for both itert==0 and itert~=0 to check constraint
      % activation logic
      
      tok = testCase.tokamak;
      
      L = fbt(tok,shot,0,'itert',itert); na = L.G.na;
      switch margin
        case 'default', limm=1;
        case 'margin', limm=0.5;
        case 'margin_single', limm=ones(na,1); limm(4)=0.2;
      end
      
      limu = 5e5*ones(na,1); liml = -limu;
      [L,LX,LY] = fbt(tok,shot,[],'limu',limu,...
                                  'liml',liml,...
                                  'limm',limm,...
                                  'debug',testCase.verbosity,...
                                  'izgrid',testCase.verbosity);
      
      % Verify FBT converged and solution is within prescribed limits
      testCase.assertEqual(limu,L.P.limu,'limu not equal to value passed in command line')
      testCase.assertEqual(liml,L.P.liml,'liml not equal to value passed in command line')
      testCase.assertTrue(meq_test.check_convergence(L,LX.t,LY),'FBT did not converge')
      testCase.verifyLessThan( L.P.limc*LY.Ia, L.P.limm.*L.P.limu + testCase.tol);
      testCase.verifyLessThan(-L.P.limc*LY.Ia,-L.P.limm.*L.P.liml + testCase.tol);
      
      if testCase.verbosity
        clf;
        subplot(121); meqplotfancy(L,LY);
        title(sprintf('limm=%s',margin),'interpreter','none');
        
        subplot(222); bar( L.P.limc*LY.Ia/1e3); hold on; plot(1:na,L.P.limm.*L.P.limu/1e3,'x');
        title('upper limit [kA]')

        subplot(224); bar(-L.P.limc*LY.Ia/1e3); hold on; plot(1:na,-L.P.limm.*L.P.liml/1e3,'x'); 
        title('lower limit [kA]')

        drawnow;
      end                           
    end
    
    function test_prescribevessel(testCase)
      [L,LX,LY1] = fbt('ana',1,[],'selu','e','nu',30);
      L.G = meqg(L.G,L.P,'Muu');
      testCase.assertTrue(all(LX.gpue(:)==0),'expected all Iu to be constrained for static FBT')

      Iu = L.G.Muu\(L.G.Tvu'*ones(L.G.nv,1)*0.1); % 0.1V inductive current
      LX.gpua = Iu;
      LY2 = fbtt(L,LX);
      testCase.assertEqual(LY2.Iu,Iu,'AbsTol',1e-9*L.Iu0,'LY.Iu does not match imposed value')

      if testCase.verbosity
        meqcompare(L,LY1,LY2)
      end
    end

    function test_novessel(testCase)
      % fbt should work without a vessel definition at all
      
      % Default run should give empty Iu and zero Iv
      myshot = 1; t=0; tok='ana';
      [L,~,LY] = fbt(tok,myshot,t);
      testCase.assertEqual(LY.Iv,zeros(L.G.nv,numel(t)))
      testCase.assertEqual(LY.Iu,zeros(     0,numel(t)))

      % Run without vessel should give empty Iv
      [~,~,LY] = fbt(tok,myshot,t,'selu','n','nv',0);
      testCase.assertEqual(LY.Iv,zeros(0,numel(t)));
      testCase.assertEqual(LY.Iu,zeros(0,numel(t)));
    end
    
    function test_direct(testCase,shot)
      % test ability of fbt to generate an equilibrium when all coil
      % currents and profile parameters are directly constrained, without
      % cost function or control points.
      tok = testCase.tokamak;
      
      bfct_ = @bfabmex;
      bfp_ = [1 2];
      agcon = {'Wk','qA'};
      fbtagcon = ['Ip',agcon];
      
      PP = {'selu','e','nu',20,... % add a vessel
        'bfct',bfct_,'bfp',bfp_};

      % get equilibrium
      [~,~,LY0] = fge(tok,shot,0,...
        'agcon',agcon,...
        'cde','OhmTor_rigid_0D',... % add CDE to have nonzero vessel currents
        'algoF','jfnk',... % use jfnk enforcing zero dz
        'tolF',1e-10,...
        'debug',1,'debugplot',0,...
        PP{:});
      
      L = fbt(tok,shot,0,PP{:},'fbtagcon',fbtagcon);
      
      % set up FBT to reproduce this exactly without constraint points
      % X = fbtgp(X,r,z,b,fa,fb,fe,br,bz,ba,be,cr,cz,ca,ce,vrr,vrz,vzz,ve,timeder);
      LX = struct();
      LX = fbtgp(LX,[],[],[],[],[],[],[],[],[],[],[],[],[],[],[],[],[],[]);
      
      % set other quantities needed
      LX.rBt = LY0.rBt;
      LX.qA  = LY0.qA;
      LX.Ip  = LY0.Ip;
      LX.Wk  = LY0.Wk;
      LX.Iy  = LY0.Iy; % initial condition
      
      % constrain coil currents
      LX.gpia = LY0.Ia;         LX.gpie = zeros(L.G.na,1); LX.gpid = 1e3;
      % constrain vessel currents
      LX.gpua = L.G.Tvu\LY0.Iv; LX.gpue = zeros(L.G.nu,1); LX.gpud = 1e3;
      
      % check inputs and run fbtt
      LX = fbtx(L,LX);
      LY = fbtt(L,LX);
      
      tol_ = 1e-10;
      testCase.assertTrue(any(LY.Iv),'want nonzero coil currents for this test')
      testCase.verifyEqual(LY.Ia,LY0.Ia,'AbsTol',tol_*L.Ia0);
      testCase.verifyEqual(LY.Iu,LY0.Iu,'AbsTol',tol_*L.Iu0);
      testCase.verifyEqual(LY.Fx,LY0.Fx,'AbsTol',tol_*L.Fx0);
      
    end
    
    function test_fbtx(testCase)
      tol_  = 1e-6;
      eqtol = 1e-11;
      %% Test for reordering
      [L,LX] = fbt('ana',1,0,'tol',tol_);
      LX = fbtgp(LX,1,1,0,[],[],[],NaN,NaN,NaN,0,[],[],[],[],[],[],[],[]);
      LX = fbtgp(LX,1,2,0,[],[],[],NaN,NaN,NaN,0,[],[],[],[],[],[],[],[]);
      LX = fbtgp(LX,1,1,0,[],[],[],NaN,NaN,NaN,0,[],[],[],[],[],[],[],[]);
      LY = fbtt(L,LX); % baseline
      LX = fbtx(L,LX);
      % check reordering
      testCase.verifyEqual(LY,fbtt(L,LX),'AbsTol',1e-12,...
        'fbt solutions with/without reordering are not the same');
      
      %% Test for eliminating redundant constraints
      excluded_fields = {'chie','res','resy','resC','resp','rese','resFx'};
      
      % If separate Br=0 Bz=0, merge and eliminate the second one
      [L,LX] = fbt('ana',1,0,'tol',tol_); % br bz ba be
      LX = fbtgp(LX,1,1,0,[],[],[],        0  ,NaN,NaN,0,[],[],[],[],[],[],[],[]);
      LX = fbtgp(LX,1,1,0,[],[],[],        NaN,0  ,NaN,0,[],[],[],[],[],[],[],[]);
      LX.gpbd = 1;
      LY = fbtt(L,LX); % baseline
      LX = fbtx(L,LX);
      % expect
      testCase.assertEqual(structcmp(LY,fbtt(L,LX),eqtol,excluded_fields),true,'fbt solutions with/without grouping are not the same');
      
      % If redundant Bz=0, eliminate the second one
      [L,LX] = fbt('ana',1,0,'tol',tol_); % br bz  ba be
      LX = fbtgp(LX,1,1,0,[],[],[],        0  ,0  ,NaN,0,[],[],[],[],[],[],[],[]);
      LX = fbtgp(LX,1,1,0,[],[],[],        NaN,0  ,NaN,0,[],[],[],[],[],[],[],[]);
      LX.gpbd = 1;
      % Disable warning for redundant constraints
      s = warning('off','ipm:RedundantEqConstraints');
      LY  = fbtt(L,LX); % baseline
      warning(s);
      LX = fbtx(L,LX);
      % expect
      testCase.assertEqual(structcmp(LY,fbtt(L,LX),eqtol,excluded_fields),true,'fbt solutions with/without grouping are not the same');
      
      % If not a constraint, don't eliminate it
      [L,LX] = fbt('ana',1,0,'tol',tol_); % br bz   ba  be
      LX = fbtgp(LX,1,1,1,[],[],[],        0  ,NaN ,NaN,1,[],[],[],[],[],[],[],[]);
      LX = fbtgp(LX,1,1,1,[],[],[],        NaN,0   ,NaN,0,[],[],[],[],[],[],[],[]);
      LX.gpbd = 1;
      % expect
      testCase.verifyEqual(LX,fbtx(L,LX),'LX changed while it should not have');
      
      % if not at same R,Z, don't eliminate it
      [L,LX] = fbt('ana',1,0,'tol',tol_); % br  bz  ba be
      LX = fbtgp(LX,2,1,0,[],[],[],        0  ,NaN,NaN,0,[],[],[],[],[],[],[],[]);
      LX = fbtgp(LX,1,1,0,[],[],[],        NaN,0  ,NaN,0,[],[],[],[],[],[],[],[]);
      LX.gpbd = 1;
      testCase.verifyEqual(LX,fbtx(L,LX),'LX changed while it should not have');
    end

    function test_itert(testCase,itert)
      % Check that when using itert the number of iterationsis is indeed fixed
      [~,~,LY] = fbt('ana',2,0:1e-3:1e-2,'itert',itert);
      testCase.assertTrue(~itert || all(LY.niter==itert),'expected niter=itert for itert>0')
    end
    
    function test_init(testCase)
      % Test providing initial condition to Picard method
      [L,LX,LY] = fbt('ana',2,0,'useSQP',false);
      testCase.verifyGreaterThan(LY.niter,1,'expected convergence to require more than 1 iteration when not providing initial condition')

      LX.Iy = LY.Iy; % initial Iy
 
      % re-solve
      LY2 = fbtt(L,LX);
      testCase.verifyEqual(LY2.niter,1,'expected to converge in 1 iteration when providing correct initial condition')
    end
    
    function test_MHy(testCase)
      % Test various methods of treating control points on computational grid
      
      [~,~ ,LY ] = fbt('ana',2,[],'MHyinterp',true);
      [L,LX,LY_] = fbt('ana',2,[],'MHyinterp',false);
      testCase.assertEqual(LY.Fx,LY_.Fx,'AbsTol',5e-3);
      
      % put a control point on the grid and check that the correct error is thrown
      ir = iround(L.ry,LX.gpr(1)); iz = iround(L.zy,LX.gpz(1));
      LX.gpr(1) = L.ry(ir); LX.gpz(1) = L.zy(iz);
      testCase.assertError(@() fbtt(L,LX),'FBT:ControlPointOnGrid') % solve fbt
    end

    function test_qpsolver(testCase)
      % run for various qpsolvers and check that we get the same answer

      shot_ = 2;
      L = fbt('ana',shot_); limu = 5e5*ones(L.G.na,1);
      PP = {'limu',limu,'liml',-limu}; % common parameters
      [~,~,LY0] = fbt('ana',shot_,[],PP{:},'qpsolver','ipm'); % default using MEQ's ipm

      maxtol = max(abs(LY0.Ia))*1e-5; % Global relative error of 1e-5
      % Test against ipm-optimized
      [~,~,LY1] = fbt('ana',shot_,[],PP{:},'qpsolver','ipm-optimized');
      testCase.assertEqual(LY1.Ia,LY0.Ia,'AbsTol',maxtol,'ipm and ipm-opzimized gave different solutions');

      % test vs quadprog if possible
      testCase.assumeFalse(isempty(which('quadprog')),'Skipping test w.r.t. quadprog due to missing optimization toolbox');
      [~,~,LY2] = fbt('ana',shot_,[],PP{:},'qpsolver','quadprog');
      testCase.assertEqual(LY2.Ia,LY0.Ia,'AbsTol',maxtol,'ipm and quadprog gave different solutions');
    end

    function test_verbosity(testCase)
      % add test cases to exercise various debugging plots/displays
      for debuglevel = 1:3
        [~,~,~] = fbt('ana',1,[],...
          'debug',debuglevel,'debugplot',debuglevel,...
          'niter',2); % only few iterations to save time
      end
    end

    function test_fbtx_defaults(testCase)
      % Test use fbtx defaults

      [L,LX] = fbt('ana',1,[],'circuit',true);
      
      % Remove some circuit equation related fields
      removed_fields = {'gpid','gpdd','gpod','gpud','gpad'};
      LX = rmfield(LX,removed_fields);
      LX = fbtx(L,LX);
      testCase.assertTrue(all(isfield(LX,removed_fields)),...
        'removed fields were not re-added')

      % add a time-derivative term only for currents
      nt = numel(LX.t);
      LX.g1id = ones(1,nt);
      LX.g1ie = ones(L.G.na,nt);
      LX.g1ia = zeros(L.G.na,nt);
      LX = fbtx(L,LX); % should add empty g1r,g1z,g1b
      testCase.assertTrue(all(isfield(LX,{'g1r','g1z','g1b'})))

    end

    function test_fbtgpclear(testCase)
      [~,LX] = fbt('ana',1,[0 0]);
      % Remove some control point related fields
      LX = fbtgpclear(LX); % Clear control points
      testCase.assertEqual(LX.gpr,zeros(0,numel(LX.t)))
    end

    function test_fbtxcat(testCase)
      [~,LX1] = fbt('ana',1);
      [~,LX2] = fbt('ana',2);
      % concatenate the slices
      LX = fbtxcat(LX1,LX2);
      np1 = numel(LX1.gpr); % number of control points for eq 1
      np2 = numel(LX2.gpr); % number of control points for eq 2

      for checked_fields = {'gpr','gpfa','gpvrz','gpbe'} % a selection to check
        fld = checked_fields{:};
        testCase.assertEqual(LX.(fld)(1:np1,1),LX1.(fld))
        testCase.assertEqual(LX.(fld)(1:np2,2),LX2.(fld))
      end
    end

    function test_deprecated_parameters(testCase)
      % Test adding a deprecated parameter
      fun = @() fbt('ana',1,[],'dissi',1e-12);

      testCase.verifyWarning(fun,'FBT:deprecatedP','Prescribing the deprecated dissi parameter did not trigger the correct error');
    end

    function test_algoNL(testCase,algoNL,useSQP)
      % Test different algoNL options with Picard and SQP
      [L,LX,LY] = fbt('ana',1,[],'algoNL',algoNL,'useSQP',useSQP);

      testCase.assertTrue(meq_test.check_convergence(L,LX.t,LY),...
        sprintf('FBT did not converge with algoNL=%s and useSQP=%d',algoNL,useSQP));
    end

    function test_algoGMRES(testCase,algoGMRES)
      % Test different GMRES options with SQP
      [L,LX,LY] = fbt('ana',1,[],'algoGMRES',algoGMRES);

      testCase.assertTrue(meq_test.check_convergence(L,LX.t,LY),...
        sprintf('FBT did not converge with algoGMRES=%s',algoGMRES));
    end

    function test_usepreconditioner(testCase,usepreconditioner)
      % Test use of Picard preconditioner for Block-GMRES method
      [L,LX,LY] = fbt('ana',1,[],'algoGMRES','qr','usepreconditioner',usepreconditioner);

      testCase.assertTrue(meq_test.check_convergence(L,LX.t,LY),...
        'FBT did not converge with algoGMRES=qr and a preconditioner');
    end
    
    function test_fgeF_failure(testCase)
      % Test failed fgeF call in fbt
      [L,LX] = fbt('ana',1);
      LX.Iy = zeros(L.nzy,L.nry); % Set Iy to 0 -> no magnetic axis
      LY = fbtt(L,LX);
      
      testCase.verifyFalse(meq_test.check_convergence(L,LX.t,LY),...
        'FBT converged when expected to fail');
    end
  end
end
