classdef doublet_tests < meq_test
  % Tests involving doublets
  %
  % [+MEQ MatlabEQuilibrium Toolbox+]

  %    Copyright 2022-2025 Swiss Plasma Center EPFL
  %
  %   Licensed under the Apache License, Version 2.0 (the "License");
  %   you may not use this file except in compliance with the License.
  %   You may obtain a copy of the License at
  %
  %       http://www.apache.org/licenses/LICENSE-2.0
  %
  %   Unless required by applicable law or agreed to in writing, software
  %   distributed under the License is distributed on an "AS IS" BASIS,
  %   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  %   See the License for the specific language governing permissions and
  %   limitations under the License.

  properties (TestParameter)
   % Deselect shot 68710@1.41 temp. since FBT eq is the same as 68466@2.43
   tok    = struct('TCV_droplet','tcv', 'ana_droplet','ana','ana_doublet','ana'); % ,'TCV_doublet','tcv'
   shot   = struct('TCV_droplet',68466, 'ana_droplet',   81,'ana_doublet',   82); % ,'TCV_doublet',68710
   t0     = struct('TCV_droplet', 2.43, 'ana_droplet',    0,'ana_doublet',    0); % ,'TCV_doublet', 1.41
  end
  
  properties
    verbosity = 0;
    bfp = [1 1]; 
    iterq = 0;
    agcon = {'Ip','bp'};
    agcon_cde = {'bp'};
    fbtagcon = {'Ip','Wk'};
  end
  
  methods(Test,TestTags={'fge'},ParameterCombination='sequential')
    function fbt_liu_fgs_fge(testCase,tok,shot,t0)
          
      [ok,msg] = meq_test.check_tok(tok,shot);
      testCase.assumeTrue(ok,msg);
      
      % ----------------- fbt starting point -------------------------------
      testCase.assumeTrue(~isequal(tok,'tcv') || ~isempty(which('mgp')),'skipping test because mgp is not available')
      [Lfbt,LXfbt] = fbt(tok,shot,t0,...
        'izgrid',true,...
        'fbtagcon',testCase.fbtagcon,...
        'icsint',true,'ilim',3,...
        'bfct',@bfabmex,'bfp',testCase.bfp,...
        'selx','X','tol',1e-7,...
        'debug',testCase.verbosity);
      LXfbt.bpD = [0.25;0.25;0];
      LXfbt = fbtx(Lfbt,LXfbt); % Necessary after manual LX update
      LYfbt = fbtt(Lfbt,LXfbt);
      
      if testCase.verbosity
        clf; meqplotfancy(Lfbt,LYfbt);
      end
      testCase.assertTrue(meq_test.check_convergence(Lfbt,LXfbt.t,LYfbt),'FBT did not converge')
     
      %% LIUQE from fbt
      nD = 3; % number of domain considered 
      ng = sum(testCase.bfp);
      wCo = zeros(ng*nD,1);
      wCo((nD-1)*ng+1:end) = Inf; % Exact constraints on third domains ag = 0;
      for useSQP = [false,true]
        inPar = liup_doublet('idoublet',1,'stabz',1,'wdz',[0;0;Inf],...
           'izgrid',true,'bfct',Lfbt.P.bfct,'bfp',Lfbt.P.bfp,...
           'selx','X','debug',testCase.verbosity,'useSQP',useSQP, ...
           'psichco',1e-7,'wCo', wCo);
        Lliu = liu(tok,shot,t0,inPar{:});
        
        minimal = true; % return minimal LXliu
        LXliu = liux(Lliu,meqxconvert(Lfbt, LYfbt, Lliu, minimal));
        LYliu = liut(Lliu, LXliu);
        
        testCase.assertTrue(meq_test.check_convergence(Lliu,LXliu.t,LYliu),sprintf('liu from fbt did not converge with useSQP=%d',useSQP))
        % Expect good agreement for low LIU tolerance (psichco) and low FBT residual
        %   The "ana_droplet" case has a high residual due to some
        %   oscillations in FBT
        reltol = 10*max(Lliu.P.psichco,LYfbt.res);
        msg = 'liu does not recover %3s fbt solution for noise free data with useSQP=%d';
        testCase.verifyEqual(LYliu.ag , LYfbt.ag             ,'AbsTol', reltol*max(Lfbt.ag0), sprintf(msg,'ag' ,useSQP));
        testCase.verifyEqual(LYliu.Bm , LYfbt.Bm             ,'AbsTol', reltol*Lfbt.P.b0    , sprintf(msg,'Bm' ,useSQP));
        testCase.verifyEqual(LYliu.IpD, LYfbt.IpD(1:Lliu.nD) ,'AbsTol', reltol*Lfbt.Ip0     , sprintf(msg,'IpD',useSQP));
        
        if testCase.verbosity>1
          clf; meqplotliu(Lliu, LXliu,LYliu);
        end
      end
      
      %% FGS from fbt
      PP = {'bfct',@bfabmex,'bfp',testCase.bfp,...
        'iterq',testCase.iterq,...
        'idoublet',true,...
        'izgrid',true}; % common params
      
      [Lfgs] = fgs(tok,shot,t0,'insrc','fbt',PP{:},...
                'agcon',testCase.agcon...
                );
      LXfgs = meqxconvert(Lfbt,LYfbt,Lfgs);
      LYfgs = fgst(Lfgs,LXfgs,'debug',testCase.verbosity,'mkryl',150,'tolF',1e-8);
      testCase.assertTrue(meq_test.check_convergence(Lfgs,LXfgs.t,LYfgs),'FGS did not converge')
   
      %% LIUQE from FGS TODO
  
      %% FGE with fixed Ip from FGS
      t = t0 + (0:1e-4:5e-3);
      [Lfge] = fge(tok,shot,t,PP{:},...
        'agcon',testCase.agcon...
        );
      LXfge = fgex(testCase.tok,t,Lfge,meqxconvert(Lfgs,LYfgs,Lfge));
      LXfge.Va(1,:) = LXfge.Va(1,:) + randn(size(LXfge.t));

      LYfge = fget(Lfge,LXfge,'debug',testCase.verbosity);
      testCase.assertTrue(meq_test.check_convergence(Lfge,t,LYfge),'FGE did not converge')
      
      if testCase.verbosity>1
        meqplotevo(Lfge,LYfge); drawnow;
      end
      
      %% FGE with CDE from FGS
      [LfgeCDE] = fge(tok,shot,t,PP{:},...
       'agcon',testCase.agcon_cde,...
       'cde','cde_ss_0D',...
       'ssinit',true... % start with stationary initial condition
       );
      
      LXfgeCDE = fgex(testCase.tok,t,LfgeCDE,meqxconvert(Lfgs,LYfgs,LfgeCDE));
      LYfge = fget(LfgeCDE,LXfgeCDE,'debug',testCase.verbosity);
      testCase.assertTrue(meq_test.check_convergence(LfgeCDE,t,LYfge),'FGE did not converge')
      
      if testCase.verbosity>1
       meqplotevo(Lfge,LYfge); drawnow;
      end
      
      if testCase.verbosity>1
        clf;
        subplot(121);
        meqplotfancy(Lfge ,meqxk(LXfge ,1),'vacuum',true); title('eq. field - no Iv');
        subplot(122);
        meqplotfancy(LfgeCDE,meqxk(LXfgeCDE,1),'vacuum',true); title('eq field - w Iv');
        drawnow;
      end

    end
  end
end
