classdef bf_tests < meq_test
  % Tests for basis functions
  %
  % These tests check that for a given set the results of the bf calls with different modes are consistent
  %
  % See also bf_comp_tests bf_jacobian_test bf_extra_tests
  %
  % [+MEQ MatlabEQuilibrium Toolbox+]

  %    Copyright 2022-2025 Swiss Plasma Center EPFL
  %
  %   Licensed under the Apache License, Version 2.0 (the "License");
  %   you may not use this file except in compliance with the License.
  %   You may obtain a copy of the License at
  %
  %       http://www.apache.org/licenses/LICENSE-2.0
  %
  %   Unless required by applicable law or agreed to in writing, software
  %   distributed under the License is distributed on an "AS IS" BASIS,
  %   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  %   See the License for the specific language governing permissions and
  %   limitations under the License.

  properties
    tol = 1e-2;
    verbosity = 0;
    n = 41; % number of FN points

    % parameters for dummy test equilibrium
    r0 = 1;
    rBt = 1.4;
    
    FN; L; LY;
    GN; IGN;
    bfname; bfh; bfp;
    alpg; alg;
  end
  
  properties (ClassSetupParameter)
    % Various combinations of boundary and axis flux
    FAs    = {-1   0   2  0.5 };
    FBs    = {0    1   0  0   };
  end
  
  properties (MethodSetupParameter)
    bf     = {'bffbt' 'bfabmex' 'bf3pmex' 'bf3imex' 'bfefmex' 'bfsp_a' 'bfsp_n' 'bfsp_z' 'bfsp_ns' 'bfsp_zs' 'bfgenD'};
  end
  
  methods(TestClassSetup,ParameterCombination='sequential')
    
    function gen_Fx(testCase,FAs,FBs)
      r0 = testCase.r0; z0=0; rBt = testCase.rBt; %#ok<*PROPLC>
      L = liu('ana',1);
      
      [testCase.L,testCase.LY] = testCase.getCircularEquilibrium(L,r0,z0,FAs,FBs,rBt);
      testCase.FN = linspace(0,1,testCase.n)';
     end
    
  end
  
  methods(TestMethodSetup,ParameterCombination='sequential')

    function bfh_setup(testCase,bf)
      % Cannot set these values as properties if calling meq functions (not
      % necessarily available in the path when initialising the class)
      testCase.bfname = bf; % save name of case
      switch bf
        case 'bffbt',   bfpar = [1.1 .25 1.5 2.2 .75 2.5]';
        case 'bfab',    bfpar = [3 1];
        case 'bfabmex', bfpar = [3 1];
        case 'bf3pmex', bfpar = false;
        case 'bf3imex', bfpar = bfp3imex();
        case 'bfefmex', bfpar = [4 4];
        case 'bfsp_a',  bfpar = bfpsp_(7,11,'a'); bf = 'bfsp';
        case 'bfsp_n',  bfpar = bfpsp_(7,11,'n'); bf = 'bfsp';
        case 'bfsp_z',  bfpar = bfpsp_(7,11,'z'); bf = 'bfsp';
        case 'bfsp_ns', bfpar = bfpsp_(2,3,'n'); bf = 'bfsp'; % s for short, testing 2, 3 basis functions
        case 'bfsp_zs', bfpar = bfpsp_(2,3,'z'); bf = 'bfsp'; % s for short, testing 1, 2 basis functions
        case 'bfgenD',  bfpar = bfpgenD({@bfabmex,[3 1],1},1);
        otherwise,      error('Basis function %s not handled yet',bf);
      end
      testCase.bfp = bfpar;
      testCase.bfh = str2func(bf); % function handle
      LY = testCase.LY;

      % Call basic basis function evaluator
      [testCase.GN ,testCase.IGN ] = testCase.bfh(2,bfpar,testCase.FN,LY.FA,LY.FB);
      
      % Conversion factors (See chapter 2 of the MEQ Redbook for definitions of alpg (alpha'_g) and alg (alpha_g))
      FBA = LY.FB - LY.FA;
      switch bf
        case {'bfab','bfabmex'}
          testCase.alpg = [FBA.^(1:bfpar(1)),FBA.^(1:bfpar(2))];
          testCase.alg  = testCase.alpg*FBA;
        case 'bf3pmex'
          testCase.alpg = [FBA,FBA,FBA.^2];
          testCase.alg  = testCase.alpg*FBA;
        case 'bfefmex'
          testCase.alpg = [FBA.^(0:bfpar(1)-1),FBA.^(0:bfpar(2)-1)];
          testCase.alg  = testCase.alpg*FBA;
        case {'bffbt','bf3imex','bfsp'}
          testCase.alpg = ones(1,size(testCase.GN,2));
          testCase.alg  = testCase.alpg*FBA;
        case 'bfgenD'
          testCase.alpg = [FBA.^(1:bfpar.bfp{1}(1)),FBA.^(1:bfpar.bfp{1}(2))];
          testCase.alg  = testCase.alpg*FBA;
        otherwise
          error('Basis function %s not handled yet',bf);
      end
          
          
    end
  end 

  methods(Test, TestTags = {'Unit','bf'})

    function test_unknown_mode(testCase)
      % Verify unknown mode throws error
      testCase.verifyError(@() testCase.bfh(-1,testCase.bfp),[func2str(testCase.bfh),':unknownmode']);
    end
   
    function test_factors(testCase)
      % simple test of factors
      [FP,FT] = testCase.bfh(0,testCase.bfp);
      testCase.assertTrue(all(FP==1 | FP==0),'FP must be 1 or 0')
      testCase.assertTrue(all(FT==1 | FT==0),'FT must be 1 or 0')
      testCase.assertEqual(size(FP),size(FT),'FP,FT unequal size')
    end
    
    function test_on_rz_grid(testCase)
      L  = testCase.L ;
      LY = testCase.LY;
      % Test evaluating basis function and its integrals on rz grid
      [TYG,TPDG,ITPDG] = testCase.bfh(1,testCase.bfp, ...
        LY.Fx,LY.FA,LY.FB,LY.Opy,L.ry,L.iry);
      
      ny = numel(L.ry)*numel(L.zy); % number of grid points
      ng = size(testCase.GN,2); % number of basis functions
      
      % test sizes
      testCase.verifyEqual(size(TYG)  ,[ny,ng], 'invalid size for TYG')
      testCase.verifyEqual(size(TPDG) ,[1,ng] , 'invalid size for TPDG')
      testCase.verifyEqual(size(ITPDG),[1,ng] , 'invalid size for ITPDG')
      
      % Compare to values obtained with mode 2
      FBA = LY.FB-LY.FA;
      FNy = (LY.Fx(2:end-1,2:end-1)-LY.FA)/FBA.*(LY.Opy>0);
      % Get values of normalized bf
      [gNyg,IgNyg] = testCase.bfh(2,testCase.bfp,FNy(:),LY.FA,LY.FB);
      % Convert to physical bf
      gy  = testCase.alpg.* gNyg;
      Igy = testCase.alg .*IgNyg;
      % Compute Tyg and ITyg
      [fPg,fTg] = testCase.bfh(0,testCase.bfp);
      rry  = reshape(repmat(L.ry.' ,L.nzy,1),L.ny,1);
      irry = reshape(repmat(L.iry.',L.nzy,1),L.ny,1);
      Tyg_   = (fPg.'.*rry+fTg.'.*irry).* gy.*(LY.Opy(:)>0);
      ITyg_  = (fPg.'.*rry+fTg.'.*irry).*Igy.*(LY.Opy(:)>0);
      % Now compute integrals per domain
      nD = numel(LY.FA);
      TpDg_  = zeros(nD,ng);
      ITpDg_ = zeros(nD,ng);
      for iD = 1:nD
        masky = LY.Opy(:)==iD;
        TpDg_ (iD,:) = sum( Tyg_(masky,:),1);
        ITpDg_(iD,:) = sum(ITyg_(masky,:),1);
      end
      % Compare
      tol = 1e-8;
      testCase.verifyEqual(  TYG,  Tyg_,'AbsTol',tol,'Tyg from mode 1 and mode 2 do not match');
      testCase.verifyEqual( TPDG, TpDg_,'AbsTol',tol,'TpDg from mode 1 and mode 2 do not match');
      testCase.verifyEqual(ITPDG,ITpDg_,'AbsTol',tol,'ITpDg from mode 1 and mode 2 do not match');

    end
    
    function test_rz_integral_value_analytical(testCase)
      L  = testCase.L ;
      LY = testCase.LY;
      % test values of integrals of basis functions and primitives
      % for cases where we can reasonably compute this analytically.
      rr = testCase.r0*ones(size(L.ry));
      [~,TPDG,~] = testCase.bfh(1,testCase.bfp, ...
        LY.Fx,LY.FA,LY.FB,LY.Opy,rr,rr);
      
      qa0 = (LY.rB-LY.rA).^2 + (LY.zB-LY.zA).^2; % Plasma minor radius
      
      switch func2str(testCase.bfh)
        case 'bf3imex'
          dS = mean(diff(L.ry)).*mean(diff(L.zy));
          if testCase.bfp.fPg(1) && ~testCase.bfp.fTg(1)
            % first basis function: linear -> quadratic in R: paraboloid
            testCase.verifyEqual(TPDG(1),pi*qa0/2/dS,'RelTol',1e-2); % paraboloid volume
          end
          if testCase.bfp.fPg(2) && ~testCase.bfp.fTg(2)
            % second basis function: sqrt -> linear in R: cone
            testCase.verifyEqual(TPDG(2),pi*qa0/3/dS,'RelTol',1e-2); % cone volume
          end
          if testCase.bfp.fPg(3) && ~testCase.bfp.fTg(3)
            % third basis function: constant -> cylinder
            testCase.verifyEqual(TPDG(3),pi*qa0/dS  ,'RelTol',1e-2); % cylinder volume
          end
        otherwise
          % no analytical value for this case
          testCase.assumeFail(sprintf('No analytical value for RZ integral of bfct=%s',func2str(testCase.bfh)))
      end
    end
    
    function test_profile_fitting(testCase)
      %% Test that fitting ag for given profiles
      % Test that when fitting p, T profiles using ag, 
      % we get back the same ag as originally used
      L  = testCase.L ;
      LY = testCase.LY;
      
      % Call bf(3,.. to get scaling coefficients for p', TT' profiles
      L.pQ = sqrt(linspace(0,1,41)'); %psiN grid
      L.idsx = 0.01.^2; % ficticious 1/|dS|
      L.bfct = testCase.bfh;
      L.bfp  = testCase.bfp;
      L.nD = 1;
      
      [L.fPg,L.fTg,L.TDg] = L.bfct(0,L.bfp);
      ag = 2*(L.fPg | L.fTg); % coefficients representing basis functions
      L.ng = numel(ag);
      
      % fitting
      rBt = 1; % dummy value used for T
      smalldia = 0; % no small diamagnetism approximation
      
      [PpQ,TTpQ] = meqprof(L.fPg,L.fTg,ag,testCase.FN,...
        LY.FA,LY.FB,rBt,L.bfct,L.bfp,L.idsx,smalldia);
      
      if testCase.verbosity
        subplot(211)
        plot(L.pQ.^2,PpQ)
        subplot(212)
        plot(L.pQ.^2,TTpQ)
      end
      
      % fit using meqfitprof
      agfit = meqfitprof(L,L.pQ.^2,LY.FA,LY.FB,PpQ,TTpQ);
      
      % check for same ag
      testCase.verifyEqual(ag,agfit,'AbsTol',testCase.tol)
      
    end

    function test_chord_integral(testCase)
      % Test evaluating integral along Z of basis function
      L  = testCase.L ;
      LY = testCase.LY;
      
      rr = mean(L.ry); % r value for z integral evaluation
      kd = find(L.ry<rr,1,'last')-1; % indices into rx 0-based
      fd = abs(rr-L.ry(kd+(1:2)))/L.drx; % linear interpolation coefficients
      nd = size(kd,1);
      
      switch func2str(testCase.bfh)
        case {'bfab','bfabmex','bf3imex','bfefmex','bfsp','bffbt','bfgenD'}
          [Tdg,Tgy] = testCase.bfh(4,testCase.bfp,...
            LY.Fx,LY.FA,LY.FB,LY.Opy,uint32(kd),fd);
        otherwise
          testCase.assumeFail(sprintf('No tests defined yet for mode=4 for %s',func2str(testCase.bfh)))
      end
      
      % Compare with computation based on mode 1
      [fPg,fTg] = testCase.bfh(0,testCase.bfp);
      Tyg = testCase.bfh(1,testCase.bfp,LY.Fx,LY.FA,LY.FB,LY.Opy,L.ry,L.iry);
      rry  = reshape(repmat(L.ry.' ,L.nzy,1),L.ny,1);
      ng = numel(fPg);
      Tgy_ = squeeze(sum(reshape(Tyg./(fPg.'.*rry + fTg.'./rry),[L.nzy,L.nry,ng]),1)).';
      Tdg_ = (Tgy_(:,kd+1).*fd(1,:) + Tgy_(:,kd+2).*fd(2,:)).';
      
      testCase.verifyEqual(size(Tgy), [ng,L.nry], 'invalid size for Tgy');
      testCase.verifyEqual(Tgy, Tgy_, 'Reltol', testCase.tol, ...
        sprintf('Vertical integrals mismatch between mode=4 and mode=1 for %s',func2str(testCase.bfh)));
      testCase.verifyEqual(size(Tdg), [nd,ng], 'invalid size for Tdg');
      testCase.verifyEqual(Tdg, Tdg_, 'Reltol', testCase.tol, ...
        sprintf('Vertical integrals mismatch between mode=4 and mode=1 for %s',func2str(testCase.bfh)));
    end
    
    function test_axis_value(testCase)
      % Test axis value of basis function and its primitive
      LY = testCase.LY;
      
      [G1,IG1] = testCase.bfh(5,testCase.bfp,[],LY.FA,LY.FB);
      
      % Get conversion factors for normalised quantities
      GN1 = G1./testCase.alpg;
      IGN1 = IG1./testCase.alg;
      
      % Dummy test with bfh(2,..) which should give same result
      [GN ,IGN ] = testCase.bfh(2,testCase.bfp,0,LY.FA,LY.FB);
      %
      testCase.verifyEqual( GN, GN1, 'Abstol',   testCase.tol, ...
        sprintf('%s: value at the axis mismatch for [FA,FB]=%3.3f, %3.3f',...
        func2str(testCase.bfh),LY.FA,LY.FB));
      testCase.verifyEqual(IGN,IGN1, 'Abstol',   testCase.tol, ...
        sprintf('%s: integral value at the axis mismatch for [FA,FB]=%3.3f, %3.3f',...
        func2str(testCase.bfh),LY.FA,LY.FB));
    end
   
    function test_regularisation(testCase)
      L  = testCase.L ;
      LY = testCase.LY;
      % Test computation of elements for the regularisation equations
      rA = testCase.r0;
      args = {[],LY.FA,LY.FB,rA,1./rA,L.dsx}; % arguments common to all bfs
      [Qqg,Xq] = testCase.bfh(6,testCase.bfp,args{:});
      switch func2str(testCase.bfh)
        case 'bf3pmex'
          [Qqg0,Xq0] = bfabmex(6,[1,2] - testCase.bfp,args{:});
          testCase.verifyEqual(Qqg,Qqg0, 'Abstol', testCase.tol, ...
            'regularisation matrix Qqg mismatch between bf3pmex and bfabmex with bfp=[1,2]');
          testCase.verifyEqual(Xq,Xq0, 'Abstol', testCase.tol, ...
            'regularisation value Xq mismatch between bf3pmex and bfabmex with bfp=[1,2]');
        case 'bfefmex'
          [Qqg0,Xq0] = bfabmex(6,max(testCase.bfp - 1,0),args{:});
          nP = testCase.bfp(1); nT = testCase.bfp(2); ng = nP + nT;
          mask = true(ng,1);
          if nP > 0, mask(1)    = false;end
          if nT > 0, mask(1+nP) = false;end
          testCase.verifyEqual(Qqg(1:end-2,mask),Qqg0, 'Abstol', testCase.tol, ...
            'regularisation matrix Qqg mismatch between bfefmex and bfabmex');
          testCase.verifyEqual(Xq(1:end-2),Xq0, 'Abstol', testCase.tol, ...
            'regularisation value Xq mismatch between bfefmex and bfabmex');
        case 'bf3imex'
          testCase.verifyEqual(Qqg,[0,0,0],sprintf('regularisation matrix Qqg should be [0,0,0] for %s',func2str(testCase.bfh)));
          testCase.verifyEqual( Xq,     0 ,sprintf('regularisation value Xq should be 0 for %s',func2str(testCase.bfh)));
        case 'bffbt'
          testCase.verifyEmpty(Qqg,sprintf('regularisation matrix Qqg should be empty for %s',func2str(testCase.bfh)));
          testCase.verifyEmpty( Xq,sprintf('regularisation value Xq should be empty for %s',func2str(testCase.bfh)));
        otherwise
          testCase.assumeFail(sprintf('No tests defined yet for mode=6 for %s',func2str(testCase.bfh)))
      end
    end
    
    function test_toroidal_field(testCase)
      % Test computation of toroidal field on y grid as computed by bf(8,...)
      % [BTY             ] = BF(8,PAR, F,FA,FB,O ,  A,RBT,IDS,IRY)
      L  = testCase.L ;
      LY = testCase.LY;
      % arguments common to all bfs
      idS = L.idsx;
      Fx  = LY.Fx;
      FA  = LY.FA;
      FB  = LY.FB;
      Opy = LY.Opy;
      rBt = 0.88*1.43;
      iry = L.iry;
      switch testCase.bfname
        case {'bf3pmex','bf3imex','bffbt'}
          ag = 1e4*rand(3,1);
        case {'bfab','bfabmex','bfgenD'}
          ag = 1e4*rand(4,1);
        case {'bfefmex'}
          ag = 1e4*rand(8,1);
        case 'bfsp_a'
          ag = 1e4*rand(7+11,1);
        case 'bfsp_n'
          ag = 1e4*rand(7+11,1);
        case 'bfsp_z'
          ag = 1e4*rand(6+10,1);
        case 'bfsp_ns'
          ag = 1e4*rand(2+3,1);
        case 'bfsp_zs'
          ag = 1e4*rand(1+2,1);
        otherwise
          testCase.assumeFail(sprintf('No tests defined yet for mode=8 for %s',func2str(testCase.bfh)))
      end
      Bty = testCase.bfh(8,testCase.bfp,Fx,FA,FB,Opy,ag,rBt,idS,iry);
      
      % Compare with computation based on mode 2 and 3
      FN = (Fx - FA)/(FB - FA);
      [fPg,fTg] = testCase.bfh(0,testCase.bfp);
      [~,IGN] = testCase.bfh(2,testCase.bfp,FN(2:end-1,2:end-1),FA,FB);
      [~,~,~,AHQT] = testCase.bfh(3,testCase.bfp,ag,FA,FB,fPg,fTg,idS);
      hqTx = reshape(IGN*AHQT,size(Opy)).*logical(Opy);
      Bty_ = (rBt+hqTx/rBt).*iry.'; % Using small diamagnetism approx.
      
      testCase.verifyEqual(Bty,Bty_, 'Reltol', testCase.tol, ...
        sprintf('Toroidal field mismatch between mode=8 and mode=2/3 for %s',func2str(testCase.bfh)));
      
    end
      
    function test_bfpr_integral(testCase)
      % test integral as returned by bf(2,..) and bfprmex and cumtrapz
      
      [nQ,ng] = size(testCase.GN);
      FN = linspace(0,1,nQ).';
      GN = testCase.GN; %#ok<*PROP>
      IGN = testCase.IGN;
      IGN2 = zeros(nQ,ng);

      % Compute integral using bfprmex
      for ii = 1:3:ng
        % bfpr takes exactly 3 basis functions
        sub = ii:min(ii+2,ng);
        GN_ = GN(:,sub);
        nsub = numel(sub);
        if nsub<3
          GN_(:,nsub+1:3) = 0;
        end
        IGN_ = bfprmex(GN_);
        IGN2(:,sub) = IGN_(:,1:nsub);
      end
      
      % Compute integral using cumtrapz
      IGN3 = cumtrapz(FN,GN); IGN3 = IGN3 - IGN3(end,:);

      % Compare them
      testCase.verifyEqual(IGN3,IGN,   'AbsTol',  testCase.tol, ...
        sprintf('%s(2,...) does not match cumtrapz()',func2str(testCase.bfh)));
      testCase.verifyEqual(IGN3,IGN2,  'AbsTol',  testCase.tol, ...
        'bfprmex() does not match cumtrapz()'   );
      
      if testCase.verbosity
        figure
        plot(testCase.FN,GN,'-k',testCase.FN,IGN,'-xk',testCase.FN,IGN2,'-+r',testCase.FN,IGN3,'-ob')
        title(char(testCase.bfh))
        shg
      end
    end
    
    function test_physical_basis_functions(testCase)
      % Compare result of bf(91,..) with bf(2,..)
        
      LY = testCase.LY;
      
      F = LY.Fx(:);
      FA = LY.FA;
      FB = LY.FB;
      % Extract values within FA and FB
      FN = (F - LY.FA)/(LY.FB - LY.FA);
      mask = FN>=0 & FN<=1;
      F = F(mask);
      FN = FN(mask);
      
      % Direct evaluation of value and primitive at F
      [G1,IG1] = testCase.bfh(91,testCase.bfp,F,FA,FB);
      
      % Get conversion factors for normalised quantities
      GN1 = G1./testCase.alpg;
      IGN1 = IG1./testCase.alg;
      
      % Call bfh(2,..) which should give same result
      [GN ,IGN ] = testCase.bfh(2,testCase.bfp,FN,FA,FB);
      
      % Compare results
      testCase.verifyEqual( GN, GN1, 'Abstol',   testCase.tol, ...
        sprintf('%s: bf(91,..) values mismatch for [FA,FB]=%3.3f, %3.3f',...
        func2str(testCase.bfh),FA,FB));
      testCase.verifyEqual(IGN,IGN1, 'Abstol',   testCase.tol, ...
        sprintf('%s: bf(91,..) integral values mismatch for [FA,FB]=%3.3f, %3.3f',...
        func2str(testCase.bfh),FA,FB));
    end
    
    function test_physical_basis_function_derivatives(testCase)
      % Compare result of bf(92,..) with bf(11,..)

      L  = testCase.L;
      LY = testCase.LY;
      
      Fy = LY.Fx(L.lxy);
      FA = LY.FA;
      FB = LY.FB;
      
      % Direct evaluation of value and primitive derivatives w.r.t. Fx-FA, FB-FA
      [dg1,dg2,dg3,~,dIg2,dIg3] = testCase.bfh(92,testCase.bfp,Fy,FA,FB);
      
      % Compute Tyg and ITyg
      [fPg,fTg] = testCase.bfh(0,testCase.bfp);
      rry  = L.rry(:);
      irry = 1./rry;
      dTyg1_   = (fPg.'.*rry+fTg.'.*irry).* dg1.*(LY.Opy(:)>0);
      dTyg2_   = (fPg.'.*rry+fTg.'.*irry).* dg2.*(LY.Opy(:)>0);
      dTyg3_   = (fPg.'.*rry+fTg.'.*irry).* dg3.*(LY.Opy(:)>0);
      dITyg2_  = (fPg.'.*rry+fTg.'.*irry).*dIg2.*(LY.Opy(:)>0);
      dITyg3_  = (fPg.'.*rry+fTg.'.*irry).*dIg3.*(LY.Opy(:)>0);
      
      % Call bfh(11,..) which should give same result
      [dTyg1 ,dTyg2, dTyg3, dITyg2, dITyg3] = testCase.bfh(11,testCase.bfp,LY.Fx,FA,FB,LY.Opy,L.ry,L.iry);
      
      % Compare results
      testCase.verifyEqual( dTyg1, dTyg1_, 'Abstol',   testCase.tol, ...
        sprintf('%s: bf(92,..) dTygdFy values mismatch for [FA,FB]=%3.3f, %3.3f',...
        func2str(testCase.bfh),FA,FB));
      testCase.verifyEqual( dTyg2, dTyg2_, 'Abstol',   testCase.tol, ...
        sprintf('%s: bf(92,..) dTygdFA values mismatch for [FA,FB]=%3.3f, %3.3f',...
        func2str(testCase.bfh),FA,FB));
      testCase.verifyEqual( dTyg3, dTyg3_, 'Abstol',   testCase.tol, ...
        sprintf('%s: bf(92,..) dTygdFB values mismatch for [FA,FB]=%3.3f, %3.3f',...
        func2str(testCase.bfh),FA,FB));
      testCase.verifyEqual(dITyg2,dITyg2_, 'Abstol',   testCase.tol, ...
        sprintf('%s: bf(92,..) dITygdFA values mismatch for [FA,FB]=%3.3f, %3.3f',...
        func2str(testCase.bfh),FA,FB));
      testCase.verifyEqual(dITyg3,dITyg3_, 'Abstol',   testCase.tol, ...
        sprintf('%s: bf(92,..) dITygdFB values mismatch for [FA,FB]=%3.3f, %3.3f',...
        func2str(testCase.bfh),FA,FB));
    end
  end
end

%% Auxiliary functions
function bfp = bfp3imex()
% parameters for bf3imex test (direct specification of basis functions)
n=41;
FN = linspace(0,1,n);
GN = [linspace(1,0,n)',1-sqrt(linspace(0,1,n)'),linspace(1,1,n)'];
IGN = flipud(cumtrapz(flipud(FN'),flipud(GN)));

FP = [1;0;0];
FT = [0;1;1];
bfp = struct('gNg',GN,'IgNg',IGN,'fPg',FP,'fTg',FT);
end

function bfp = bfpsp_(nP, nT, ec)
% parameters for bfsp test (B-spline basis functions)
tPg = linspace(0,1,nP);
tTg = linspace(0,1,nT);
bfp = bfpsp(tPg,tTg,ec);
end
