classdef GS_vacuum_analytic_test < meq_test
% Test GS_homogeneous
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.
  
  properties
    verbosity = 0;
  end
  
  properties(TestParameter)
    % various coefficients for homogeneous solution
    c = {[0 1 0 0 0],[0 0 0 0 1],[0 0 0 0 0],[0 3 0 0 -2]};
    d = {[0 0 0 0]  ,[0 0 0 0]  ,[0 1 0 0  ],[0 -3 0 0  ]};
    scal = {1,2,-1,0.5}
  end

  methods(TestClassSetup)
    function setup_path(testCase)
      import matlab.unittest.fixtures.PathFixture
      meqpath = fullfile(fileparts(mfilename('fullpath')),'..');
      scriptspath = fullfile(meqpath,'scripts');
      testCase.applyFixture(PathFixture(scriptspath));
    end
  end
  
  methods(Test,TestTags={'Unit'},ParameterCombination = 'sequential')
    function test_analytical_GS(testCase)
      % Check that symbolic expression results match function handles
      % Prefer not to use symbolic expressions directly in GS_homogeneous
      % for those who don't have the symbolic toolbox.

      testCase.assumeTrue(~isempty(ver('symbolic')),'Symbolic toolbox does not seem to be installed')
      c_ = rand(1,5); d_=rand(1,4); s_ = 1; % scale parameter
      R = [1;1]; Z=[1;-2]; % R,Z for evaluation

      % symbolic expressions
      [psi,psir,psiz,psirr,psirz,psizz,psirrr,psirrz,psirzz,psizzz] = GS_symbolic();

      % function handles
      [psi_,Br_,Bz_,~,H_,~,M_] = GS_vacuum_analytic(c_,d_,s_);
      Hval = H_(R,Z); Mval = M_(R,Z);

      cdargs = num2cell([c_,d_]); % c,d arguments
      f_psi    = matlabFunction(psi);       psival = s_*   f_psi(R,Z,cdargs{:});
      f_psir   = matlabFunction(psir);     psirval = s_*  f_psir(R,Z,cdargs{[2,3,4,5,7,8,9]});
      f_psiz   = matlabFunction(psiz);     psizval = s_*  f_psiz(R,Z,cdargs{[3,4,5,6,7,8,9]});
      f_psirr  = matlabFunction(psirr);   psirrval = s_* f_psirr(R,Z,cdargs{[2,3,4,5,7,8,9]});
      f_psirz  = matlabFunction(psirz);   psirzval = s_* f_psirz(R,Z,cdargs{[3,5,7,8,9]});
      f_psizz  = matlabFunction(psizz);   psizzval = s_* f_psizz(R,Z,cdargs{[3,4,5,8,9]});
      f_psirrr = matlabFunction(psirrr); psirrrval = s_*f_psirrr(R,Z,cdargs{[3,4,5,8,9]});
      f_psirrz = matlabFunction(psirrz); psirrzval = s_*f_psirrz(R,Z,cdargs{[3,5,7,8,9]});
      f_psirzz = matlabFunction(psirzz); psirzzval = s_*f_psirzz(R,Z,cdargs{[3,5,8]});
      f_psizzz = matlabFunction(psizzz); psizzzval = s_*f_psizzz(R,Z,cdargs{[5,8,9]});

      testCase.verifyEqual(psival,psi_(R,Z),               'AbsTol',1e-12);
      testCase.verifyEqual(psirval,+2*pi*R.*Bz_(R,Z),      'AbsTol',1e-12);
      testCase.verifyEqual(psizval,-2*pi*R.*Br_(R,Z),      'AbsTol',1e-12);
      testCase.verifyEqual(psirrval,squeeze(Hval(1,1,:)),  'AbsTol',1e-12);
      testCase.verifyEqual(psirzval,squeeze(Hval(1,2,:)),  'AbsTol',1e-12);
      testCase.verifyEqual(psizzval,squeeze(Hval(2,2,:)),  'AbsTol',1e-12);
      testCase.verifyEqual(psirrrval,squeeze(Mval(1,1,1,:)),   'AbsTol',1e-12);
      testCase.verifyEqual(psirrzval,squeeze(Mval(1,1,2,:)),   'AbsTol',1e-12);
      testCase.verifyEqual(psirzzval,squeeze(Mval(1,2,2,:)),   'AbsTol',1e-12);
      testCase.verifyEqual(psizzzval,squeeze(Mval(2,2,2,:)),   'AbsTol',1e-12);
    end
   
    function test_vacuum_fields(testCase,c,d,scal)
      
      %% run numerical tests for various vacuum field shapes
      Rgrid = linspace(0.3,1,101);
      Zgrid = linspace(-0.5,0.5,81);
      [psi,Br,Bz,absBp,H,gradB] = GS_vacuum_analytic(c,d,scal);

      [RR,ZZ] = meshgrid(Rgrid,Zgrid);
      
      % check physics
      R0 = mean(Rgrid); Z0 = mean(Zgrid);
      gradB0 = gradB(R0,Z0);
      Br0 = Br(R0,Z0);
      divB0 = (Br0./R0 + sum(diag(gradB0))); %div(B) = 1/r*d/dr(rBr) + dBz/dz
      mu0jphi0 = gradB0(2,1) - gradB0(1,2); % mu0*jphi = curl(B) = Brdz - Bzdr
      
      assert(mu0jphi0<1e-10,'jphi is not 0 - not a vacuum solution')
      assert(divB0<1e-10,'divB is not 0')
      
      if testCase.verbosity
        subplot(121); cla;
        contourf(Rgrid,Zgrid,psi(RR,ZZ),21); axis equal;
        colorbar('location','southoutside')
        title('psi')
        
        subplot(122); cla;
        contourf(Rgrid,Zgrid,absBp(RR,ZZ),21); axis equal;
        colorbar('location','southoutside')
        title('|B_p|');
        drawnow;
      end
    end
  end
end
