function [x, gmres_iter] = solve_linear_problem(A, b, x0, Pinv, mask, Adiag, P)
% SOLVE_LINEAR_PROBLEM Solve linear problem
%
%   [x, gmres_iter] = solve_linear_problem(A, b, x0, Pinv, mask, Adiag, P)
%
% Solves the linear problem A*x=b
%
% inputs:
%   A:          Numeric matrix (possibly sparse) or function handle
%   b:          Vector or matrix, system rhs
%   x0:         Initial guess for solution, must have the same size as b.
%   Pinv:       Preconditioner for use in GMRES methods, either a function handle
%               or numeric type (matrix or scalar). Set Pinv=[] for no preconditioner.
%   mask:       Vector of logical, mask indicating rows of A where only diagonal
%               elements reside
%   Adiag:      Vector, vector containing the diagonal elements of A, where
%               mask==true, otherwise zeros
%   P:          struct, parameter structure
% returns:
%   x:         Vector, Solution of A\b
%   gmres_iter: iter, Number of GMRES iterations done
%
% The number of evaluations of A per RHS is:
%     gmres_iter
%   + 1 if x0 is provided
%   + 1 if the problem is effectively reduced
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

gmres_iter = 0;

% Use initial guess when provided
hasx0 = ~isempty(x0) && any(x0(:)) && ~strcmp(P.algoGMRES,'direct_inversion');
if hasx0
  if isa(A,'function_handle')
    bt = b - A(x0);
  else
    bt = b - A*x0;
  end
  % Check if residual has been reduced
  x0mask = sum(bt.^2,1) < sum(b.^2,1);
  b(:,x0mask) = bt(:,x0mask);
end

if P.userowmask && any(mask)
  % reducing the system if parts of J are known to be diagonal
  % dx contains already the right values in the reduced DOFs
  [A_, b_, x, Pinv_] = reduce_system(A, b, Pinv, P.userowmask, mask, Adiag);

  % if something remains after reduction, do the system solve
  if numel(b_)
    [x_, gmres_iter] = invert_jacobian(A_, b_, Pinv_, P);
    x(~mask,:) = x_;
  end
else
  [x, gmres_iter] = invert_jacobian(A, b, Pinv, P);
end

% Include initial guess
if hasx0 && any(x0mask)
  x(:,x0mask) = x(:,x0mask)+x0(:,x0mask);
end
end
