function [x,solverinfo] = newton_solver(F,x0,P,stop_early)
% function [x,solverinfo,P] = newton_solver(F,x0,P,stop_early)
% Newton method to solve F(x)=0, linear system is solved using
% - GMRES
% - matlab inversion
% depending on P.algoGMRES parameter
%
% inputs:
%   F:                  Function handle, to solve F==0 for. For requirements, see below
%   x0:                 Vector, inital solution guess
%   P:                  struct, solver parameters. For more info, see %% Parse inputs below
%   stop_early:         boolean, whether to stop if reach fast convergence regime,
%                       i.e., |F(x_t)|/|F(x_t-1)|<1/2. Used to switch back to a faster solver
% returns:
%   x:          Vector, Solution, if solver converged, otherwise its set to x0
%   solverinfo: Struct, Info struct for solver operation
%
% Check solveF description for the requirements for F
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

%% debug/verbosity options
dispprogress   = (P.debug>1); % textual display of each iteration results
displinesrch   = (P.debug>2); % display line search information
dispFeachnewton= (P.debug>3); % display F call internals on each newton iteration
dispFeachkryl  = (P.debug>4); % display F call internals on each krylov iteration

plotrejected   = (P.debugfail  ); % plot rejected Newton iterations
ploteachnewton = (P.debugplot>3); % each call F in newton iteration produces a plot
ploteachkryl   = (P.debugplot>4); % each call of F during krylov iteration produces a plot
plotlinesrch   = (P.debuglinesrch);

%% options for F evaluation
% For each newton iteration
optsnewton = optsF('dodisp',dispFeachnewton,'doplot',ploteachnewton);
if P.anajac && strcmp(P.algoF,'newton')
  if P.jacobian_handle
    optsnewton.dojacF = true;
  else
    optsnewton.dojacx = true;
  end
end

% For each krylov iteration
optskrylov = optsF('dodisp',dispFeachkryl,'doplot',ploteachkryl);
% For each line search step
optslinesrch = optsF('dodisp',dispFeachnewton,'doplot',ploteachnewton);
% For each rejected newton direction
optsreject = optsF('dodisp',dispFeachnewton,'doplot',plotrejected);

% initialize
knewt = 0; % Newton step iteration counter
last_restart = 0;
nfeval = 0; % Counter for number of function call steps
gmres_iter_total = 0; % number of completed inner gmres iterations (bounded by mkryl)
restarts_newton = 0; % restart counter of the newton iteration

x = x0; % Initial solution
resnorm = Inf;

if dispprogress
  header_str = sprintf('\n%12s | %15s | %20s | %7s | %4s | %7s\n', ...
    'k_step/kmax', '||dx||/||x||', 'norm(residual)', 'alpha', 'mkry', 'accept');
  format_iter = '%8d/%-3d | %15.3e | %20.13e | %2.1e | %4d | %7s\n';
end

kmax = P.kmax;

alpha_step_default = 1.0 - P.relax; % default Newton step size
alpha_step = alpha_step_default; % init

% Those are parameters of the linesearch, which controls the convergence rate
%       v(a)<(rho+tau_t)*v(0)−c * a * dv0_da, c=1e-4, rho=1
% tau_t must satisfy sum_t tau_t<inf
% For larger t, this bounds convergences to the usual Armijo rule (tau_t->0)
%       v(a)<v(0)−c * a * dv0_da, c=1e-4
% The choice tau(it)=1./(2*it+1).^2 seems to work well practice.
% In particular, allowing an increase of the residuals is useful at the beginning of the optimization
% process, where some "walls", that need to be jumped over, might appear.
tau = @(it)1./(2*it+1).^2;
rho = 1;

%% Start Newton iteration
while ~(resnorm < P.tolF) && (knewt < kmax) % iterate until tolerance is reached

  % Evaluate F on first iteration
  %   or if J is needed (cost should be much larger than one F eval)
  %   or if a plot is needed (idem)
  if (knewt == 0 || (optsnewton.dojacx || optsnewton.dojacF) || (optsnewton.doplot && ~optslinesrch.doplot))
    [hnormsq,nfeval,Fx,mask,Jdiag,J] = evaluate_F(F,x,optsnewton,P,nfeval,optskrylov); % optskrylov is used in J function handle
    if isnan(hnormsq)
      if dispprogress
        fprintf('Newton solver encountered Fx=NaN for knewt=%d/%d, aborting...\n',knewt,kmax);
      end
      break % Abort newton iterations
    end
    resnorm = sqrt(2*hnormsq);
  end

  alpha_step = min(alpha_step*2,alpha_step_default);

  % initialize preconditioner
  if strcmp(P.precupdate,'newton') || ...                          % When an update is required or
     (knewt == 0 && ~strcmp(P.prectype,'user') && isempty(P.Prec)) % If preconditioner is required but not provided
    Pinv = preconditioner_init(P,J,numel(x));
  elseif (knewt==0)
    Pinv = P.Prec;
  end

  if (knewt == 0)
    % Show initial residual
    if dispprogress
      fprintf(header_str);
      fprintf(format_iter,knewt,kmax,NaN,resnorm,alpha_step,NaN, 'false')
    end
    if resnorm < P.tolF, break; end
  end

  knewt = knewt+1; % iteration counter

  %% Compute newton step
  [dx,gmres_iter] = solve_linear_problem(J,-Fx,[],Pinv,mask,Jdiag,P);
  gmres_iter_total = gmres_iter_total + gmres_iter;
  % If algoF is jfnk, update number of function evaluations
  if strcmp(P.algoF,'jfnk')
    nfeval = nfeval + gmres_iter + (P.userowmask && any(mask));
  end

  % line search along Newton step direction
  lsfun = @(a,nfeval) evaluate_F(F,x+a*dx,optslinesrch,P,nfeval,optskrylov);

  % value at x (alpha = 0)
  v0 = hnormsq;

  % Line search needs dv/da at a=0
  %
  %   v = 0.5*Fx'*Fx, dv/da = Fx'*dFx/dx*dx
  % Using directly the fact that dx is a newton step, so Fx ~ -dFx/dx * dx
  %   dv/da ~ -Fx'*Fx = -2*v0
  %   This is cheaper than a directional derivative but inaccurate
  %   for approximate Newton schemes.
  dv0da = -2*v0;

  a1 = alpha_step; % initial guess for step size
  [alpha_step,hnormsq,accept,~,nfeval,Fx,mask,Jdiag,J] = ...
    linesearch(lsfun,v0,dv0da,[],rho,tau(knewt-1),a1,nfeval,displinesrch,plotlinesrch);
  resnorm_ = sqrt(2*hnormsq);

  % take a step of this size
  xtmp = x + alpha_step * dx;

  % compute relative dx norm
  dx_rel_norm = alpha_step*norm(dx)/norm(xtmp);

  if accept
    % Residual decreased in Newton direction, update the state
    x = xtmp;

    if dispprogress
      if displinesrch, fprintf(header_str); end
      fprintf(format_iter,knewt,kmax,dx_rel_norm,resnorm_,alpha_step,gmres_iter,mat2str(accept))
    end

    % Reached fast decreasing regime. Use to switch to faster algorithms
    if stop_early
      theta = resnorm_/resnorm;
      accept_early = (theta<0.5) && (knewt - last_restart>2); % need to wait few iterations before stopping early
      if accept_early
        resmax = max(abs(Fx));
        resnorm = resnorm_;
        solverinfo = initsolverinfo('isconverged',(resnorm <= P.tolF) && (knewt <= kmax) && ~any(isnan(x)), ...
          'niter',knewt,'residual_max',resmax,'residual_2norm',resnorm,'res',resnorm,'restarts',restarts_newton, ...
          'mkryl',gmres_iter_total,'nfeval',nfeval,'stopped_early',true);
        return
      end
    end
    % update norm
    resnorm = resnorm_;

  else
    % Line search failed, Newton step direction does not seem to be a descent direction, restart
    restarts_newton = restarts_newton + 1;
    last_restart = knewt; % To count iterations since last restart

    if dispprogress
      fprintf('Newton solver did not accept solution. restarts=%d/%d, random=%d, k=%d/%d\n', ...
        restarts_newton,P.nrestarts,P.restart_random,knewt,P.kmax);
    end

    if plotrejected, evaluate_F(F,xtmp,optsreject,P,nfeval); end

    if restarts_newton > P.nrestarts
      if dispprogress
        if displinesrch, fprintf(header_str); end
        fprintf(format_iter, knewt, kmax, dx_rel_norm, resnorm_, alpha_step, gmres_iter, mat2str(accept))
      end
      break % Abort newton iterations
    else
      % Draw new valid state
      if P.restart_random
        ntries = P.random_tries; % Number of allowed random tries
      else
        ntries = 1; % only 1 try for non-random restarts
      end
      for ii = 1:ntries
        if P.restart_random
          xtmp = (1 + 1e-3*randn(size(x))).*x;
        else
          xtmp = (1 + 1e-5)*x;
        end
        [hnormsq,nfeval,Fx,mask,Jdiag,J] = evaluate_F(F,xtmp,optslinesrch,P,nfeval,optskrylov);
        if ~isnan(hnormsq)
          % Valid state found
          x = xtmp;
          resnorm = sqrt(2*hnormsq);
          break;
        end
      end
      if isnan(hnormsq)
        % No Valid state found
        if dispprogress
          fprintf('  Newton solver could not find a valid restart state, aborting...\n')
        end
        break % Abort newton iterations
      end
    end

    if knewt > ( (1-P.restart_iter_frac) * kmax)
      kmax = ceil(kmax*(1+P.restart_iter_frac)); % add more newton iterations to avoid exiting just after a restart
    end
  end
end % end outer Newton while loop

% Output statistics
solverinfo=initsolverinfo('isconverged',(resnorm <= P.tolF) && (knewt <= P.kmax) && ~any(isnan(x)), ...
  'niter',knewt,'residual_max',max(abs(Fx)),'residual_2norm',resnorm,'res',resnorm,'restarts',restarts_newton, ...
  'mkryl',gmres_iter_total,'nfeval',nfeval,'stopped_early',false);

end


