function SC = shapobs(SC,L,LY,varargin)
% Compute a shape observer from an FBT control definition, providing a measure of 
% plasma displacement (error) as a function of fluxes and fields 
% perturbations estimated/reconstructed at a set of control points.
%
% Inputs: 
% SC: shape control structure, from SC = fbt2SC(L,LX) or SC = LY2SC(L,LY)
% L,LY: L,LY structures.
%       LY must contain Fn, Brn, Bzn values of the shape control points
% ns: if the number of shape errors is less than ns, pad them to reach ns
%
% Outputs:
% SC with added fields:
%  - Mysyc: starting from the magnetic measurements yc = [dFn;dBrn;dBzn]
%           computes the differences between the control point fluxes and
%           the reference one, and projects the magnetic field
%           of the limiter point along the limiter. 
%           yc = [dFn,dBrn;dBzn] = [Fn,Brn;Bzn] - [Fn0,Brn0;Bzn0],
%           where [Fn,Brn;Bzn] are the values from the reconstructed equilibrium
%           and [Fn0,Brn0;Bzn0] the reference equilibrium values
%  - Mesys: converts the shape measurements into the shape errors 
%           es = Mesys*ys = Mesys*Mysyc*yc. Mesys is computed as
%           pinv(Myses), where Myses takes into account the local
%           flux/field gradient magnitude. For instance, a small flux
%           gradient (e.g. close to an X-point) means that locally a small 
%           flux error can be translated into a large displacement of the
%           associated control point and vice-versa.
%  - drcdes: converts shape errors in radial displacements
%            drc = drcdes*es
%  - dzcdes: converts shape errors in vertical displacements
%            dzc = dzcdes*es
%  - drsepdes: converts Dsep error in radial displacement
%              drsep = drspedes*es
%  - dzsepdes: converts Dsep error in vertical displacement
%              dzsep = dzspedes*es
%  - ns: number of shape errors
%  - dims: labels for shape errors
%  - lsC,lsS,lsL,lsX1,lsX2,lsD: shape errors classifiers
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

%% Parse inputs
p = inputParser;

% Validation functions
checkSC = @(SC) isstruct(SC);
checkL  = @(L)  isstruct(L);
checkLY = @(LY) isstruct(LY);
checkns = @(nc) isnumeric(nc);

% Add inputs
addRequired(p,'SC',checkSC);
addRequired(p,'L' ,checkL);
addRequired(p,'LY',checkLY);
addOptional(p,'ns',SC.nc,checkns);

% Get parser results
parse(p,SC,L,LY,varargin{:});
ns = p.Results.ns;

% Sanity check
assert(isequal(SC.t,LY.t),'SC.t and LY.t are different')
assert(isfield(LY,'Fn') ,'LY.Fn missing')
assert(isfield(LY,'Brn'),'LY.Brn missing')
assert(isfield(LY,'Bzn'),'LY.Bzn missing')
assert(size(L.P.rn,1) == SC.nc, 'L.P.rn wrong size')
assert(size(L.P.zn,1) == SC.nc, 'L.P.zn wrong size')
assert(ns >= SC.nc, 'ns must be greater than nc')

%%
SC.ns = ns;
nc = SC.nc;
nt = numel(LY.t);

% Pre-allocate structs
SC.Mysyc  = zeros(ns,3*nc,nt);
SC.Myses  = zeros(ns,ns,nt);
SC.Mesys  = zeros(ns,ns,nt);
SC.drcdes = zeros(nc,ns,nt);
SC.dzcdes = zeros(nc,ns,nt);
SC.dims   = cell(ns,nt);
SC.lsC    = false(ns,nt);
SC.lsL    = false(ns,nt);
SC.lsS    = false(ns,nt);
SC.lsX1r  = false(ns,nt);
SC.lsX1z  = false(ns,nt);
SC.lsX2r  = false(ns,nt);
SC.lsX2z  = false(ns,nt);
SC.lsD    = false(ns,nt);
SC.ref    = zeros(ns,nt);
SC.W      = zeros(ns,nt);
SC.rp     = zeros(1,nt);
SC.zp     = zeros(1,nt);

% Iterate over time slices
for it = 1:nt
  
  % Slice SC, LY and run main shape observer cycle
  SCt = meqxk(SC,it);
  LYt = meqxk(LY,it);

  [dims,lsC,lsL,lsS,lsX1r,lsX1z,lsX1f,lsX2r,lsX2z,lsD,lsI,W, ...
    esD,drsep,dzsep,drcdes,dzcdes,drsepdes,dzsepdes,...
    Mysyc,Myses,Mesys] = main_shapobs(SCt,L,LYt);

  % Update SC struct
  SC.Mysyc(:,:,it)  = Mysyc;
  SC.Myses(:,:,it)  = Myses;
  SC.Mesys(:,:,it)  = Mesys;
  SC.drcdes(:,:,it) = drcdes;
  SC.dzcdes(:,:,it) = dzcdes;
  SC.dims(:,it)     = dims;
  SC.lsC(:,it)      = lsC;
  SC.lsL(:,it)      = lsL;
  SC.lsS(:,it)      = lsS;
  SC.lsX1r(:,it)    = lsX1r;
  SC.lsX1z(:,it)    = lsX1z;
  SC.lsX1f(:,it)    = lsX1f;
  SC.lsX2r(:,it)    = lsX2r;
  SC.lsX2z(:,it)    = lsX2z;
  SC.lsD(:,it)      = lsD;
  SC.lsI(:,it)      = lsI;
  SC.ref(:,it)      = Mysyc*[LYt.Fn;LYt.Brn;LYt.Bzn];
  SC.W(:,it)        = W;
  SC.rp(:,it)       = LYt.rIp/LYt.Ip;
  SC.zp(:,it)       = LYt.zIp/LYt.Ip;
  SC.esD(:,it)      = esD(1);
  SC.drsep(:,it)    = drsep(1);
  SC.dzsep(:,it)    = dzsep(1);
  SC.drsepdes(:,it) = drsepdes(1);
  SC.dzsepdes(:,it) = dzsepdes(1);

end

end

function [dvr, dvz] = limiter_unit_vector(rp,zp,rl,zl)
  % Get the unit vector aligned with the limiter at the point closest 
  % to [rp, zp]
  [~,il] = min(sum([rl-rp,zl-zp].^2,2));
  iv = il+[-1,1];
  if il == 1; iv(1) = numel(rl); end
  if il == numel(rl); iv(2) = 1; end
  vr = diff(rl(iv)); vz = diff(zl(iv));
  normv = sqrt(vr.^2+vz.^2);
  if vr~=0, sgn = sign(vr); else, sgn = sign(vz); end
  dvr = sgn * vr./normv; dvz = sgn * vz./normv;
end

function [dims,lsC,lsL,lsS,lsX1r,lsX1z,lsX1f,lsX2r,lsX2z,lsD,lsI,W, ...
    esD,drsep,dzsep,drcdes,dzcdes,drsepdes,dzsepdes,...
    Mysyc,Myses,Mesys] = main_shapobs(SCt,L,LYt)

 %% Setup
  rc    = SCt.rc;    zc    = SCt.zc;
  lcC   = SCt.lcC;   ncC   = sum(lcC);
  lcL   = SCt.lcL;   ncL   = sum(lcL);   iL  = find(lcL);
  lcS   = SCt.lcS;   ncS   = sum(lcS);   iS  = find(lcS);
  lcX1  = SCt.lcX1;  ncX1  = sum(lcX1);  iX1 = find(lcX1);
  lcX2  = SCt.lcX2;  ncX2  = sum(lcX2);  iX2 = find(lcX2);
  lcD   = SCt.lcD;   ncD   = sum(lcX2);
  lcI   = SCt.lcI;
  lcRef = lcX1|lcL;  ncRef = sum(lcRef);
  ncX1f = max(ncX1-1,0);
  ns = SCt.ns; nc = SCt.nc;
  nsI = ns-(ncC+ncL+ncS+ncX1+ncX1+ncX1f+ncX2+ncX2+ncD);
  [isC,isL,isS,isX1r,isX1z,isX1f,isX2r,isX2z,isD,isI,ns] = n2k(ncC,ncL,ncS,ncX1,ncX1,ncX1f,ncX2,ncX2,ncD,nsI);
  [lsC,lsL,lsS,lsX1r,lsX1z,lsX1f,lsX2r,lsX2z,lsD,lsI] = deal(false(ns,1));
  [lsC(isC),lsL(isL),lsS(isS),lsX1r(isX1r),lsX1z(isX1z),lsX1f(isX1f),lsX2r(isX2r),lsX2z(isX2z),lsD(isD),lsI(isI)] = deal(true);
  
  %% Labels
  % Shape errors labels
  dims = cell(ns,1); % dims = strings(ns,1);  
  ii = lsC;   dims(ii) = cellstr(num2str((1:sum(ii))','Surf%02d'));
  ii = lsL;   dims(ii) = cellstr(num2str((1:sum(ii))','Limi%02d'));
  ii = lsS;   dims(ii) = cellstr(num2str((1:sum(ii))','Stri%02d'));
  ii = lsX1r; dims(ii) = cellstr(num2str((1:sum(ii))','rXpri%02d'));
  ii = lsX1z; dims(ii) = cellstr(num2str((1:sum(ii))','zXpri%02d'));
  ii = lsX1f; dims(ii) = cellstr(num2str((1:sum(ii))','fXpri%02d'));
  ii = lsX2r; dims(ii) = cellstr(num2str((1:sum(ii))','rXsec%02d'));
  ii = lsX2z; dims(ii) = cellstr(num2str((1:sum(ii))','zXsec%02d'));
  ii = lsD;   dims(ii) = cellstr(num2str((1:sum(ii))','DSep%02d'));
  ii = lsI;   dims(ii) = cellstr(num2str((1:sum(ii))','Ignr%02d'));
  
  %% Flux and magnetic field gradient
%   Fn = LYt.Fn; 
  Brn = LYt.Brn; Bzn = LYt.Bzn;
  Frc   =  2*pi*SCt.rc.*Bzn;
  Fzc   = -2*pi*SCt.rc.*Brn;
  normF =  sqrt(Frc.^2+Fzc.^2);
  Brrc  =  LYt.Brrn; Brzc = LYt.Brzn;
  Bzrc  =  LYt.Bzrn; Bzzc = LYt.Bzzn;

  %% Contour points
  % Error displacement is along the flux gradient.
  dvr = -sign(LYt.Ip)*Frc(lcC)./normF(lcC);
  dvz = -sign(LYt.Ip)*Fzc(lcC)./normF(lcC);
  drcdesC = diag(dvr);
  dzcdesC = diag(dvz);
  
  % Error magnitude is given by the projection of the error along the
  % flux gradient.
  MysesC = -diag(Frc(SCt.lcC).*dvr+Fzc(SCt.lcC).*dvz);
  
  % Flux differences with respect to the reference flux
  MysycC = zeros(ncC,nc);
  MysycC(:,lcC)   =  eye(ncC);
  MysycC(:,lcRef) = -1/ncRef;

  %% Limiter points
  % Error displacement is tangential to the limiter. Error magnitude is given
  % by the projection of the magnetic field gradient along the unit vector
  % orthogonal to the limiter
  [drcdesL,dzcdesL] = deal(zeros(ncL));
  MysesL  = zeros(ncL);
  MysycLr = zeros(ncL,nc); MysycLz = zeros(ncL,nc);
  
  % Iterate on limiter points
  for ii = 1:ncL
    % Limiter unit vector
    [dvr,dvz] = limiter_unit_vector(rc(iL(ii)),zc(iL(ii)),L.G.rl,L.G.zl);
    drcdesL(ii,ii) = dvr; dzcdesL(ii) = dvz;
    
    % Error magnitude
    MysesL(ii,ii) = -[-dvz,dvr]*[Brrc(iL(ii)),Brzc(iL(ii));
                                 Bzrc(iL(ii)),Bzzc(iL(ii))]*[dvr;dvz];
                               
    % Orthogonal unit vector for the projection
    MysycLr(ii,iL(ii)) = -dvz; MysycLz(ii,iL(ii)) = dvr;
  end

  %% Strike points
  % Error displacement is tangential to the limiter. Error magnitude is given
  % by the projection of the flux gradient along the unit vector orthogonal
  % to the limiter
  [drcdesS, dzcdesS] = deal(zeros(ncS));
  MysesS = zeros(ncS);
  
  % Iterate on strike points
  for ii = 1:ncS
    % Limiter unit vector
    [dvr,dvz] = limiter_unit_vector(rc(iS(ii)),zc(iS(ii)),L.G.rl,L.G.zl);
    drcdesS(ii,ii) = dvr; dzcdesS(ii,ii) = dvz;
    
    % Error magnitude
    MysesS(ii,ii) = -(Frc(iS(ii)).*dvr+Fzc(iS(ii)).*dvz);
  end
  
  % Flux differences with respect to the reference flux
  MysycS = zeros(ncS,nc);
  MysycS(:,lcS )  =  eye(ncS);
  MysycS(:,lcRef) = -1/ncRef;

  %% Primary x points
  % Error displacement is in any `r,z` direction.
  drcdesX1r = eye(ncX1); dzcdesX1z = eye(ncX1);
  
  % Error magnitude is given by the magnetic field gradient
  MysesX1rr = -diag(Brrc(iX1)); MysesX1rz = -diag(Brzc(iX1)); 
  MysesX1zr = -diag(Bzrc(iX1)); MysesX1zz = -diag(Bzzc(iX1));
  
  % Measurements conversion
  MysycX1r         = zeros(ncX1,nc); MysycX1z         = zeros(ncX1,nc);
  MysycX1r(:,lcX1) = eye(ncX1);      MysycX1z(:,lcX1) = eye(ncX1);
  
  %% Primary X-points delta flux
  MysycX1f = zeros(ncX1f,nc);
  if ncX1f > 0
    MysycX1f(:,iX1(1))     =  1;
    MysycX1f(:,iX1(2:end)) = -eye(ncX1-1);
  end

  %% Secondary x points
  % Error displacement is in any `r,z` direction
  drcdesX2r = eye(ncX2); dzcdesX2z = eye(ncX2);
  
  % Error magnitude
  MysesX2rr = -diag(Brrc(iX2)); MysesX2rz = -diag(Brzc(iX2));
  MysesX2zr = -diag(Bzrc(iX2)); MysesX2zz = -diag(Bzzc(iX2));
  
  % Measurements conversion
  MysycX2r         = zeros(ncX2,nc); MysycX2z         = zeros(ncX2,nc);
  MysycX2r(:,lcX2) = eye(ncX2);      MysycX2z(:,lcX2) = eye(ncX2);

  %% Dsep point
  % Error displacement is along the flux gradient.
  dvr      = -sign(LYt.Ip)*Frc(lcD)./normF(lcD); dvz      = -sign(LYt.Ip)*Fzc(lcD)./normF(lcD);
  drsepdes =  diag(dvr);                         dzsepdes =  diag(dvz);
  
  % Error magnitude is given by the projection of the error along the 
  % flux gradient.
  MysesD = -diag(Frc(lcD).*dvr+Fzc(lcD).*dvz);
  
  % Flux differences with respect to the reference flux
  icX2 = find(lcX2);
  MysycD = zeros(ncD,nc);
  for i = 1:ncD
    MysycD(i,icX2(i)) = -1;
    MysycD(i,lcRef) = 1/ncRef;
  end
  
  % Point on the secondary separatrix
  if ncD>0
    yc = LYt.Fn; yc(isnan(yc)) = 0;
    esD = MysesD\(MysycD*yc);
    drsep = drsepdes*esD;
    dzsep = dzsepdes*esD;
  else
    [esD,drsep,dzsep,drsepdes,dzsepdes] = deal(nan);
  end

  %% Compose the total matrices
  % Error directions
  [drcdes,dzcdes]    = deal(zeros(nc,ns));
  drcdes(lcC ,lsC)   = drcdesC;   dzcdes(lcC ,lsC)   = dzcdesC;
  drcdes(lcL ,lsL)   = drcdesL;   dzcdes(lcL ,lsL)   = dzcdesL;
  drcdes(lcS ,lsS)   = drcdesS;   dzcdes(lcS ,lsS)   = dzcdesS;
  drcdes(lcX1,lsX1r) = drcdesX1r; dzcdes(lcX1,lsX1z) = dzcdesX1z;
  drcdes(lcX2,lsX2r) = drcdesX2r; dzcdes(lcX2,lsX2z) = dzcdesX2z;
  
  % Error magnitudes
  Myses = zeros(ns);
  Myses(lsC  ,lsC )  = MysesC;
  Myses(lsL  ,lsL )  = MysesL;
  Myses(lsS  ,lsS )  = MysesS;
  Myses(lsX1r,lsX1r) = MysesX1rr; Myses(lsX1r,lsX1z) = MysesX1rz; 
  Myses(lsX1z,lsX1r) = MysesX1zr; Myses(lsX1z,lsX1z) = MysesX1zz;
  Myses(lsX2r,lsX2r) = MysesX2rr; Myses(lsX2r,lsX2z) = MysesX2rz; 
  Myses(lsX2z,lsX2r) = MysesX2zr; Myses(lsX2z,lsX2z) = MysesX2zz;
  Myses(lsD  ,lsD)   = MysesD;
  Mesys = pinv(Myses);
  
  % Measurements conversions
  T = true(1,nc);
  F = false(1,nc);
  Mysyc = zeros(ns,3*nc);
  
  %            Fn-Fn0 Br-Br0 Bz-Bz0
  Mysyc(lsC  ,[T,     F,     F     ]) = MysycC;
  Mysyc(lsL  ,[F,     T,     F     ]) = MysycLr;
  Mysyc(lsL  ,[F,     F,     T     ]) = MysycLz;
  Mysyc(lsS  ,[T,     F,     F     ]) = MysycS;
  Mysyc(lsX1r,[F,     T,     F     ]) = MysycX1r;
  Mysyc(lsX1z,[F,     F,     T     ]) = MysycX1z;
  Mysyc(lsX1f,[T,     F,     F     ]) = MysycX1f;
  Mysyc(lsX2r,[F,     T,     F     ]) = MysycX2r;
  Mysyc(lsX2z,[F,     F,     T     ]) = MysycX2z;
  Mysyc(lsD  ,[T,     F,     F     ]) = MysycD;

  % weights
  invnormF = 1./normF;
  invnormF(lcI) = 0.0;
  W = ones(ns,1);
  W(lsC) = invnormF(lcC)./norm(invnormF(lcC)) * sum(lcC);
  W(lsS) = invnormF(lcS)./norm(invnormF(lcS)) * sum(lcS);
  W(lsD) = invnormF(lcD)./norm(invnormF(lcD)) * sum(lcD);

end
