function [L_fbt,LX_fbt,L_fge,LX_fge] = generate_doubletTCV(varargin)
% Compute FGE L and LX structure for doublet with prescribed shape parameters
%
% A multitude of shape parameters influencing the doublet in symmetric and
% antisymmetric ways are available. Additionally, quantities like Ip, qA
% and bp can be prescribed. The equilibrium can be computed with or without
% cde_ss_0D and on a finer or normal grid
%
% First, the control points for FBT are computed, FBT is evaluated and then
% fgs is run to yield an initial equilibrium for FGE
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

%% Parse inputs
p=inputParser;
% shape parameters, symmetric % defaults are for an adapted squished
% plasma in upper half
p.addParameter('center_r', 0.887);
p.addParameter('center_z', 0.119);
p.addParameter('xpoint_offset_r', -0.0343);
p.addParameter('size', 0.32);
p.addParameter('elong', 0.8363);
p.addParameter('triang', 0.0652);
p.addParameter('square', 0.047);
p.addParameter('lobe_square', -0.0234);
p.addParameter('waist_expansion', 0);
% shape parameters, asymmetric
p.addParameter('asym_size', 0);
p.addParameter('asym_qA', 0);
p.addParameter('asym_elong', -0.0064);
p.addParameter('asym_triang', -0.0112);
p.addParameter('asym_square', 0.0421);
p.addParameter('rotation', -0.0074);
% Current distribution parameters
p.addParameter('Ip', 2e+5);
p.addParameter('sIp',1); %sign of Ip
p.addParameter('IOH0',0); % initial IOH current 
p.addParameter('Ip_mantle', 0);
p.addParameter('qA', 1.5);
p.addParameter('beta', 0.05);
p.addParameter('beta_mantle', 0.01);
% script parameters
p.addParameter('constrain_zero_mantle', 1); % if mantle current should stay zero its done with ag constraints
p.addParameter('use_cde', 0); % cde_ss_0D is used in this case
p.addParameter('use_SQP', 1); % is faster but sometimes use_SQP=1 does not converge for doublets
p.addParameter('use_fine_grid', 0); % doublet grid resolution, good when mantle is thin but with current
p.addParameter('include_pp_tiles', 1); % if port protection tiles are part of limiter
p.addParameter('n_control_points', 32); % number of control points for fbt
p.addParameter('use_icsint', 0); % number of control points for fbt
p.addParameter('debug', 1); % number of control points for fbt
p.addParameter('tol', 1e-11); % tolerance for fbt and fge
p.addParameter('sBt',1); % Bt sign
p.addParameter('sela',{'E','F','OH'}); % coils to be used
p.addParameter('meq_parameters',{}); % specific meq parameter overrides passed to fbt/fge as last input
parse(p,varargin{:}); Pars = p.Results;

assert(~Pars.use_cde || Pars.constrain_zero_mantle, 'CDE can only be used when mantle is fixed to be zero!');
assert(~Pars.constrain_zero_mantle || Pars.Ip_mantle==0, 'Supplied mantle current must be zero if "zero_mantle_current"');


%% generate geometry
% how many flux surface points to use as constraints in FBT
ntheta = Pars.n_control_points;

% set Ip for lobes and mantle. lobe current is roughly scaled by size
% (assuming circles) of lobe for asymmetric case
r1 = 0.5 * (1 + Pars.asym_size);
r2 = 0.5 * (1 - Pars.asym_size);
IpD = [(Pars.Ip - Pars.Ip_mantle) * r1.^2 ./ (r1.^2 + r2.^2); ...
       (Pars.Ip - Pars.Ip_mantle) * r2.^2 ./ (r1.^2 + r2.^2); ...
       Pars.Ip_mantle];

% generate theta and a few values going into the shape equation
th = linspace(0,2*pi,ntheta+1)'; th = th(1:ntheta);
k = Pars.elong + sin(th) * Pars.asym_elong;
d = Pars.triang + sin(th) * Pars.asym_triang;
l = Pars.square + sin(th) * Pars.asym_square;
q = -Pars.lobe_square; % lobe squareness makes each lobe more square shape but
% normal squareness makes the flux surface more flat at top and bottom
s = Pars.size + sin(th) * Pars.asym_size;

% some trig stuff
thh = th-pi/2;
r_ = (2*s.*sin(thh).*cos(thh))./(1+sin(thh).^2)+Pars.xpoint_offset_r.*(1 - abs(cos(thh)));
z_ = (2*s.*k.*cos(thh+d.*sin(th)+l.*sin(2*th)+q.*sin(4*th)))./(1+sin(thh).^2);

% turn around lower lobe to not have "crossing of line"
r_(z_ < 0) = flip(r_(z_ < 0));

% generate ellipse outline based on size
% if waist-expansion is bigger than zero, flux surface gets interpolated
% between the 8-shape and a 0-shape
r_circ = -Pars.size .* cos(th);
z_circ = 2 * Pars.size .* Pars.elong .* sin(th);

alpha = Pars.waist_expansion;
r_ = r_ + (r_circ - r_) .* (1 - (1 - alpha).^2 * abs(sin(th)).^0.5) * alpha .* cos(th).^2;
z_ = z_ + (z_circ - z_) .* (1 - (1 - alpha).^2 * abs(sin(th)).^0.5) * alpha .* cos(th).^2;

% flux surface is rotated according to rotation parameter
r = cos(Pars.rotation) * r_ - sin(Pars.rotation) * z_ + Pars.center_r;
z = cos(Pars.rotation) * z_ + sin(Pars.rotation) * r_ + Pars.center_z;
% X point is also rotated
rX = Pars.center_r + cos(Pars.rotation) * Pars.xpoint_offset_r;
zX = Pars.center_z + sin(Pars.rotation) * Pars.xpoint_offset_r;


%% get FBT equilibrium
% basis functions
bfct = @bfgenD;

if ~Pars.constrain_zero_mantle
  % mantle basis functions stretch over whole domain
  bfp = {@bfsp, bfpsp([0, 1], [0, 0.5, 1], 'z'), [1;0;0]; ...
         @bfsp, bfpsp([0, 1], [0, 0.5, 1], 'z'), [0;1;0]; ...
         @bfsp, bfpsp([0, 1], [0,      1], 'z'), [1;1;1]};
  agcon = {{'Ip', 'bp', 'qA'}, ...
           {'Ip', 'bp', 'qA'}, ...
           {'Ip', 'bp'}};
else
  % mantle basis functions only for mantle, constrained to zero anyways
  bfp = {@bfsp, bfpsp([0, 1], [0, 0.5, 1], 'z'), [1;0;0]; ...
         @bfsp, bfpsp([0, 1], [0, 0.5, 1], 'z'), [0;1;0]};
  agcon = {{'Ip', 'bp', 'qA'}, ...
           {'Ip', 'bp', 'qA'}};
end

% FBT parameters
if Pars.include_pp_tiles
  shot = 67151; % reference doublet shot with port protection tiles
else
  shot = 55331; % reference doublet shot without port protection tiles
end

% we want a finer grid as too coarse of a grid sometimes gives
% convergence problems when mantle is thin
% tolerance is super small such that FGS can later instantly converge
fbt_params = {'fbtagcon', agcon, 'niter', 150, ...
  'selx', 'X', ...
  'sela',Pars.sela,...
  'bfct', bfct, 'bfp', bfp, 'useSQP', Pars.use_SQP, 'tol', Pars.tol};

extra_params = {};
if Pars.use_icsint
  extra_params = [extra_params, {'ilim', 3, 'icsint', true}];
end


if Pars.use_fine_grid
  P = fbtptcv(shot,fbt_params{:});
  G = fbtgtcv(shot,P); % to get grid range of selx=X grid
  extra_params = [extra_params, ...
                 {'nr', 62, 'nz', 64, 'selx', '', ...
                  'ri', min(G.rx), 'ro', max(G.rx), ...
                  'zu', max(G.zx), 'zl', min(G.zx)}];
end
fbt_params = [fbt_params, extra_params, Pars.meq_parameters];


% build FBT L structure
L_fbt = fbt('tcv', shot, [], fbt_params{:});

% create LX structure
LX_fbt = fbtgp(struct(),[],[],[],[],[],[],[],[],[],[],[],[],[],[],[],[],[],[]);

% some parameters to reproduce fbt behaviour from shot 67151
iE  = startsWith(L_fbt.G.dima,'E');
iF  = startsWith(L_fbt.G.dima,'F');
iOH = startsWith(L_fbt.G.dima,'OH');

LX_fbt.gpbd = 5e-3;
LX_fbt.gpcd = 5e-3;
LX_fbt.gpfd = 5e-3;
LX_fbt.gpid = 988.5591;
LX_fbt.gpud = 2e+4;
LX_fbt.gpod = 988.5591;
LX_fbt.gpie = ones(L_fbt.G.na,1);
LX_fbt.gpia = zeros(L_fbt.G.na, 1);
LX_fbt.gpdw = [repmat(8.5,sum(iE),1);repmat(18,sum(iF),1);repmat(18,sum(iOH),1)];
LX_fbt.gpde = [0.065;0.065;0.065;0.065;0.065;0.065;0.065;1.72159;0.1708;0.3059;0.1708;0.3059;0.1708;0.3059;0.1708];

% exactly constrain IOH current
iOH = contains(L_fbt.G.dima,'OH');
LX_fbt.gpie(iOH,1) = 0;
LX_fbt.gpia(iOH,1) = Pars.IOH0;


% get Iy initial guess, current FBT implementation is not good for doublets
LX_fbt.Iy = get_initial_Iy_guess(L_fbt, r, z, zX, IpD);

% add constraints and parameters
LX_fbt.IpD = IpD;
LX_fbt.Ip = Pars.Ip;
LX_fbt.bpD = [Pars.beta; Pars.beta; Pars.beta_mantle];
LX_fbt.qA = [Pars.qA - Pars.asym_qA; Pars.qA + Pars.asym_qA; 0];
LX_fbt.rBt = Pars.sBt*L_fbt.P.b0 * L_fbt.P.r0;

% remove points too close to the x point
ielim = abs(z) < 0.2;
z=z(~ielim); r=r(~ielim);

 % add our own fbt constraints for r z
LX_fbt = fbtgp(LX_fbt, r, z, 1, 0, 1, 1,[],[],[],[],[],[],[],[], [], [], [],[]);
%             (     X, r, z, b,fa,fb,fe,br,bz,ba,be,cr,cz,ca,ce,vrr,vrz,vzz,ve)
% Specify our x point
LX_fbt = fbtgp(LX_fbt,rX,zX, 1, 0, 1, 0, 0, 0,[],0 ,[],[],[],[],[] ,[] ,[] ,[]);
%             (     X, r, z, b,fa,fb,fe,br,bz,ba,be,cr,cz,ca,ce,vrr,vrz,vzz,ve)
LX_fbt.gpfd = 1e-3;
%LX_fbt.gpdd = 1e5;

% compute equilibrium
LX_fbt = fbtx(L_fbt, LX_fbt);
LY_fbt = fbtt(L_fbt, LX_fbt, 'debug',Pars.debug);

%% Now do the FGE equilibrium
fge_params = {'agcon', agcon, 'bfct', bfct, 'bfp', bfp, ...
              'idoublet', 1, 'tolF', Pars.tol,...
              'sela',{'E','F','OH'}};
fge_params = [fge_params, extra_params, Pars.meq_parameters];


% convert FBT input to LX structure
L_fge = fge('tcv', shot, [], fge_params{:});
LX_fge = meqxconvert(L_fbt, LY_fbt, L_fge);

% reconsolidate for FGS (same as in fgeinit)
Ls = fgsc(L_fge.P,L_fge.G);
% fgs call for initial FGE equilibrium, no precon needed as instant convergence
LY_fgs = fgst(Ls, LX_fge, 'debug', Pars.debug, 'usepreconditioner', false);

%% if needed, construct new FGE structure with CDE stuff
if ~Pars.use_cde
  % convert to FGE equilibrium and call it a day
  LX_fge = meqxconvert(Ls, LY_fgs, L_fge);
else
  agcon = {{'bp', 'qA'}, ...
           {'bp', 'qA'}};
  fge_params = [fge_params, {'agcon', agcon, 'cde', 'cde_ss_0D'}];
  
  % redo the L structure and generate LX from fgs solution
  L_fge = fge('tcv', shot, [], fge_params{:});
  LX_fge = fgex('tcv', [], L_fge, LY_fgs);
end

% get rid of shot number, as this is formally neither 55331 nor 67151
L_fge.P.shot = 0;
LX_fge.shot = 0;
end


function Iy = get_initial_Iy_guess(L, r, z, zX, IpD)
  % Iy initial guess based on distance to plasma boundary
  Op = inpolygon(L.rry, L.zzy, r, z);
  
  sep_dist = reshape(min(sqrt((L.rry(:) - r.').^2 + (L.zzy(:) - z.').^2), [], 2), L.nzy, L.nry);
  
  Iy = sep_dist .* Op .* (0.5 + abs(L.zzy - zX));
  Iy = Iy .* sum(IpD) ./ sum(Iy(:));
  Iy = double(Iy);
end
