function [v,w,res,it,gamma_right,gamma_left] = rzpfasteig(gamma,A,iter_max,tol,v_init)
%% [v,w,res,it,gamma_right,gamma_left] = rzpfasteig(gamma,A,iter_max,tol,v_init)
% Function to be mexified. It calculates the eigenvectors (for a given 
% eigenvalue) of the state matrix A starting from the estimate of the 
% eigenvalue gamma. It is used to calculate the eigenvectors for the
% vertical instability growth rate (gammma>0) and to estimate the maximum
% vertical displacement (see Humpreys et al 2009 Nucl. Fusion 49 115003 and
% rzpfastdzmax). To compute the eigenvectors is used the inverse power
% method.
%
% Input: gamma: eigenvalue to which the eigenvectors must be estimated.
%        A: state matrix.
%        iter_max: maximum number of iteration for the inverse power method
%        tol: tolerance for the inverse power method.
%        v: estimate of the previous eigenvector in case the method is 
%           repeated over time.
%
% Output: v: right eigenvector.
%         w: left eigenvector.
%         res: maximum of the residuals for the inverse power method 
%
% An example of the use of the function and how to retrieve the matrix A
% from fgess follow:
%
% [L,LX] = rzp(tok,shot,t);
%
% %% Code parameters
% iter_max = 1000;
% tol = 1.e-6;
% v = [];
%
% %% Using rzpfasteig (and rzpfastgr. fastest method)
% [~,~,alpha,beta,c] = rzpfastbase(LX.Ia,LX.Iu,LX.Iy,LX.Ip,LX.rIp,LX.Opy,...
%         L.N,L.F,L.Trd,L.Tzd,L.Tfzd,L.Tfrd,L.Tbzd,L.Tbrd,...
%         L.ny,L.np,L.ne,L.RBrye,L.RBzye,L.RBrye_fast,L.RBzye_fast);
% [gamma,res_gamma,it_gamma] = rzpfastgr(alpha,c,L.dd,L.dmax,L.P.griterm,L.P.grtol);
% A = rzpfastA(alpha,beta,L.DD);
% [v,w,res_eig,it_eig] = rzpfasteig(gamma,A,L.P.eigiterm,L.P.eigtol,v);
%
% %% Using fgess
% sys = fgess(L);
% A = sys.A; % you can calculate this matrix also with rzpfastA.m
% [V,D,W] = eig(A);
% [D,indx] = esort(diag(D));
% V = V(:,indx);
% v = V(:,1);  %right eigenvector
% W = W(:,indx);
% w = W(:,1);
% w = w';
% w = w/(w*v); %left eigenvector
% gamma = D(1); %eigenvalue (growth-rate)
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

%#codegen

%Calculate right and left eigenvectors of the state matrix A
if isempty(v_init)
  v_init = rand(size(A,1),1,'single');
end

[gamma_right,v,res_right,it_right] = inverse_power_method(A,gamma-1.e-6,iter_max,tol,'right',v_init);
[gamma_left,w,res_left,it_left]    = inverse_power_method(A,gamma-1.e-6,iter_max,tol,'left',v);

res = max(res_right,res_left);
it = it_right + it_left;

%Normalize the left eigenvector to be consistent with W = V^-1
w = w/(w'*v);
end

function [gamma,v,res,it] = inverse_power_method(A,mu,iter_max,tol,type,v) 
  if mu <= 0 
    gamma = mu;
    v = single(zeros(size(A,1),1));
    res = single(0);
    it = int32(0);
    return
  end
  
  Ainv = (A-mu*eye(size(A)))\eye(size(A));
  [gamma,v,res,it] = power_method(Ainv,iter_max,tol,type,v);
end

function [gamma,v,res,it] = power_method(A,iter_max,tol,type,v)
  %% gamma = power_method(A,iter)
  %calculate the biggest (in modulus) eigenvalue gamma of the matrix A with the
  %power method algorithm
  %
  %gamma: most positive eigenvalue (growth rate)
  %A: square matrix for the eigenvalue calculation
  %iter_max: number of iterations of the algorithm
  %tol: tolerance of the algorithm
  %v: starting eigenvector on which the power method is calculated 
  
  v_prio = v;
  gamma = single(0);
  res = single(0);
  it = int32(0);
  isTypeRight = strcmp(type,'right');
  for ii=1:iter_max
    if isTypeRight
      v = A*v;
      nv = norm(v);
      v = v/nv;
    else
      v = v'*A;
      nv = norm(v);
      v = v'/nv;
    end
    gamma = v'*A*v;
    res = norm(v-v_prio);
    it = ii;
    if res < tol
       break
    else
      v_prio = v;
    end
  end
end