function [K,b,alpha,beta,c,Kinv] = rzpfastbase(Ia,Iu,Iy,Ip,rIp,...
  N,F,Trd,Tzd,ny,np,ne,RBrye,RBzye,RBrye_fast,RBzye_fast)
%% [K,b,alpha,beta,c] = rzpfastbase(Ia,Iu,Iy,Ip,rIp,N,F,Trd,Tzd,ny,np,ne,RBrye,RBzye,RBrye_fast,RBzye_fast)
% Function to calculate necessary quantities for vertical instability
% metrics (growth rate, inductive instability margin and deltazmax). Make
% the fast linearisation (K,b) and apply the base change [see Olofsson 2022]
% to the linearised elements (alpha,beta,c). These are then used in
% subsequent codes to estimate the vertical instabilities matrices (see
% rzpfastm, rzpfastgr, rzpfastdzmax). 
%
%
% Example on how to use the code to estimate the growth-rate and compare 
% the results with the slow codes.
%
% %% Using rzpfastgr
% [L,LX] = rzp('tcv',61400,0.5,'gritermax',100,'grtol',1.e-4);
% [K,b,alpha,beta,c] = rzpfastbase(LX.Ia,LX.Iu,LX.Iy,LX.Ip,LX.rIp,...
%        L.N,L.F,L.Trd,L.Tzd,L.ny,L.np,L.ne,L.RBrye,L.RBzye,L.RBrye_fast,L.RBzye_fast);
% [gamma_fast,res,it] = rzpfastgr(alpha,c,L.dd,L.dmax,L.P.gritermax,L.P.grtol);
%
% %% Using fgess
% gamma_slow = fgeeig(L,'LX',LX);
%
%
% Input:  Ia: equilibrium current in the coils
%         Iu: equilibrium current in the vessel
%         Iy: equilibrium plasma current distribution
%         Ip: equilibrium total plasma current
%         rIp: equilibrium radial position*Ip
%         N: 1/sqrt(Le_fast); 
%         F: sqrt(diag(Re_fast)/Le_fast)*L.N;
%         Tzd: tridiagonal matrix for the finite difference in Iy in the z direction
%         Trd: tridiagonal matrix for the finite difference in Iy in the r direction
%         ny: number of elements in Iy
%         np: number of total plasma currents
%         ne: number of external conductors elements (coil + vessel)
%         RBrye: RBrye*Ie = RBry, that is R*the radial magnetic field in the y grid
%         RBzye: RBzye*Ie = RBzy, that is R*the vertical magnetic field in the y grid
%         RBrye_fast: same as RBrye, but with precalculated fast base change
%         RBzye_fast: same as RBzye, but with precalculated fast base change
%
% Output: K: [dFrdRc dFrdZc; dFzdRc dFzdZc]
%         b: [dFrdIe; dFzdIe]'
%         alpha: inverse of -K+beta'*beta
%         beta: N*b, that is the transformed linearisation element b after the first fast base change
%         c: F*b, that is the transformed linearisation element b after the second fast base change
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

P = struct('rzpfast',true);
L  = struct('Trd',Trd,'Tzd',Tzd,'ny',ny,'np',np,'ne',ne,'RBrye',RBrye,'RBzye',RBzye,...
  'RBrye_fast',RBrye_fast,'RBzye_fast',RBzye_fast,'P',P);
LY = struct('Ia',Ia,'Iu',Iu,'Iy',Iy,'Ip',Ip,'rIp',rIp);

%% Force balance linearisation
[~,dFprzdag,~,dFprzdIe] = rzpFlinfast(L,LY);

K = dFprzdag(2:3,2:3); % K = [dFrdRc dFrdZc; dFzdRc dFzdZc];
b = dFprzdIe(2:3,:)';  % b = [dFrdIe; dFzdIe]';

%% Compute new quantities after the change of basis
%Change the base from Ie to \xi = U^T * L^T * Ie
if nargout > 2
  %Calculate \beta = U^{-1} * L{-1} * b
  beta = N*b; % N = U^{-1} * L{-1}
  %Calculate \alpha = -(K - \beta^T * \beta)^{-1}
  alpha_inv = - K + beta'*beta; 
  det_alpha_inv = alpha_inv(1,1)*alpha_inv(2,2)-alpha_inv(1,2)*alpha_inv(2,1);
  alpha11 =  alpha_inv(2,2)/det_alpha_inv; 
  alpha22 =  alpha_inv(1,1)/det_alpha_inv;
  alpha12 = -alpha_inv(1,2)/det_alpha_inv;
  alpha21 = -alpha_inv(2,1)/det_alpha_inv;
  alpha  = [alpha11 alpha12; alpha21 alpha22];
end

%% Calculate a further change of basis
%Change the base from \xi to \Chi = \Lambda{1/2} * \xi
if nargout > 4
  %Calculate c = \Lambda{1/2} * U^{-1} * L{-1} * b
  c = F*b; % F = \Lambda{1/2} * U^{-1} * L{-1};
end

%% Calculate the inverse of K (used in rzpfastA and rzpfastdzmax)
if nargout > 5
  det_K   = K(1,1)*K(2,2)-K(1,2)*K(2,1);
  K11_inv =  K(2,2)/det_K; 
  K22_inv =  K(1,1)/det_K;
  K12_inv = -K(1,2)/det_K;
  K21_inv = -K(2,1)/det_K;
  Kinv    = [K11_inv K12_inv; K21_inv K22_inv];
end

end