%PSIWTCV  Write PSITBX specific data in TCV tree
% PSIWTCV(SHOT,TREE,BRANCH,L,LY,DEP) writes the
% results of the PSITBX analysis of MEQ data L and LY in
% the BRANCH of TREE for SHOT.
% DEP is the list of node paths to be written in NODES_USED
% subnodes (used in TCV to automatically rerun LIUQE)
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

function psiwtcv(shot,tree,branch,L,LY,dep)

 if nargin < 6, dep = {};end

 %% Run psitbx if necessary
 if L.P.iterq == 0, LY = psipostq(L,LY); end

 %% Checks
 assert(isequal(numel(LY.t),numel(unique(LY.t))),'LY.t has some repeated values, this run cannot be stored in MDS');
 assert(~isempty(LY.aq),'No flux-surface contours found in LY.aq, this run cannot be stored in MDS');
 
 %% 
 
 meqmdsopen(shot,tree,branch)
 
 v = meqver(); % Get meq version number
 
 put = @(node,x,f,d,u,h) meqput(node,x,f,d,u,h,v,dep);

 %% Pre-assembly
 nt = numel(LY.t);
 A_RHO = permute([zeros(L.noq+1,1,nt),LY.aq([1:end,1],:,:)],[2,1,3]);
 R_RHO = 'SET_RANGE(SHAPE(RHO,0),SHAPE(THETA,0),SHAPE(TIME_PSI,0),REPLICATE(R_AXES[0,*],0,SHAPE(RHO,0)*SHAPE(THETA,0)))-SPREAD(SPREAD(COS(THETA),0,SHAPE(RHO,0)),2,SHAPE(TIME_PSI,0))*A_RHO';
 Z_RHO = 'SET_RANGE(SHAPE(RHO,0),SHAPE(THETA,0),SHAPE(TIME_PSI,0),REPLICATE(Z_AXES[0,*],0,SHAPE(RHO,0)*SHAPE(THETA,0)))+SPREAD(SPREAD(SIN(THETA),0,SHAPE(RHO,0)),2,SHAPE(TIME_PSI,0))*A_RHO';
 A_EDGE = reshape(A_RHO(end,:,:),L.noq+1,nt);
 R_EDGE = reshape(LY.rA,[1,nt]) + L.crq([1:end,1]).*A_EDGE;
 Z_EDGE = reshape(LY.zA,[1,nt]) + L.czq([1:end,1]).*A_EDGE;

 khfs = iround(L.oq,        0);
 klfs = iround(L.oq,       pi);

 R_IN_MID    = LY.rA - reshape(A_RHO(:,khfs,:),L.nQ,nt);
 R_OUT_MID   = LY.rA + reshape(A_RHO(:,klfs,:),L.nQ,nt);
 A_MINOR_MID = (R_OUT_MID - R_IN_MID)/2;
 R_GEOM_MID  = (R_OUT_MID + R_IN_MID)/2;

 KAPPA_95     =     L.c95*LY.kappa (L.i95,:);
 DELTA_95_TOP =     L.c95*LY.deltau(L.i95,:);
 DELTA_95_BOT =     L.c95*LY.deltal(L.i95,:);
 DELTA_95     =     L.c95*LY.delta (L.i95,:);
 Q_95         = 1./(L.c95*LY.iqQ   (L.i95,:));

 if L.P.iterq == 0
  % This is the differential volume/area (as in Psi-Toolbox)
  DVOL  = [zeros(1,nt);diff(LY.VQ,1,1)];
  DAREA = [zeros(1,nt);diff(LY.AQ,1,1)];
 else
  % These are the derivatives of the volume/area w.r.t. to rho
  DVOL  = zeros(L.nQ,nt);
  DAREA = DVOL;
  DVOL (2:end,:) = (L.M1q.'*LY.VQ(2:end,:)).*L.pq.';
  DAREA(2:end,:) = (L.M1q.'*LY.AQ(2:end,:)).*L.pq.';
 end

 % Total toroidal flux
 %  Phi = - int (q*dPsi) = 1/(2*pi) int( T/R^2 dV)
 dV = LY.VQ(2:end,:) - LY.VQ(1:end-1,:);
 F = (LY.TQ .* LY.Q2Q)/(2*pi);
 TOR_FLUX_TOT = cumsum([zeros(1,numel(LY.t));0.5*(F(1:end-1,:)+F(2:end,:)).*dV]);

 % Flux conventions
 FAB = LY.FA;
 FXB = LY.FX;
 FxB_help = '0 at r=0, for 2*pi radian';
 FX_help = 'in descencing order of flux value';
 if L.nD == 1
  FAB = FAB - LY.FB;
  FXB = FXB - LY.FB;
  FxB_help = '0 on LCFS, for 2*pi radian';
  FX_help = 'sorted by asbolute flux difference from LCFS';
 end

 %% Grids
 put('RHO'      ,L.pQ.'       ,'f',{'*'},''   ,'rho grid, 0 on axis, 1 on LCFS'                 )
 put('THETA'    ,[L.oq;2*pi]  ,'f',{'*'},'Rad','theta grid, 0 on HFS, CW [Rad]'                 )
 put('TIME_FAST',LY.t         ,'f',{'*'},'s'  ,'Time base for scalars, and some 1D profiles [s]')
 put('TIME_PSI' ,LY.t         ,'f',{'*'},'s'  ,'Time base for 2D functions [s]'                 )
 
 %% Flux surface geometry
 put('R_AXES'      ,LY.rA           ,'f',{'*'          ,'TIME_PSI'},'m'     , 'r position of magnetic axes, sorted by vertical position (*,t) [m]'                                                         )
 put('Z_AXES'      ,LY.zA           ,'f',{'*'          ,'TIME_PSI'},'m'     , 'z position of magnetic axes, sorted by vertical position (*,t) [m]'                                                         )
 put('PSI_AXES'    ,FAB             ,'f',{'*'          ,'TIME_PSI'},'Wb'    ,['Poloidal flux on magnetic axes, sorted by vertical position (*,t)[Wb]',FxB_help]                                            )
 put('DR2_PSI_AXES',LY.dr2FA        ,'f',{'*'          ,'TIME_PSI'},'Wb/m^2', '2nd derivative wrt to r of poloidal flux on magnetic axes, sorted by vertical position, for 2*pi radian (*,t)[Wb/m^2]'      )
 put('DZ2_PSI_AXES',LY.dz2FA        ,'f',{'*'          ,'TIME_PSI'},'Wb/m^2', '2nd derivative wrt to z of poloidal flux on magnetic axes, sorted by vertical position, for 2*pi radian (*,t)[Wb/m^2]'      )
 put('DRZ_PSI_AXES',LY.drzFA        ,'f',{'*'          ,'TIME_PSI'},'Wb/m^2', '2nd derivative wrt to r and z of poloidal flux on magnetic axes, sorted by vertical position, for 2*pi radian (*,t)[Wb/m^2]')
 put('R_XPTS'      ,LY.rX           ,'f',{'*'          ,'TIME_PSI'},'m'     ,['r position of X points, ',FX_help,' (*,t) [m]']                                                                             )
 put('Z_XPTS'      ,LY.zX           ,'f',{'*'          ,'TIME_PSI'},'m'     ,['z position of X points, ',FX_help,' (*,t) [m]']                                                                             )
 put('PSI_XPTS'    ,FXB             ,'f',{'*'          ,'TIME_PSI'},'Wb'    ,['Poloidal flux on X points, ',FX_help,' (*,t)[Wb]',FxB_help]                                                                 )
 put('DR2_PSI_XPTS',LY.dr2FX        ,'f',{'*'          ,'TIME_PSI'},'Wb/m^2',['2nd derivative wrt to r of poloidal flux on X points, ',FX_help,', for 2*pi radian (*,t)[Wb/m^2]']                          )
 put('DZ2_PSI_XPTS',LY.dz2FX        ,'f',{'*'          ,'TIME_PSI'},'Wb/m^2',['2nd derivative wrt to z of poloidal flux on X points, ',FX_help,', for 2*pi radian (*,t)[Wb/m^2]']                          )
 put('DRZ_PSI_XPTS',LY.drzFX        ,'f',{'*'          ,'TIME_PSI'},'Wb/m^2',['2nd derivative wrt to r and z of poloidal flux on X points, ',FX_help,', for 2*pi radian (*,t)[Wb/m^2]']                    )
 put('A_RHO'       ,A_RHO           ,'f',{'RHO','THETA','TIME_PSI'},'m'     , 'Distance to magnetic axis of flux surfaces (rho,theta,t) [m]'         )
 put('R_RHO'       ,R_RHO           ,'x',{'RHO','THETA','TIME_PSI'},'m'     , 'r position of flux surfaces (rho,theta,t) [m]'                        )
 put('Z_RHO'       ,Z_RHO           ,'x',{'RHO','THETA','TIME_PSI'},'m'     , 'z position of flux surfaces (rho,theta,t) [m]'                        )
 put('A_EDGE'      ,A_EDGE          ,'f',{      'THETA','TIME_PSI'},'m'     , 'Distance to magnetic axis of the LCFS (theta,t) [m]'                  )
 put('R_EDGE'      ,R_EDGE          ,'f',{      'THETA','TIME_PSI'},'m'     , 'r position of the LCFS (theta,t) [m]'                                 )
 put('Z_EDGE'      ,Z_EDGE          ,'f',{      'THETA','TIME_PSI'},'m'     , 'z position of the LCFS (theta,t) [m]'                                 )
 put('R_IN_MID'    ,R_IN_MID        ,'f',{'RHO'        ,'TIME_PSI'},'m'     , 'r position of the flux surfaces on inner midplane (rho,t) [m]'        )
 put('R_OUT_MID'   ,R_OUT_MID       ,'f',{'RHO'        ,'TIME_PSI'},'m'     , 'r position of the flux surfaces on outer midplane (rho,t) [m]'        )
 put('A_MINOR_MID' ,A_MINOR_MID     ,'f',{'RHO'        ,'TIME_PSI'},'m'     , 'Minor radius on midplane (rho,t) [m]'                                 )
 put('R_GEOM_MID'  ,R_GEOM_MID      ,'f',{'RHO'        ,'TIME_PSI'},'m'     , 'Geometrical major radius on midplane (rho,t) [m]'                     )
 put('R_IN'        ,LY.rrmin        ,'f',{'RHO'        ,'TIME_PSI'},'m'     , 'r position of the inner most point of flux surfaces (rho,t) [m]'      )
 put('Z_IN'        ,LY.zrmin        ,'f',{'RHO'        ,'TIME_PSI'},'m'     , 'z position of the inner most point of flux surfaces (rho,t) [m]'      )
 put('R_OUT'       ,LY.rrmax        ,'f',{'RHO'        ,'TIME_PSI'},'m'     , 'r position of the outer most point of flux surfaces (rho,t) [m]'      )
 put('Z_OUT'       ,LY.zrmax        ,'f',{'RHO'        ,'TIME_PSI'},'m'     , 'z position of the outer most point of flux surfaces (rho,t) [m]'      )
 put('R_TOP'       ,LY.rzmax        ,'f',{'RHO'        ,'TIME_PSI'},'m'     , 'r position of the top most point of flux surfaces (rho,t) [m]'        )
 put('Z_TOP'       ,LY.zzmax        ,'f',{'RHO'        ,'TIME_PSI'},'m'     , 'z position of the top most point of flux surfaces (rho,t) [m]'        )
 put('R_BOT'       ,LY.rzmin        ,'f',{'RHO'        ,'TIME_PSI'},'m'     , 'r position of the bottom most point of flux surfaces (rho,t) [m]'     )
 put('Z_BOT'       ,LY.zzmin        ,'f',{'RHO'        ,'TIME_PSI'},'m'     , 'z position of the bottom most point of flux surfaces (rho,t) [m]'     )
 put('KAPPA'       ,LY.kappa        ,'f',{'RHO'        ,'TIME_PSI'},''      , 'Elongation of flux surfaces (rho,t)'                                  )
 put('KAPPA_AXIS'  ,LY.kappa(1,:)   ,'f',{              'TIME_PSI'},''      , 'Elongation on magnetic axis (t)'                                      )
 put('KAPPA_95'    ,KAPPA_95        ,'f',{              'TIME_PSI'},''      , 'Elongation of flux surface with rho**2=0.95 (t)'                      )
 put('KAPPA_EDGE'  ,LY.kappa(end,:) ,'f',{              'TIME_PSI'},''      , 'Elongation of LCFS (t)'                                               )
 put('DELTA_TOP'   ,LY.deltau       ,'f',{'RHO'        ,'TIME_PSI'},''      , 'Top triangularity of flux surfaces (rho,t)'                           )
 put('DELTA_BOT'   ,LY.deltal       ,'f',{'RHO'        ,'TIME_PSI'},''      , 'Bottom triangularity of flux surfaces (rho,t)'                        )
 put('DELTA'       ,LY.delta        ,'f',{'RHO'        ,'TIME_PSI'},''      , 'Top/bottom average triangularity of flux surfaces (rho,t)'            )
 put('DELTA_95_TOP',DELTA_95_TOP    ,'f',{              'TIME_PSI'},''      , 'Top triangularity of flux surface with rho**2=0.95 (t)'               )
 put('DELTA_95_BOT',DELTA_95_BOT    ,'f',{              'TIME_PSI'},''      , 'Bottom triangularity of flux surface with rho**2=0.95 (t)'            )
 put('DELTA_95'    ,DELTA_95        ,'f',{              'TIME_PSI'},''      , 'Top/bottom average triangularity of flux surface with rho**2=0.95 (t)')
 put('DELTA_ED_TOP',LY.deltau(end,:),'f',{              'TIME_PSI'},''      , 'Top triangularity of LCFS (t)'                                        )
 put('DELTA_ED_BOT',LY.deltal(end,:),'f',{              'TIME_PSI'},''      , 'Bottom triangularity of LCFS (t)'                                     )
 put('DELTA_EDGE'  ,LY.delta(end,:) ,'f',{              'TIME_PSI'},''      , 'Top/bottom average triangularity of LCFS (t)'                         )
 
 %% Flux surface integrals
 put('AREA'        ,LY.AQ            ,'f',{'RHO','TIME_PSI'},'m^2'       ,'Cross section of flux surfaces (rho,t) [m^2]'                     )
 put('AREA_EDGE'   ,LY.AQ(end,:)     ,'f',{      'TIME_PSI'},'m^2'       ,'Cross section of LCFS (t) [m^2]'                                  )
 put('VOL'         ,LY.VQ            ,'f',{'RHO','TIME_PSI'},'m^3'       ,'Volume of flux surfaces (rho,t) [m^3]'                            )
 put('VOL_EDGE'    ,LY.VQ(end,:)     ,'f',{      'TIME_PSI'},'m^3'       ,'Volume of LCFS (t) [m^3]'                                         )
 put('SURF'        ,LY.SlQ           ,'f',{'RHO','TIME_PSI'},'m^2'       ,'Surface of flux surfaces (rho,t) [m^2]'                           )
 put('DRHO_AREA'   ,DAREA            ,'f',{'RHO','TIME_PSI'},'m^2'       ,'Differential cross section of flux surfaces by drho (rho,t) [m^2]')
 put('DRHO_VOL'    ,DVOL             ,'f',{'RHO','TIME_PSI'},'m^3'       ,'Differential volume of flux surfaces by drho (rho,t) [m^3]'       )
 put('GPSI0_R1_FSA',LY.Q0Q           ,'f',{'RHO','TIME_PSI'},'m^-1'      ,'<1/R> flux surface average (rho,t) [m^-1]'                        )
 put('GPSI0_R2_FSA',LY.Q2Q           ,'f',{'RHO','TIME_PSI'},'m^-2'      ,'<1/R^2> flux surface average (rho,t) [m^-2]'                      )
 put('GPSI2_R0_FSA',LY.Q4Q           ,'f',{'RHO','TIME_PSI'},'Wb^2/m^2'  ,'<|grad psi|^2> flux surface average (rho,t) [Wb^2/m^2]'           )
 put('GPSI2_R2_FSA',LY.Q3Q           ,'f',{'RHO','TIME_PSI'},'Wb^2/m^4'  ,'<|grad psi|^2/R^2> flux surface average (rho,t) [Wb^2/m^4]'       )
 put('Q'           ,1./LY.iqQ        ,'f',{'RHO','TIME_PSI'},''          ,'Safety factor of flux surfaces (rho,t)'                           )
 put('Q_EDGE'      ,1./LY.iqQ(end,:) ,'f',{      'TIME_PSI'},''          ,'Safety factor of LCFS (t)'                                        )
 put('Q_95'        ,Q_95             ,'f',{      'TIME_PSI'},''          ,'Safety factor of flux surface with rho**2=0.95 (t)'               )
 put('Q_AXIS'      ,1./LY.iqQ(1,:)   ,'f',{      'TIME_PSI'},''          ,'Safety factor of axis (t)'                                        )
 put('TOR_FLUX_TOT',TOR_FLUX_TOT     ,'f',{'RHO','TIME_PSI'},'Wb'        ,'Total toroidal flux in flux surfaces (rho,t) [Wb]'                )

 %% Unused nodes
 % put('DPSI_AREA'   ,,'f',{'RHO','TIME_PSI'},'m^2/Wb'       ,'Differential cross section of flux surfaces by dpsi (rho,t) [m^2/Wb]')
 % put('DPSI_VOL'    ,,'f',{'RHO','TIME_PSI'},'m^3/Wb'       ,'Differential volume of flux surfaces by dpsi (rho,t) [m^3/Wb]'       )
 
end
