function [LY] = psipostq(L,LY)
% PSIPOSTQ PsiTbx-based replacement for meqpostq
%
%   [LY] = psipostq(L,LY)
%
% This is implemented for comparison with meqpostq only
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

if isempty(LY), return; end
if isempty(which('psitbxp2p')), error('PsiTbx routines not found in the path, exiting.'); end

psi = meq2psi(L,LY);
fsd = psitbxp2p(psi,'FS',psitbxgrid('Flux','Grid',{L.pQ,[L.oq;2*pi],NaN}));
rzs = psitbxg2g(fsd.grid,'C',fsd);
fsg = psitbxfsg(fsd);
klfs = iround(fsd.grid.x{2},pi);
nt = numel(LY.t);

% Temporary fix for doublets, only treat the upper half
if L.P.idoublet
  % Boundary point
  LY.rB = LY.rB(1,:);
  LY.zB = LY.zB(1,:);
  LY.FB = LY.FB(1,:);
  % Axes
  LY.rA = LY.rA(1,:);
  LY.zA = LY.zA(1,:);
  LY.FA = LY.FA(1,:);
  LY.dr2FA = LY.dr2FA(1,:);
  LY.dz2FA = LY.dz2FA(1,:);
  LY.drzFA = LY.drzFA(1,:);
  % Domain characteristics
  LY.nA = min(LY.nA,1);
  LY.nB = min(LY.nB,1);
  LY.FR = LY.FR(1,:);
  LY.F0 = LY.F0(1,:);
  LY.F1 = LY.F1(1,:);
  % Integral quantities on rho grid per plasma domain
  LY.IpQ  = reshape(LY.IpQ (:,1,:),L.nQ,nt);
  LY.VpQ  = reshape(LY.VpQ (:,1,:),L.nQ,nt);
  LY.PpQ  = reshape(LY.PpQ (:,1,:),L.nQ,nt);
  LY.TTpQ = reshape(LY.TTpQ(:,1,:),L.nQ,nt);
  LY.PQ   = reshape(LY.PQ  (:,1,:),L.nQ,nt);
  LY.TQ   = reshape(LY.TQ  (:,1,:),L.nQ,nt);
  LY.iTQ  = reshape(LY.iTQ (:,1,:),L.nQ,nt);
  LY.FtPQ = reshape(LY.FtPQ(:,1,:),L.nQ,nt);
  % Integral quantities per plasma domain
  LY.IpD  = LY.IpD (1,:);
  LY.zYD  = LY.zYD (1,:);
  LY.rYD  = LY.rYD (1,:);
  LY.zIpD = LY.zIpD(1,:);
  LY.rIpD = LY.rIpD(1,:);
  LY.WkD  = LY.WkD (1,:);
  LY.WpD  = LY.WpD (1,:);
  LY.WtD  = LY.WtD (1,:);
  LY.Wt0D = LY.Wt0D(1,:);
  LY.WND  = LY.WND (1,:);
  LY.VpD  = LY.VpD (1,:);
  LY.FtD  = LY.FtD (1,:);
  LY.Ft0D = LY.Ft0D(1,:);
  LY.bpD  = LY.bpD (1,:);
  LY.btD  = LY.btD (1,:);
  LY.liD  = LY.liD (1,:);
  % Others
  LY.qA = LY.qA(1,:);
end

% Axes
LY.rA = fsd.rmag;
LY.zA = fsd.zmag;
LY.FA = fsd.psimag + LY.FB;
LY.dr2FA = fsd.dr2mag;
LY.dz2FA = fsd.dz2mag;
LY.drzFA = fsd.drzmag;
% X-points
LY.rX = fsd.rxpt;
LY.zX = fsd.zxpt;
LY.FX = fsd.psixpt + LY.FB;
LY.dr2FX = fsd.dr2xpt;
LY.dz2FX = fsd.dz2xpt;
LY.drzFX = fsd.drzxpt;

% Contour locations
LY.aq = permute(fsd.x(2:end,1:end-1,:)   ,[2,1,3]); % (theta,rho,t)
LY.rq = permute(rzs.x{1}(2:end,1:end-1,:),[2,1,3]);
LY.zq = permute(rzs.x{2}(2:end,1:end-1,:),[2,1,3]);

% Wall gaps
LY.aW = NaN(L.G.nW,L.P.nFW,nt);
LY.FW = NaN(1,L.P.nFW,nt);

% FS metrics
% contour integrals
LY.Q0Q = fsg.ngpsi0r1.x;
LY.Q1Q = 1./fsg.ugpsiir1.x;
LY.Q2Q = fsg.ngpsi0r2.x;
LY.Q3Q = fsg.ngpsi2r2.x;
LY.Q4Q = fsg.ngpsi2r0.x;
LY.iqQ = 2*pi .* LY.iTQ ./ fsg.ugpsiiri.x;
LY.q95 = 1./(L.c95*LY.iqQ(L.i95,:))';
% FS volume/area/surface
LY.VQ = fsg.vol.x;
LY.AQ = fsg.area.x;
LY.SlQ = fsg.surf.x;
% shape profiles
%  - extremas
LY.rrmax = fsg.rout.x;
LY.zrmax = fsg.zout.x;
LY.rzmin = fsg.rbot.x;
LY.zzmin = fsg.zbot.x;
LY.rrmin = fsg.rin.x;
LY.zrmin = fsg.zin.x;
LY.rzmax = fsg.rtop.x;
LY.zzmax = fsg.ztop.x;
%  - elongation and triangularity
LY.kappa = double((fsg.ztop - fsg.zbot) ./ (fsg.rout - fsg.rin));
LY.kappa(1,:) = sqrt(fsd.dr2mag./fsd.dz2mag);
LY.deltau = double((fsg.rin + fsg.rout - 2*fsg.rtop) ./ (fsg.rout - fsg.rin));
LY.deltau(1,:) = 0;
LY.deltal = double((fsg.rin + fsg.rout - 2*fsg.rbot) ./ (fsg.rout - fsg.rin));
LY.deltal(1,:) = 0;
LY.delta = (LY.deltau + LY.deltal)*0.5;
%  - major/minor radius, elongation, inverse aspect ratio
LY.aminor = (LY.rrmax-LY.rrmin)/2; % minor radius
LY.rgeom  = (LY.rrmax+LY.rrmin)/2;
LY.zgeom  = (LY.zzmax+LY.zzmin)/2;
LY.epsilon = LY.aminor./LY.rgeom; % inverse aspect ratio
% outboard radius
LY.raQ = fsd.x(:,klfs)./fsd.x(end,klfs);
% Not implemented
LY.ItQ = NaN(L.nQ,L.nD,nt);
LY.LpQ = NaN(L.nQ,L.nD,nt);
LY.rbQ = NaN(L.nQ,L.nD,nt);
LY.Q5Q = NaN(L.nQ,L.nD,nt);
LY.lp = NaN(1,nt);
LY.rbary = NaN(1,nt);
LY.raqmin = NaN(1,nt);
LY.qmin = NaN(1,nt);
if L.nR
  LY.raR = NaN(L.nR,L.P.naR,nt);
end
if L.nS
  LY.rS = NaN(L.noq,L.nS,L.nD,nt);
  LY.zS = NaN(L.noq,L.nS,L.nD,nt);
end

% Others
LY.jtorQ = 2*pi*L.P.r0 * (LY.PpQ + (LY.TTpQ.*LY.Q2Q)/mu0);

end
