%MKMEQH  Generates include file MEQ.H
% MKMEQH(FILES) generates the include file MEQ.H for LIBMEQ
%
% Files in LIBMEQ are written without specifying the floating point precision.
% We instead use macros from the C preprocessor to generate copies of each
% function in single and double precision (with different names).
%
% This function identifies all function declarations in LIBMEQ and adds
% declarations with both types in the generated file MEQ.H. This allows to use
% these functions in MEX-files and C S-functions without declaring them
% explicitely again.
%
% Used by MAKEFILE.
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

function mkmeqh(files)

fh = fopen('include/meq.h','w');
% header
%   Open file containing reference license header
fid = fopen('include/meqh.h','r');
%   Copy first comment block containing license header
while ~feof(fid)
  line = fgetl(fid);
  if startsWith(line,{'/*',' *'})
    fprintf(fh,[line,newline]);
  else
    % Stop as soon as we exit the first comment block
    break 
  end
end
fclose(fid);
fprintf(fh,newline);
fprintf(fh,'/* Generated by mkmeqh.m %s */\n# include "meqh.h"\n',datestr(now)); %#ok<TNOW1,DATST> % lint suggestion to use datetime() does not work in octave
fprintf(fh,'#ifdef __cplusplus\nextern "C" {\n#endif\n');

files = strsplit(files,' ');
for myfile = files
 file = myfile{:};
 [~,grep] = system(['grep -e "\<[sd]' file '[0-9]*\>" mexc/*.c csfc/*.c -l -s']);
 grep(grep==10) = ',';
 fprintf('Doing %12s in %s\n',file,grep(1:end-1))
 ff = fopen(['libmeq/' file '.c'],'r');
 t = fread(ff,Inf,'*char')';
 fclose(ff);
 % Match expression starting at the beginning of a line with a word followed by FLT_NAME(word)( ... )
 % followed by some space and either an open curly bracket or a semicolon
 t = regexp(t,'\n(\w+\s+FLT_NAME\(\w+\)\(.*?\))\s*(\{|;)','tokens');
 for k = 1:length(t)
  tn = regexprep(t{k}{1},'\s+',' '); % Replace all white space by a regular space
  % Replace CPP macros with full names for single and double precision
  patterns = {'FLT_NAME\((\w+)\)','FLT'};
  ts = regexprep(tn,patterns,{'s$1',' float'});
  td = regexprep(tn,patterns,{'d$1','double'});
  fprintf(fh,'%s;\n%s;\n',ts,td);
 end
end
fprintf(fh,'#ifdef __cplusplus\n}\n#endif\n');
fclose(fh);
