%IPM4MEX  Interior point method for aX>0 and no equality constraints 
%
% This is the MATLAB equivalent implementation of libmeq/ipm4.c.
% A more detailed help is available in IPM4MEX.
%
% We assume abs(a)=1, and we use the substitutions s=a*x, z=a*z compared to
% the general algorithm.
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

function [x,z,kt,stat] = ipm4mexm(uH,c,a,x,z,tol,nit,verb)
 if nargin<8, verb=0; end
 stat = 0;
 n  = size(c,1);
 % Construct H from upper triangular form
 H = zeros(n);
 ku = triu(true(n,n));
 H(ku) = uH;
 H = H + triu(H,1)';
 % Iteration loop
 for kt=1:nit
  % Sanity check
  stat = ~any(x==0);
  if ~stat, return;end
  % Predictor step
  d = z./x;
  U = H + diag(d);
  [U,p] = chol(U);
  if p>0, stat = false; return, end
  dx = H*x + c;
  rd = dx - z;
  rc = z.*x;
  dx = U\(U'\dx); % Here we actually compute -dx
  dz = z - d.*dx; %                  ... and -dz
  % Largest step ensuring a*x>0 and a*z>0
  alpha = 1;
  k = (a*dx > 0); alpha = min([alpha ; x(k)./dx(k)]);
  k = (a*dz > 0); alpha = min([alpha ; z(k)./dz(k)]);
  % Update barrier parameter
  tau = sum((x - alpha*dx).*(z - alpha*dz)); % Since r0 and r1 can be both very small,
  tau = (tau/sum(x.*z))^2*(tau/n);           % this is more precise and less prone to NaNs
  % Centering-corrector step
  rc = rc + (dx.*dz-tau);
  e = (dx.*dz-tau)./x;
  dz = U\(U'\e);
  dx = dx + dz;
  dz = e + z - d.*dx;
  % Largest step ensuring a*x>0 and a*z>0 with some margin
  alpha = realmax;
  k = (a*dx > 0); alpha = min([alpha ; x(k)./dx(k)]);
  k = (a*dz > 0); alpha = min([alpha ; z(k)./dz(k)]);
  alpha = min(1,0.99*alpha);
  % Update
  x = x - alpha*dx;
  z = z - alpha*dz;
  % Exit conditions
  res  = max(abs([rd;rc]));
  step = max(abs([dx;dz]));
  if verb
   fprintf('ipm-4 it=%3d res=%5.2e step=%5.2e tol=%5.2e\n',kt,res,step,tol)
  end
  stat = res  < tol;
  if stat, break, end
  stat = step < tol;
  if stat, break, end
end
